#ifndef HYPERELASTIC_SOLVER_HPP
#define HYPERELASTIC_SOLVER_HPP

#include <memory>
#include <tuple>
#include <vector>

class IDiscreteFunction;
class IBoundaryConditionDescriptor;
class IMesh;
class ItemValueVariant;
class SubItemValuePerItemVariant;

double hyperelastic_dt(const std::shared_ptr<const IDiscreteFunction>& c);

class HyperelasticSolverHandler
{
 public:
  enum class SolverType
  {
    Glace,
    Eucclhyd
  };

 private:
  struct IHyperelasticSolver
  {
    virtual std::tuple<const std::shared_ptr<const ItemValueVariant>,
                       const std::shared_ptr<const SubItemValuePerItemVariant>>
    compute_fluxes(
      const SolverType& solver_type,
      const std::shared_ptr<const IDiscreteFunction>& rho,
      const std::shared_ptr<const IDiscreteFunction>& aL,
      const std::shared_ptr<const IDiscreteFunction>& aT,
      const std::shared_ptr<const IDiscreteFunction>& u,
      const std::shared_ptr<const IDiscreteFunction>& sigma,
      const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list) const = 0;

    virtual std::tuple<std::shared_ptr<const IMesh>,
                       std::shared_ptr<const IDiscreteFunction>,
                       std::shared_ptr<const IDiscreteFunction>,
                       std::shared_ptr<const IDiscreteFunction>,
                       std::shared_ptr<const IDiscreteFunction>>
    apply_fluxes(const double& dt,
                 const std::shared_ptr<const IDiscreteFunction>& rho,
                 const std::shared_ptr<const IDiscreteFunction>& u,
                 const std::shared_ptr<const IDiscreteFunction>& E,
                 const std::shared_ptr<const IDiscreteFunction>& CG,
                 const std::shared_ptr<const ItemValueVariant>& ur,
                 const std::shared_ptr<const SubItemValuePerItemVariant>& Fjr) const = 0;

    virtual std::tuple<std::shared_ptr<const IMesh>,
                       std::shared_ptr<const IDiscreteFunction>,
                       std::shared_ptr<const IDiscreteFunction>,
                       std::shared_ptr<const IDiscreteFunction>,
                       std::shared_ptr<const IDiscreteFunction>>
    apply(const SolverType& solver_type,
          const double& dt,
          const std::shared_ptr<const IDiscreteFunction>& rho,
          const std::shared_ptr<const IDiscreteFunction>& u,
          const std::shared_ptr<const IDiscreteFunction>& E,
          const std::shared_ptr<const IDiscreteFunction>& CG,
          const std::shared_ptr<const IDiscreteFunction>& aL,
          const std::shared_ptr<const IDiscreteFunction>& aT,
          const std::shared_ptr<const IDiscreteFunction>& p,
          const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list) const = 0;

    IHyperelasticSolver()                                 = default;
    IHyperelasticSolver(IHyperelasticSolver&&)            = default;
    IHyperelasticSolver& operator=(IHyperelasticSolver&&) = default;

    virtual ~IHyperelasticSolver() = default;
  };

  template <size_t Dimension>
  class HyperelasticSolver;

  std::unique_ptr<IHyperelasticSolver> m_hyperelastic_solver;

 public:
  const IHyperelasticSolver&
  solver() const
  {
    return *m_hyperelastic_solver;
  }

  HyperelasticSolverHandler(const std::shared_ptr<const IMesh>& mesh);
};

#endif   // HYPERELASTIC_SOLVER_HPP
