#include <language/utils/EmbeddedIDiscreteFunctionMathFunctions.hpp>

#include <language/utils/EmbeddedIDiscreteFunctionUtils.hpp>
#include <mesh/IMesh.hpp>
#include <scheme/DiscreteFunctionP0.hpp>
#include <scheme/DiscreteFunctionP0Vector.hpp>
#include <scheme/DiscreteFunctionUtils.hpp>
#include <scheme/IDiscreteFunction.hpp>
#include <scheme/IDiscreteFunctionDescriptor.hpp>

#define DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(FUNCTION, ARG)                                                           \
  if (ARG->dataType() == ASTNodeDataType::double_t and ARG->descriptor().type() == DiscreteFunctionType::P0) {        \
    switch (ARG->mesh()->dimension()) {                                                                               \
    case 1: {                                                                                                         \
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;                                                     \
      return std::make_shared<const DiscreteFunctionType>(FUNCTION(dynamic_cast<const DiscreteFunctionType&>(*ARG))); \
    }                                                                                                                 \
    case 2: {                                                                                                         \
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;                                                     \
      return std::make_shared<const DiscreteFunctionType>(FUNCTION(dynamic_cast<const DiscreteFunctionType&>(*ARG))); \
    }                                                                                                                 \
    case 3: {                                                                                                         \
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;                                                     \
      return std::make_shared<const DiscreteFunctionType>(FUNCTION(dynamic_cast<const DiscreteFunctionType&>(*ARG))); \
    }                                                                                                                 \
    default: {                                                                                                        \
      throw UnexpectedError("invalid mesh dimension");                                                                \
    }                                                                                                                 \
    }                                                                                                                 \
  } else {                                                                                                            \
    throw NormalError(EmbeddedIDiscreteFunctionUtils::invalidOperandType(ARG));                                       \
  }

std::shared_ptr<const IDiscreteFunction>
sqrt(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(sqrt, f);
}

std::shared_ptr<const IDiscreteFunction>
abs(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(abs, f);
}

std::shared_ptr<const IDiscreteFunction>
sin(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(sin, f);
}

std::shared_ptr<const IDiscreteFunction>
cos(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(cos, f);
}

std::shared_ptr<const IDiscreteFunction>
tan(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(tan, f);
}

std::shared_ptr<const IDiscreteFunction>
asin(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(asin, f);
}

std::shared_ptr<const IDiscreteFunction>
acos(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(acos, f);
}

std::shared_ptr<const IDiscreteFunction>
atan(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(atan, f);
}

std::shared_ptr<const IDiscreteFunction>
atan2(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  if ((f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (g->dataType() == ASTNodeDataType::double_t and g->descriptor().type() == DiscreteFunctionType::P0)) {
    std::shared_ptr mesh = getCommonMesh({f, g});

    if (mesh.use_count() == 0) {
      throw NormalError("operands are defined on different meshes");
    }

    switch (mesh->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(
        atan2(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(
        atan2(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(
        atan2(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(f, g));
  }
}

std::shared_ptr<const IDiscreteFunction>
atan2(const double a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(atan2(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(atan2(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(atan2(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(a, f));
  }
}

std::shared_ptr<const IDiscreteFunction>
atan2(const std::shared_ptr<const IDiscreteFunction>& f, const double a)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(atan2(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(atan2(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(atan2(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(f, a));
  }
}

std::shared_ptr<const IDiscreteFunction>
sinh(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(sinh, f);
}

std::shared_ptr<const IDiscreteFunction>
cosh(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(cosh, f);
}

std::shared_ptr<const IDiscreteFunction>
tanh(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(tanh, f);
}

std::shared_ptr<const IDiscreteFunction>
asinh(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(asinh, f);
}

std::shared_ptr<const IDiscreteFunction>
acosh(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(acosh, f);
}

std::shared_ptr<const IDiscreteFunction>
atanh(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(atanh, f);
}

std::shared_ptr<const IDiscreteFunction>
exp(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(exp, f);
}

std::shared_ptr<const IDiscreteFunction>
log(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_VH_TO_VH_REAL_FUNCTION_CALL(log, f);
}

std::shared_ptr<const IDiscreteFunction>
pow(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  if ((f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (g->dataType() == ASTNodeDataType::double_t and g->descriptor().type() == DiscreteFunctionType::P0)) {
    std::shared_ptr mesh = getCommonMesh({f, g});

    if (mesh.use_count() == 0) {
      throw NormalError("operands are defined on different meshes");
    }

    switch (mesh->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(
        pow(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(
        pow(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(
        pow(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(f, g));
  }
}

std::shared_ptr<const IDiscreteFunction>
pow(const double a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(pow(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(pow(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(pow(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(a, f));
  }
}

std::shared_ptr<const IDiscreteFunction>
pow(const std::shared_ptr<const IDiscreteFunction>& f, const double a)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(pow(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(pow(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(pow(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(f, a));
  }
}

template <size_t Dimension>
std::shared_ptr<const IDiscreteFunction>
dot(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  Assert(((f->descriptor().type() == DiscreteFunctionType::P0Vector) and
          (g->descriptor().type() == DiscreteFunctionType::P0Vector)) or
         ((f->dataType() == ASTNodeDataType::vector_t and f->descriptor().type() == DiscreteFunctionType::P0) and
          (g->dataType() == ASTNodeDataType::vector_t and g->descriptor().type() == DiscreteFunctionType::P0) and
          (f->dataType().dimension() == g->dataType().dimension())));

  if ((f->descriptor().type() == DiscreteFunctionType::P0Vector) and
      (g->descriptor().type() == DiscreteFunctionType::P0Vector)) {
    using DiscreteFunctionResultType = DiscreteFunctionP0<Dimension, double>;
    using DiscreteFunctionType       = DiscreteFunctionP0Vector<Dimension, double>;

    const DiscreteFunctionType& f_vector = dynamic_cast<const DiscreteFunctionType&>(*f);
    const DiscreteFunctionType& g_vector = dynamic_cast<const DiscreteFunctionType&>(*g);

    if (f_vector.size() != g_vector.size()) {
      throw NormalError("operands have different dimension");
    } else {
      return std::make_shared<const DiscreteFunctionResultType>(dot(f_vector, g_vector));
    }

  } else {
    using DiscreteFunctionResultType = DiscreteFunctionP0<Dimension, double>;

    switch (f->dataType().dimension()) {
    case 1: {
      using Rd                   = TinyVector<1>;
      using DiscreteFunctionType = DiscreteFunctionP0<Dimension, Rd>;
      return std::make_shared<const DiscreteFunctionResultType>(
        dot(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 2: {
      using Rd                   = TinyVector<2>;
      using DiscreteFunctionType = DiscreteFunctionP0<Dimension, Rd>;
      return std::make_shared<const DiscreteFunctionResultType>(
        dot(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 3: {
      using Rd                   = TinyVector<3>;
      using DiscreteFunctionType = DiscreteFunctionP0<Dimension, Rd>;
      return std::make_shared<const DiscreteFunctionResultType>(
        dot(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError(EmbeddedIDiscreteFunctionUtils::invalidOperandType(f));
    }
      // LCOV_EXCL_STOP
    }
  }
}

std::shared_ptr<const IDiscreteFunction>
dot(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  if (((f->descriptor().type() == DiscreteFunctionType::P0Vector) and
       (g->descriptor().type() == DiscreteFunctionType::P0Vector)) or
      ((f->dataType() == ASTNodeDataType::vector_t and f->descriptor().type() == DiscreteFunctionType::P0) and
       (g->dataType() == ASTNodeDataType::vector_t and g->descriptor().type() == DiscreteFunctionType::P0) and
       (f->dataType().dimension() == g->dataType().dimension()))) {
    std::shared_ptr mesh = getCommonMesh({f, g});

    if (mesh.use_count() == 0) {
      throw NormalError("operands are defined on different meshes");
    }

    switch (mesh->dimension()) {
    case 1: {
      return dot<1>(f, g);
    }
    case 2: {
      return dot<2>(f, g);
    }
    case 3: {
      return dot<3>(f, g);
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(f, g));
  }
}

template <size_t Dimension, size_t VectorDimension>
std::shared_ptr<const IDiscreteFunction>
dot(const std::shared_ptr<const IDiscreteFunction>& f, const TinyVector<VectorDimension>& a)
{
  Assert((f->dataType() == ASTNodeDataType::vector_t and f->descriptor().type() == DiscreteFunctionType::P0) and
         (f->dataType().dimension() == a.dimension()));

  using DiscreteFunctionResultType = DiscreteFunctionP0<Dimension, double>;
  using DiscreteFunctionType       = DiscreteFunctionP0<Dimension, TinyVector<VectorDimension>>;

  return std::make_shared<const DiscreteFunctionResultType>(dot(dynamic_cast<const DiscreteFunctionType&>(*f), a));
}

template <size_t VectorDimension>
std::shared_ptr<const IDiscreteFunction>
dot(const std::shared_ptr<const IDiscreteFunction>& f, const TinyVector<VectorDimension>& a)
{
  if ((f->dataType() == ASTNodeDataType::vector_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (f->dataType().dimension() == a.dimension())) {
    switch (f->mesh()->dimension()) {
    case 1: {
      return dot<1, VectorDimension>(f, a);
    }
    case 2: {
      return dot<2, VectorDimension>(f, a);
    }
    case 3: {
      return dot<3, VectorDimension>(f, a);
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(f, a));
  }
}

template <size_t Dimension, size_t VectorDimension>
std::shared_ptr<const IDiscreteFunction>
dot(const TinyVector<VectorDimension>& a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  Assert((f->dataType() == ASTNodeDataType::vector_t and f->descriptor().type() == DiscreteFunctionType::P0) and
         (f->dataType().dimension() == a.dimension()));

  using DiscreteFunctionResultType = DiscreteFunctionP0<Dimension, double>;
  using DiscreteFunctionType       = DiscreteFunctionP0<Dimension, TinyVector<VectorDimension>>;

  return std::make_shared<const DiscreteFunctionResultType>(dot(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
}

template <size_t VectorDimension>
std::shared_ptr<const IDiscreteFunction>
dot(const TinyVector<VectorDimension>& a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  if ((f->dataType() == ASTNodeDataType::vector_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (f->dataType().dimension() == a.dimension())) {
    switch (f->mesh()->dimension()) {
    case 1: {
      return dot<1, VectorDimension>(a, f);
    }
    case 2: {
      return dot<2, VectorDimension>(a, f);
    }
    case 3: {
      return dot<3, VectorDimension>(a, f);
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(a, f));
  }
}

template std::shared_ptr<const IDiscreteFunction> dot(const std::shared_ptr<const IDiscreteFunction>&,
                                                      const TinyVector<1>&);

template std::shared_ptr<const IDiscreteFunction> dot(const std::shared_ptr<const IDiscreteFunction>&,
                                                      const TinyVector<2>&);

template std::shared_ptr<const IDiscreteFunction> dot(const std::shared_ptr<const IDiscreteFunction>&,
                                                      const TinyVector<3>&);

template std::shared_ptr<const IDiscreteFunction> dot(const TinyVector<1>&,
                                                      const std::shared_ptr<const IDiscreteFunction>&);

template std::shared_ptr<const IDiscreteFunction> dot(const TinyVector<2>&,
                                                      const std::shared_ptr<const IDiscreteFunction>&);

template std::shared_ptr<const IDiscreteFunction> dot(const TinyVector<3>&,
                                                      const std::shared_ptr<const IDiscreteFunction>&);

template <typename MatrixType>
std::shared_ptr<const IDiscreteFunction>
det(const std::shared_ptr<const IDiscreteFunction>& A)
{
  switch (A->mesh()->dimension()) {
  case 1: {
    using DiscreteFunctionType = DiscreteFunctionP0<1, MatrixType>;
    return std::make_shared<const DiscreteFunctionP0<1, double>>(det(dynamic_cast<const DiscreteFunctionType&>(*A)));
  }
  case 2: {
    using DiscreteFunctionType = DiscreteFunctionP0<2, MatrixType>;
    return std::make_shared<const DiscreteFunctionP0<2, double>>(det(dynamic_cast<const DiscreteFunctionType&>(*A)));
  }
  case 3: {
    using DiscreteFunctionType = DiscreteFunctionP0<3, MatrixType>;
    return std::make_shared<const DiscreteFunctionP0<3, double>>(det(dynamic_cast<const DiscreteFunctionType&>(*A)));
  }
  default: {
    throw UnexpectedError("invalid mesh dimension");
  }
  }
}

std::shared_ptr<const IDiscreteFunction>
det(const std::shared_ptr<const IDiscreteFunction>& A)
{
  if (A->dataType() == ASTNodeDataType::matrix_t and A->descriptor().type() == DiscreteFunctionType::P0) {
    if (A->dataType().numberOfRows() != A->dataType().numberOfColumns()) {
      throw NormalError(EmbeddedIDiscreteFunctionUtils::invalidOperandType(A));
    }
    switch (A->dataType().numberOfRows()) {
    case 1: {
      return det<TinyMatrix<1>>(A);
    }
    case 2: {
      return det<TinyMatrix<2>>(A);
    }
    case 3: {
      return det<TinyMatrix<3>>(A);
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid matrix type");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::invalidOperandType(A));
  }
}

template <typename MatrixType>
std::shared_ptr<const IDiscreteFunction>
inverse(const std::shared_ptr<const IDiscreteFunction>& A)
{
  switch (A->mesh()->dimension()) {
  case 1: {
    using DiscreteFunctionType = DiscreteFunctionP0<1, MatrixType>;
    return std::make_shared<const DiscreteFunctionType>(inverse(dynamic_cast<const DiscreteFunctionType&>(*A)));
  }
  case 2: {
    using DiscreteFunctionType = DiscreteFunctionP0<2, MatrixType>;
    return std::make_shared<const DiscreteFunctionType>(inverse(dynamic_cast<const DiscreteFunctionType&>(*A)));
  }
  case 3: {
    using DiscreteFunctionType = DiscreteFunctionP0<3, MatrixType>;
    return std::make_shared<const DiscreteFunctionType>(inverse(dynamic_cast<const DiscreteFunctionType&>(*A)));
  }
  default: {
    throw UnexpectedError("invalid mesh dimension");
  }
  }
}

std::shared_ptr<const IDiscreteFunction>
inverse(const std::shared_ptr<const IDiscreteFunction>& A)
{
  if (A->dataType() == ASTNodeDataType::matrix_t and A->descriptor().type() == DiscreteFunctionType::P0) {
    if (A->dataType().numberOfRows() != A->dataType().numberOfColumns()) {
      throw NormalError(EmbeddedIDiscreteFunctionUtils::invalidOperandType(A));
    }
    switch (A->dataType().numberOfRows()) {
    case 1: {
      return inverse<TinyMatrix<1>>(A);
    }
    case 2: {
      return inverse<TinyMatrix<2>>(A);
    }
    case 3: {
      return inverse<TinyMatrix<3>>(A);
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid matrix type");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::invalidOperandType(A));
  }
}

double
min(const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return min(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return min(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return min(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::invalidOperandType(f));
  }
}

std::shared_ptr<const IDiscreteFunction>
min(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  if ((f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (g->dataType() == ASTNodeDataType::double_t and g->descriptor().type() == DiscreteFunctionType::P0)) {
    std::shared_ptr mesh = getCommonMesh({f, g});

    if (mesh.use_count() == 0) {
      throw NormalError("operands are defined on different meshes");
    }

    switch (mesh->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(
        min(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(
        min(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(
        min(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(f, g));
  }
}

std::shared_ptr<const IDiscreteFunction>
min(const double a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(min(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(min(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(min(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(a, f));
  }
}

std::shared_ptr<const IDiscreteFunction>
min(const std::shared_ptr<const IDiscreteFunction>& f, const double a)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(min(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(min(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(min(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(f, a));
  }
}

double
max(const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return max(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return max(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return max(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::invalidOperandType(f));
  }
}

std::shared_ptr<const IDiscreteFunction>
max(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  if ((f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (g->dataType() == ASTNodeDataType::double_t and g->descriptor().type() == DiscreteFunctionType::P0)) {
    std::shared_ptr mesh = getCommonMesh({f, g});

    if (mesh.use_count() == 0) {
      throw NormalError("operands are defined on different meshes");
    }

    switch (mesh->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(
        max(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(
        max(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(
        max(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(f, g));
  }
}

std::shared_ptr<const IDiscreteFunction>
max(const double a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(max(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(max(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(max(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(a, f));
  }
}

std::shared_ptr<const IDiscreteFunction>
max(const std::shared_ptr<const IDiscreteFunction>& f, const double a)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(max(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(max(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(max(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::incompatibleOperandTypes(f, a));
  }
}

template <typename ValueT>
ValueT
sum_of(const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ast_node_data_type_from<ValueT> and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, ValueT>;
      return sum(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, ValueT>;
      return sum(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, ValueT>;
      return sum(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::invalidOperandType(f));
  }
}

std::shared_ptr<const IDiscreteFunction>
sum_of_Vh_components(const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->descriptor().type() == DiscreteFunctionType::P0Vector) {
    switch (f->mesh()->dimension()) {
    case 1: {
      constexpr size_t Dimension = 1;
      using DiscreteFunctionType = DiscreteFunctionP0Vector<Dimension, double>;
      return std::make_shared<const DiscreteFunctionP0<Dimension, double>>(
        sumOfComponents(dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 2: {
      constexpr size_t Dimension = 2;
      using DiscreteFunctionType = DiscreteFunctionP0Vector<Dimension, double>;
      return std::make_shared<const DiscreteFunctionP0<Dimension, double>>(
        sumOfComponents(dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 3: {
      constexpr size_t Dimension = 3;
      using DiscreteFunctionType = DiscreteFunctionP0Vector<Dimension, double>;
      return std::make_shared<const DiscreteFunctionP0<Dimension, double>>(
        sumOfComponents(dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::invalidOperandType(f));
  }
}

std::shared_ptr<const IDiscreteFunction>
vectorize(const std::vector<std::shared_ptr<const IDiscreteFunction>>& discrete_function_list)
{
  Assert(discrete_function_list.size() > 0);
  std::shared_ptr p_i_mesh = [&] {
    auto i = discrete_function_list.begin();

    std::shared_ptr<const IMesh> i_mesh = (*i)->mesh();
    ++i;
    for (; i != discrete_function_list.end(); ++i) {
      if ((*i)->mesh() != i_mesh) {
        throw NormalError("discrete functions are not defined on the same mesh");
      }
    }

    return i_mesh;
  }();

  for (auto&& i_discrete_function : discrete_function_list) {
    if ((i_discrete_function->descriptor().type() != DiscreteFunctionType::P0) or
        (i_discrete_function->dataType() != ASTNodeDataType::double_t)) {
      throw NormalError(EmbeddedIDiscreteFunctionUtils::invalidOperandType(i_discrete_function));
    }
  }

  switch (p_i_mesh->dimension()) {
  case 1: {
    constexpr size_t Dimension       = 1;
    using DiscreteFunctionVectorType = DiscreteFunctionP0Vector<Dimension, double>;
    using DiscreteFunctionType       = DiscreteFunctionP0<Dimension, double>;
    std::shared_ptr<const Mesh<Connectivity<Dimension>>> p_mesh =
      std::dynamic_pointer_cast<const Mesh<Connectivity<Dimension>>>(p_i_mesh);

    DiscreteFunctionVectorType vector_function(p_mesh, discrete_function_list.size());
    for (size_t i = 0; i < discrete_function_list.size(); ++i) {
      const DiscreteFunctionType& f = dynamic_cast<const DiscreteFunctionType&>(*discrete_function_list[i]);
      for (CellId cell_id = 0; cell_id < p_mesh->numberOfCells(); ++cell_id) {
        vector_function[cell_id][i] = f[cell_id];
      }
    }

    return std::make_shared<DiscreteFunctionVectorType>(vector_function);
  }
  case 2: {
    constexpr size_t Dimension       = 2;
    using DiscreteFunctionVectorType = DiscreteFunctionP0Vector<Dimension, double>;
    using DiscreteFunctionType       = DiscreteFunctionP0<Dimension, double>;
    std::shared_ptr<const Mesh<Connectivity<Dimension>>> p_mesh =
      std::dynamic_pointer_cast<const Mesh<Connectivity<Dimension>>>(p_i_mesh);

    DiscreteFunctionVectorType vector_function(p_mesh, discrete_function_list.size());
    for (size_t i = 0; i < discrete_function_list.size(); ++i) {
      const DiscreteFunctionType& f = dynamic_cast<const DiscreteFunctionType&>(*discrete_function_list[i]);
      for (CellId cell_id = 0; cell_id < p_mesh->numberOfCells(); ++cell_id) {
        vector_function[cell_id][i] = f[cell_id];
      }
    }

    return std::make_shared<DiscreteFunctionVectorType>(vector_function);
  }
  case 3: {
    constexpr size_t Dimension       = 3;
    using DiscreteFunctionVectorType = DiscreteFunctionP0Vector<Dimension, double>;
    using DiscreteFunctionType       = DiscreteFunctionP0<Dimension, double>;
    std::shared_ptr<const Mesh<Connectivity<Dimension>>> p_mesh =
      std::dynamic_pointer_cast<const Mesh<Connectivity<Dimension>>>(p_i_mesh);

    DiscreteFunctionVectorType vector_function(p_mesh, discrete_function_list.size());
    for (size_t i = 0; i < discrete_function_list.size(); ++i) {
      const DiscreteFunctionType& f = dynamic_cast<const DiscreteFunctionType&>(*discrete_function_list[i]);
      for (CellId cell_id = 0; cell_id < p_mesh->numberOfCells(); ++cell_id) {
        vector_function[cell_id][i] = f[cell_id];
      }
    }

    return std::make_shared<DiscreteFunctionVectorType>(vector_function);
  }
    // LCOV_EXCL_START
  default: {
    throw UnexpectedError("invalid mesh dimension");
  }
    // LCOV_EXCL_STOP
  }
}

template double sum_of<double>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyVector<1> sum_of<TinyVector<1>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyVector<2> sum_of<TinyVector<2>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyVector<3> sum_of<TinyVector<3>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyMatrix<1> sum_of<TinyMatrix<1>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyMatrix<2> sum_of<TinyMatrix<2>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyMatrix<3> sum_of<TinyMatrix<3>>(const std::shared_ptr<const IDiscreteFunction>&);

template <typename ValueT>
ValueT
integral_of(const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ast_node_data_type_from<ValueT> and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, ValueT>;
      return integrate(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, ValueT>;
      return integrate(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, ValueT>;
      return integrate(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
      // LCOV_EXCL_STOP
    }
  } else {
    throw NormalError(EmbeddedIDiscreteFunctionUtils::invalidOperandType(f));
  }
}

template double integral_of<double>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyVector<1> integral_of<TinyVector<1>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyVector<2> integral_of<TinyVector<2>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyVector<3> integral_of<TinyVector<3>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyMatrix<1> integral_of<TinyMatrix<1>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyMatrix<2> integral_of<TinyMatrix<2>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyMatrix<3> integral_of<TinyMatrix<3>>(const std::shared_ptr<const IDiscreteFunction>&);
