#ifndef SYNCHRONIZER_HPP
#define SYNCHRONIZER_HPP

#include <mesh/Connectivity.hpp>
#include <mesh/ItemArray.hpp>
#include <mesh/ItemValue.hpp>
#include <mesh/SubItemArrayPerItem.hpp>
#include <mesh/SubItemValuePerItem.hpp>
#include <utils/Exceptions.hpp>
#include <utils/Messenger.hpp>

#include <utils/pugs_config.hpp>

#include <iostream>
#include <map>
#include <memory>

#ifdef PUGS_HAS_MPI

class Synchronizer
{
 private:
  template <ItemType item_type>
  using ExchangeItemTypeInfo = std::vector<Array<const ItemIdT<item_type>>>;

  std::unique_ptr<ExchangeItemTypeInfo<ItemType::cell>> m_requested_cell_info;
  std::unique_ptr<ExchangeItemTypeInfo<ItemType::cell>> m_provided_cell_info;

  std::unique_ptr<ExchangeItemTypeInfo<ItemType::face>> m_requested_face_info;
  std::unique_ptr<ExchangeItemTypeInfo<ItemType::face>> m_provided_face_info;

  std::unique_ptr<ExchangeItemTypeInfo<ItemType::edge>> m_requested_edge_info;
  std::unique_ptr<ExchangeItemTypeInfo<ItemType::edge>> m_provided_edge_info;

  std::unique_ptr<ExchangeItemTypeInfo<ItemType::node>> m_requested_node_info;
  std::unique_ptr<ExchangeItemTypeInfo<ItemType::node>> m_provided_node_info;

  using ExchangeSubItemPerItemSize = std::vector<std::map<std::pair<ItemType, ItemType>, size_t>>;

  ExchangeSubItemPerItemSize m_sub_item_per_item_requested_size_list;
  ExchangeSubItemPerItemSize m_sub_item_per_item_provided_size_list;

  template <ItemType item_type>
  PUGS_INLINE constexpr auto&
  _getRequestedItemInfo()
  {
    if constexpr (item_type == ItemType::cell) {
      return m_requested_cell_info;
    } else if constexpr (item_type == ItemType::face) {
      return m_requested_face_info;
    } else if constexpr (item_type == ItemType::edge) {
      return m_requested_edge_info;
    } else if constexpr (item_type == ItemType::node) {
      return m_requested_node_info;
    }
  }

  template <ItemType item_type>
  PUGS_INLINE constexpr auto&
  _getProvidedItemInfo()
  {
    if constexpr (item_type == ItemType::cell) {
      return m_provided_cell_info;
    } else if constexpr (item_type == ItemType::face) {
      return m_provided_face_info;
    } else if constexpr (item_type == ItemType::edge) {
      return m_provided_edge_info;
    } else if constexpr (item_type == ItemType::node) {
      return m_provided_node_info;
    }
  }

  template <typename ConnectivityType, ItemType item_type>
  void
  _buildSynchronizeInfo(const ConnectivityType& connectivity)
  {
    const auto& item_owner = connectivity.template owner<item_type>();
    using ItemId           = ItemIdT<item_type>;

    auto& p_requested_item_info = this->_getRequestedItemInfo<item_type>();
    p_requested_item_info       = [&]() {
      std::vector<std::vector<ItemId>> requested_item_vector_info(parallel::size());
      for (ItemId item_id = 0; item_id < item_owner.numberOfItems(); ++item_id) {
        if (const size_t owner = item_owner[item_id]; owner != parallel::rank()) {
          requested_item_vector_info[owner].emplace_back(item_id);
        }
      }
      ExchangeItemTypeInfo<item_type> requested_item_info(parallel::size());
      for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
        const auto& requested_item_vector = requested_item_vector_info[i_rank];
        requested_item_info[i_rank]       = convert_to_array(requested_item_vector);
      }
      return std::make_unique<ExchangeItemTypeInfo<item_type>>(std::move(requested_item_info));
    }();

    auto& requested_item_info = *p_requested_item_info;

    Array<unsigned int> local_number_of_requested_values(parallel::size());
    for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
      local_number_of_requested_values[i_rank] = requested_item_info[i_rank].size();
    }

    Array<unsigned int> local_number_of_values_to_send = parallel::allToAll(local_number_of_requested_values);

    std::vector<Array<const int>> requested_item_number_list_by_proc(parallel::size());
    const auto& item_number = connectivity.template number<item_type>();
    for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
      const auto& requested_item_info_from_rank = requested_item_info[i_rank];
      Array<int> item_number_list{requested_item_info_from_rank.size()};
      parallel_for(
        requested_item_info_from_rank.size(),
        PUGS_LAMBDA(size_t i_item) { item_number_list[i_item] = item_number[requested_item_info_from_rank[i_item]]; });
      requested_item_number_list_by_proc[i_rank] = item_number_list;
    }

    std::vector<Array<int>> provided_item_number_list_by_rank(parallel::size());
    for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
      provided_item_number_list_by_rank[i_rank] = Array<int>{local_number_of_values_to_send[i_rank]};
    }

    parallel::exchange(requested_item_number_list_by_proc, provided_item_number_list_by_rank);

    std::map<int, ItemId> item_number_to_id_correspondance;
    for (ItemId item_id = 0; item_id < item_number.numberOfItems(); ++item_id) {
      item_number_to_id_correspondance[item_number[item_id]] = item_id;
    }

    auto& p_provided_item_info = this->_getProvidedItemInfo<item_type>();
    p_provided_item_info       = [&]() {
      ExchangeItemTypeInfo<item_type> provided_item_info(parallel::size());
      for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
        Array<ItemId> provided_item_id_to_rank{local_number_of_values_to_send[i_rank]};
        const Array<int>& provided_item_number_to_rank = provided_item_number_list_by_rank[i_rank];
        for (size_t i = 0; i < provided_item_number_to_rank.size(); ++i) {
          provided_item_id_to_rank[i] = item_number_to_id_correspondance.find(provided_item_number_to_rank[i])->second;
        }
        provided_item_info[i_rank] = provided_item_id_to_rank;
      }
      return std::make_unique<ExchangeItemTypeInfo<item_type>>(provided_item_info);
    }();

    m_sub_item_per_item_provided_size_list.resize(parallel::size());
    m_sub_item_per_item_requested_size_list.resize(parallel::size());
  }

  template <ItemType item_type, ItemType sub_item_type, size_t Dimension>
  PUGS_INLINE size_t
  _getSubItemPerItemRequestedSize(const Connectivity<Dimension>& connectivity, const size_t i_rank)
  {
    Assert(m_sub_item_per_item_requested_size_list.size() == parallel::size());

    auto key = std::make_pair(item_type, sub_item_type);
    if (auto i_size_list = m_sub_item_per_item_requested_size_list[i_rank].find(key);
        i_size_list != m_sub_item_per_item_requested_size_list[i_rank].end()) {
      return i_size_list->second;
    } else {
      const auto& p_requested_item_info = this->_getRequestedItemInfo<item_type>();

      Assert(static_cast<bool>(p_requested_item_info) == true,
             "this function should be called after calculation of exchange info");
      const auto& requested_item_info_from_rank = (*p_requested_item_info)[i_rank];

      const auto& item_to_item_matrix = connectivity.template getItemToItemMatrix<item_type, sub_item_type>();

      size_t count = 0;
      for (size_t i = 0; i < requested_item_info_from_rank.size(); ++i) {
        count += item_to_item_matrix[requested_item_info_from_rank[i]].size();
      }

      m_sub_item_per_item_requested_size_list[i_rank][key] = count;
      return count;
    }
  }

  template <ItemType item_type, ItemType sub_item_type, size_t Dimension>
  PUGS_INLINE size_t
  _getSubItemPerItemProvidedSize(const Connectivity<Dimension>& connectivity, const size_t i_rank)
  {
    Assert(m_sub_item_per_item_provided_size_list.size() == parallel::size());

    auto key = std::make_pair(item_type, sub_item_type);
    if (auto i_size_list = m_sub_item_per_item_provided_size_list[i_rank].find(key);
        i_size_list != m_sub_item_per_item_provided_size_list[i_rank].end()) {
      return i_size_list->second;
    } else {
      const auto& p_provided_item_info = this->_getProvidedItemInfo<item_type>();

      Assert(static_cast<bool>(p_provided_item_info) == true,
             "this function should be called after calculation of exchange info");
      const auto& provided_item_info_from_rank = (*p_provided_item_info)[i_rank];

      const auto& item_to_item_matrix = connectivity.template getItemToItemMatrix<item_type, sub_item_type>();

      size_t count = 0;
      for (size_t i = 0; i < provided_item_info_from_rank.size(); ++i) {
        count += item_to_item_matrix[provided_item_info_from_rank[i]].size();
      }

      m_sub_item_per_item_provided_size_list[i_rank][key] = count;
      return count;
    }
  }

  template <typename ConnectivityType, typename DataType, ItemType item_type, typename ConnectivityPtr>
  PUGS_INLINE void
  _synchronize(const ConnectivityType& connectivity, ItemValue<DataType, item_type, ConnectivityPtr>& item_value)
  {
    static_assert(not std::is_abstract_v<ConnectivityType>, "_synchronize must be called on a concrete connectivity");

    using ItemId = ItemIdT<item_type>;

    const auto& p_provided_item_info  = this->_getProvidedItemInfo<item_type>();
    const auto& p_requested_item_info = this->_getRequestedItemInfo<item_type>();

    Assert(static_cast<bool>(p_provided_item_info) == static_cast<bool>(p_requested_item_info));

    if (not p_provided_item_info) {
      this->_buildSynchronizeInfo<ConnectivityType, item_type>(connectivity);
    }

    const auto& provided_item_info  = *p_provided_item_info;
    const auto& requested_item_info = *p_requested_item_info;

    Assert(requested_item_info.size() == provided_item_info.size());

    std::vector<Array<const DataType>> provided_data_list(parallel::size());
    for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
      const Array<const ItemId>& provided_item_info_to_rank = provided_item_info[i_rank];
      Array<DataType> provided_data{provided_item_info_to_rank.size()};
      parallel_for(
        provided_item_info_to_rank.size(),
        PUGS_LAMBDA(size_t i) { provided_data[i] = item_value[provided_item_info_to_rank[i]]; });
      provided_data_list[i_rank] = provided_data;
    }

    std::vector<Array<DataType>> requested_data_list(parallel::size());
    for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
      const auto& requested_item_info_from_rank = requested_item_info[i_rank];
      requested_data_list[i_rank]               = Array<DataType>{requested_item_info_from_rank.size()};
    }

    parallel::exchange(provided_data_list, requested_data_list);

    for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
      const auto& requested_item_info_from_rank = requested_item_info[i_rank];
      const auto& requested_data                = requested_data_list[i_rank];
      parallel_for(
        requested_item_info_from_rank.size(),
        PUGS_LAMBDA(size_t i) { item_value[requested_item_info_from_rank[i]] = requested_data[i]; });
    }
  }

  template <typename ConnectivityType, typename DataType, ItemType item_type, typename ConnectivityPtr>
  PUGS_INLINE void
  _synchronize(const ConnectivityType& connectivity, ItemArray<DataType, item_type, ConnectivityPtr>& item_array)
  {
    static_assert(not std::is_abstract_v<ConnectivityType>, "_synchronize must be called on a concrete connectivity");

    using ItemId = ItemIdT<item_type>;

    const auto& p_provided_item_info  = this->_getProvidedItemInfo<item_type>();
    const auto& p_requested_item_info = this->_getRequestedItemInfo<item_type>();

    Assert(static_cast<bool>(p_provided_item_info) == static_cast<bool>(p_requested_item_info));

    if (not p_provided_item_info) {
      this->_buildSynchronizeInfo<ConnectivityType, item_type>(connectivity);
    }

    const auto& provided_item_info  = *p_provided_item_info;
    const auto& requested_item_info = *p_requested_item_info;

    Assert(requested_item_info.size() == provided_item_info.size());

    const size_t size_of_arrays = item_array.sizeOfArrays();

    std::vector<Array<const DataType>> provided_data_list(parallel::size());
    for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
      const Array<const ItemId>& provided_item_info_to_rank = provided_item_info[i_rank];
      Array<DataType> provided_data{provided_item_info_to_rank.size() * size_of_arrays};
      parallel_for(
        provided_item_info_to_rank.size(), PUGS_LAMBDA(size_t i) {
          const size_t j   = i * size_of_arrays;
          const auto array = item_array[provided_item_info_to_rank[i]];
          for (size_t k = 0; k < size_of_arrays; ++k) {
            provided_data[j + k] = array[k];
          }
        });
      provided_data_list[i_rank] = provided_data;
    }

    std::vector<Array<DataType>> requested_data_list(parallel::size());
    for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
      const auto& requested_item_info_from_rank = requested_item_info[i_rank];
      requested_data_list[i_rank] = Array<DataType>{requested_item_info_from_rank.size() * size_of_arrays};
    }

    parallel::exchange(provided_data_list, requested_data_list);

    for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
      const auto& requested_item_info_from_rank = requested_item_info[i_rank];
      const auto& requested_data                = requested_data_list[i_rank];
      parallel_for(
        requested_item_info_from_rank.size(), PUGS_LAMBDA(size_t i) {
          const size_t j = i * size_of_arrays;
          auto array     = item_array[requested_item_info_from_rank[i]];
          for (size_t k = 0; k < size_of_arrays; ++k) {
            array[k] = requested_data[j + k];
          }
        });
    }
  }

  template <typename ConnectivityType, typename DataType, typename ItemOfItem, typename ConnectivityPtr>
  PUGS_INLINE void
  _synchronize(const ConnectivityType& connectivity,
               SubItemValuePerItem<DataType, ItemOfItem, ConnectivityPtr>& sub_item_value_per_item)
  {
    static_assert(not std::is_abstract_v<ConnectivityType>, "_synchronize must be called on a concrete connectivity");
    if constexpr (ItemTypeId<ConnectivityType::Dimension>::dimension(ItemOfItem::item_type) >
                  ItemTypeId<ConnectivityType::Dimension>::dimension(ItemOfItem::sub_item_type)) {
      constexpr ItemType item_type     = ItemOfItem::item_type;
      constexpr ItemType sub_item_type = ItemOfItem::sub_item_type;

      using ItemId = ItemIdT<item_type>;

      const auto& p_provided_item_info  = this->_getProvidedItemInfo<item_type>();
      const auto& p_requested_item_info = this->_getRequestedItemInfo<item_type>();

      Assert(static_cast<bool>(p_provided_item_info) == static_cast<bool>(p_requested_item_info));

      if (not p_provided_item_info) {
        this->_buildSynchronizeInfo<ConnectivityType, item_type>(connectivity);
      }

      const auto& provided_item_info  = *p_provided_item_info;
      const auto& requested_item_info = *p_requested_item_info;

      Assert(requested_item_info.size() == provided_item_info.size());

      std::vector<Array<const DataType>> provided_data_list(parallel::size());
      for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
        const Array<const ItemId>& provided_item_info_to_rank = provided_item_info[i_rank];
        const size_t send_size = _getSubItemPerItemProvidedSize<item_type, sub_item_type>(connectivity, i_rank);

        Array<DataType> provided_data{send_size};
        size_t index = 0;
        for (size_t i = 0; i < provided_item_info_to_rank.size(); ++i) {
          const ItemId item_id   = provided_item_info_to_rank[i];
          const auto item_values = sub_item_value_per_item.itemArray(item_id);
          for (size_t j = 0; j < item_values.size(); ++j) {
            provided_data[index++] = item_values[j];
          }
        }
        provided_data_list[i_rank] = provided_data;
      }

      std::vector<Array<DataType>> requested_data_list(parallel::size());
      for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
        const size_t recv_size      = _getSubItemPerItemRequestedSize<item_type, sub_item_type>(connectivity, i_rank);
        requested_data_list[i_rank] = Array<DataType>{recv_size};
      }

      parallel::exchange(provided_data_list, requested_data_list);
      for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
        const auto& requested_item_info_from_rank = requested_item_info[i_rank];
        const auto& requested_data                = requested_data_list[i_rank];

        size_t index = 0;
        for (size_t i = 0; i < requested_item_info_from_rank.size(); ++i) {
          const ItemId item_id   = requested_item_info_from_rank[i];
          const auto item_values = sub_item_value_per_item.itemArray(item_id);
          for (size_t j = 0; j < item_values.size(); ++j) {
            item_values[j] = requested_data[index++];
          }
        }
      }
    } else {
      std::ostringstream os;
      os << "synchronization requires sub-item type (" << itemName(ItemOfItem::sub_item_type)
         << ") to be of lower dimension than item (" << itemName(ItemOfItem::item_type) << ")";
      throw UnexpectedError(os.str());
    }
  }

  template <typename ConnectivityType, typename DataType, typename ItemOfItem, typename ConnectivityPtr>
  PUGS_INLINE void
  _synchronize(const ConnectivityType& connectivity,
               SubItemArrayPerItem<DataType, ItemOfItem, ConnectivityPtr>& sub_item_array_per_item)
  {
    static_assert(not std::is_abstract_v<ConnectivityType>, "_synchronize must be called on a concrete connectivity");
    if constexpr (ItemTypeId<ConnectivityType::Dimension>::dimension(ItemOfItem::item_type) >
                  ItemTypeId<ConnectivityType::Dimension>::dimension(ItemOfItem::sub_item_type)) {
      constexpr ItemType item_type     = ItemOfItem::item_type;
      constexpr ItemType sub_item_type = ItemOfItem::sub_item_type;

      using ItemId = ItemIdT<item_type>;

      const auto& p_provided_item_info  = this->_getProvidedItemInfo<item_type>();
      const auto& p_requested_item_info = this->_getRequestedItemInfo<item_type>();

      Assert(static_cast<bool>(p_provided_item_info) == static_cast<bool>(p_requested_item_info));

      if (not p_provided_item_info) {
        this->_buildSynchronizeInfo<ConnectivityType, item_type>(connectivity);
      }

      const auto& provided_item_info  = *p_provided_item_info;
      const auto& requested_item_info = *p_requested_item_info;

      Assert(requested_item_info.size() == provided_item_info.size());

      std::vector<Array<const DataType>> provided_data_list(parallel::size());
      for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
        const Array<const ItemId>& provided_item_info_to_rank = provided_item_info[i_rank];
        const size_t send_size = _getSubItemPerItemProvidedSize<item_type, sub_item_type>(connectivity, i_rank);

        Array<DataType> provided_data{send_size * sub_item_array_per_item.sizeOfArrays()};
        size_t index = 0;
        for (size_t i = 0; i < provided_item_info_to_rank.size(); ++i) {
          const ItemId item_id  = provided_item_info_to_rank[i];
          const auto item_table = sub_item_array_per_item.itemTable(item_id);
          for (size_t j = 0; j < item_table.numberOfRows(); ++j) {
            Assert(item_table.numberOfColumns() == sub_item_array_per_item.sizeOfArrays());
            for (size_t k = 0; k < sub_item_array_per_item.sizeOfArrays(); ++k) {
              provided_data[index++] = item_table(j, k);
            }
          }
        }
        provided_data_list[i_rank] = provided_data;
      }

      std::vector<Array<DataType>> requested_data_list(parallel::size());
      for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
        const size_t recv_size      = _getSubItemPerItemRequestedSize<item_type, sub_item_type>(connectivity, i_rank);
        requested_data_list[i_rank] = Array<DataType>{recv_size * sub_item_array_per_item.sizeOfArrays()};
      }

      parallel::exchange(provided_data_list, requested_data_list);
      for (size_t i_rank = 0; i_rank < parallel::size(); ++i_rank) {
        const auto& requested_item_info_from_rank = requested_item_info[i_rank];
        const auto& requested_data                = requested_data_list[i_rank];

        size_t index = 0;
        for (size_t i = 0; i < requested_item_info_from_rank.size(); ++i) {
          const ItemId item_id  = requested_item_info_from_rank[i];
          const auto item_table = sub_item_array_per_item.itemTable(item_id);
          for (size_t j = 0; j < item_table.numberOfRows(); ++j) {
            Assert(item_table.numberOfColumns() == sub_item_array_per_item.sizeOfArrays());
            for (size_t k = 0; k < sub_item_array_per_item.sizeOfArrays(); ++k) {
              item_table(j, k) = requested_data[index++];
            }
          }
        }
      }
    } else {
      std::ostringstream os;
      os << "synchronization requires sub-item type (" << itemName(ItemOfItem::sub_item_type)
         << ") to be of lower dimension than item (" << itemName(ItemOfItem::item_type) << ")";
      throw UnexpectedError(os.str());
    }
  }

 public:
  template <typename DataType, ItemType item_type, typename ConnectivityPtr>
  PUGS_INLINE void
  synchronize(ItemValue<DataType, item_type, ConnectivityPtr>& item_value)
  {
    Assert(item_value.connectivity_ptr().use_count() > 0, "No connectivity is associated to this ItemValue");
    const IConnectivity& connectivity = *item_value.connectivity_ptr();

    switch (connectivity.dimension()) {
    case 1: {
      this->_synchronize(static_cast<const Connectivity1D&>(connectivity), item_value);
      break;
    }
    case 2: {
      this->_synchronize(static_cast<const Connectivity2D&>(connectivity), item_value);
      break;
    }
    case 3: {
      this->_synchronize(static_cast<const Connectivity3D&>(connectivity), item_value);
      break;
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("unexpected dimension");
    }
      // LCOV_EXCL_STOP
    }
  }

  template <typename DataType, ItemType item_type, typename ConnectivityPtr>
  PUGS_INLINE void
  synchronize(ItemArray<DataType, item_type, ConnectivityPtr>& item_array)
  {
    Assert(item_array.connectivity_ptr().use_count() > 0, "No connectivity is associated to this ItemArray");
    const IConnectivity& connectivity = *item_array.connectivity_ptr();

    switch (connectivity.dimension()) {
    case 1: {
      this->_synchronize(static_cast<const Connectivity1D&>(connectivity), item_array);
      break;
    }
    case 2: {
      this->_synchronize(static_cast<const Connectivity2D&>(connectivity), item_array);
      break;
    }
    case 3: {
      this->_synchronize(static_cast<const Connectivity3D&>(connectivity), item_array);
      break;
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("unexpected dimension");
    }
      // LCOV_EXCL_STOP
    }
  }

  template <typename DataType, typename ItemOfItem, typename ConnectivityPtr>
  PUGS_INLINE void
  synchronize(SubItemValuePerItem<DataType, ItemOfItem, ConnectivityPtr>& sub_item_value_per_item)
  {
    Assert(sub_item_value_per_item.connectivity_ptr().use_count() > 0,
           "No connectivity is associated to this SubItemValuePerItem");

    const IConnectivity& connectivity = *sub_item_value_per_item.connectivity_ptr();

    switch (connectivity.dimension()) {
    case 1: {
      this->_synchronize(static_cast<const Connectivity1D&>(connectivity), sub_item_value_per_item);
      break;
    }
    case 2: {
      this->_synchronize(static_cast<const Connectivity2D&>(connectivity), sub_item_value_per_item);
      break;
    }
    case 3: {
      this->_synchronize(static_cast<const Connectivity3D&>(connectivity), sub_item_value_per_item);
      break;
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("unexpected dimension");
    }
      // LCOV_EXCL_STOP
    }
  }

  template <typename DataType, typename ItemOfItem, typename ConnectivityPtr>
  PUGS_INLINE void
  synchronize(SubItemArrayPerItem<DataType, ItemOfItem, ConnectivityPtr>& sub_item_value_per_item)
  {
    Assert(sub_item_value_per_item.connectivity_ptr().use_count() > 0,
           "No connectivity is associated to this SubItemValuePerItem");

    const IConnectivity& connectivity = *sub_item_value_per_item.connectivity_ptr();

    switch (connectivity.dimension()) {
    case 1: {
      this->_synchronize(static_cast<const Connectivity1D&>(connectivity), sub_item_value_per_item);
      break;
    }
    case 2: {
      this->_synchronize(static_cast<const Connectivity2D&>(connectivity), sub_item_value_per_item);
      break;
    }
    case 3: {
      this->_synchronize(static_cast<const Connectivity3D&>(connectivity), sub_item_value_per_item);
      break;
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("unexpected dimension");
    }
      // LCOV_EXCL_STOP
    }
  }

  Synchronizer(const Synchronizer&) = delete;
  Synchronizer(Synchronizer&&)      = delete;

 private:
  friend class SynchronizerManager;

  PUGS_INLINE
  Synchronizer()
  {
    ;
  }
};

#else   // PUGS_HAS_MPI

class Synchronizer
{
 public:
  template <typename DataType, ItemType item_type, typename ConnectivityPtr>
  PUGS_INLINE void
  synchronize(ItemValue<DataType, item_type, ConnectivityPtr>& item_value)
  {
    Assert(item_value.connectivity_ptr().use_count() > 0, "No connectivity is associated to this ItemValue");
  }

  template <typename DataType, ItemType item_type, typename ConnectivityPtr>
  PUGS_INLINE void
  synchronize(ItemArray<DataType, item_type, ConnectivityPtr>& item_value)
  {
    Assert(item_value.connectivity_ptr().use_count() > 0, "No connectivity is associated to this ItemValue");
  }

  template <typename DataType, typename ItemOfItem, typename ConnectivityPtr>
  PUGS_INLINE void
  synchronize(SubItemValuePerItem<DataType, ItemOfItem, ConnectivityPtr>& sub_item_value_per_item)
  {
    Assert(sub_item_value_per_item.connectivity_ptr().use_count() > 0,
           "No connectivity is associated to this SubItemValuePerItem");

    const IConnectivity& connectivity = *sub_item_value_per_item.connectivity_ptr();

    switch (connectivity.dimension()) {
    case 1: {
      if constexpr (ItemTypeId<1>::dimension(ItemOfItem::item_type) <=
                    ItemTypeId<1>::dimension(ItemOfItem::sub_item_type)) {
        std::ostringstream os;
        os << "synchronization requires sub-item type (" << itemName(ItemOfItem::sub_item_type)
           << ") to be of lower dimension than item (" << itemName(ItemOfItem::item_type) << ")";
        throw UnexpectedError(os.str());
      }
      break;
    }
    case 2: {
      if constexpr (ItemTypeId<2>::dimension(ItemOfItem::item_type) <=
                    ItemTypeId<2>::dimension(ItemOfItem::sub_item_type)) {
        std::ostringstream os;
        os << "synchronization requires sub-item type (" << itemName(ItemOfItem::sub_item_type)
           << ") to be of lower dimension than item (" << itemName(ItemOfItem::item_type) << ")";
        throw UnexpectedError(os.str());
      }
      break;
    }
    case 3: {
      if constexpr (ItemTypeId<3>::dimension(ItemOfItem::item_type) <=
                    ItemTypeId<3>::dimension(ItemOfItem::sub_item_type)) {
        std::ostringstream os;
        os << "synchronization requires sub-item type (" << itemName(ItemOfItem::sub_item_type)
           << ") to be of lower dimension than item (" << itemName(ItemOfItem::item_type) << ")";
        throw UnexpectedError(os.str());
      }
      break;
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("unexpected dimension");
    }
      // LCOV_EXCL_STOP
    }
  }

  template <typename DataType, typename ItemOfItem, typename ConnectivityPtr>
  PUGS_INLINE void
  synchronize(SubItemArrayPerItem<DataType, ItemOfItem, ConnectivityPtr>& sub_item_array_per_item)
  {
    Assert(sub_item_array_per_item.connectivity_ptr().use_count() > 0,
           "No connectivity is associated to this SubItemArrayPerItem");

    const IConnectivity& connectivity = *sub_item_array_per_item.connectivity_ptr();

    switch (connectivity.dimension()) {
    case 1: {
      if constexpr (ItemTypeId<1>::dimension(ItemOfItem::item_type) <=
                    ItemTypeId<1>::dimension(ItemOfItem::sub_item_type)) {
        std::ostringstream os;
        os << "synchronization requires sub-item type (" << itemName(ItemOfItem::sub_item_type)
           << ") to be of lower dimension than item (" << itemName(ItemOfItem::item_type) << ")";
        throw UnexpectedError(os.str());
      }
      break;
    }
    case 2: {
      if constexpr (ItemTypeId<2>::dimension(ItemOfItem::item_type) <=
                    ItemTypeId<2>::dimension(ItemOfItem::sub_item_type)) {
        std::ostringstream os;
        os << "synchronization requires sub-item type (" << itemName(ItemOfItem::sub_item_type)
           << ") to be of lower dimension than item (" << itemName(ItemOfItem::item_type) << ")";
        throw UnexpectedError(os.str());
      }
      break;
    }
    case 3: {
      if constexpr (ItemTypeId<3>::dimension(ItemOfItem::item_type) <=
                    ItemTypeId<3>::dimension(ItemOfItem::sub_item_type)) {
        std::ostringstream os;
        os << "synchronization requires sub-item type (" << itemName(ItemOfItem::sub_item_type)
           << ") to be of lower dimension than item (" << itemName(ItemOfItem::item_type) << ")";
        throw UnexpectedError(os.str());
      }
      break;
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("unexpected dimension");
    }
      // LCOV_EXCL_STOP
    }
  }

  Synchronizer(const Synchronizer&) = delete;
  Synchronizer(Synchronizer&&)      = delete;

 private:
  friend class SynchronizerManager;

  PUGS_INLINE
  Synchronizer()
  {
    ;
  }
};

#endif   // PUGS_HAS_MPI

#endif   // SYNCHRONIZER_HPP
