#include <mesh/DiamondDualMeshBuilder.hpp>

#include <mesh/Connectivity.hpp>
#include <mesh/DualConnectivityManager.hpp>
#include <mesh/ItemValueUtils.hpp>
#include <mesh/Mesh.hpp>
#include <mesh/MeshData.hpp>
#include <mesh/MeshDataManager.hpp>
#include <mesh/PrimalToDiamondDualConnectivityDataMapper.hpp>
#include <utils/Stringify.hpp>

template <size_t Dimension>
void
DiamondDualMeshBuilder::_buildDualDiamondMeshFrom(const IMesh& i_mesh)
{
  static_assert(Dimension > 1);

  using ConnectivityType = Connectivity<Dimension>;
  using MeshType         = Mesh<Connectivity<Dimension>>;

  const MeshType& primal_mesh = dynamic_cast<const MeshType&>(i_mesh);

  DualConnectivityManager& manager = DualConnectivityManager::instance();

  std::shared_ptr<const ConnectivityType> p_diamond_connectivity =
    manager.getDiamondDualConnectivity(primal_mesh.connectivity());

  const ConnectivityType& diamond_connectivity = *p_diamond_connectivity;

  const NodeValue<const TinyVector<Dimension>> primal_xr = primal_mesh.xr();

  MeshData<Dimension>& primal_mesh_data                  = MeshDataManager::instance().getMeshData(primal_mesh);
  const CellValue<const TinyVector<Dimension>> primal_xj = primal_mesh_data.xj();

  std::shared_ptr primal_to_diamond_dual_connectivity_data_mapper =
    manager.getPrimalToDiamondDualConnectivityDataMapper(primal_mesh.connectivity());

  NodeValue<TinyVector<Dimension>> diamond_xr{diamond_connectivity};
  primal_to_diamond_dual_connectivity_data_mapper->toDualNode(primal_xr, primal_xj, diamond_xr);

  m_mesh = std::make_shared<MeshType>(p_diamond_connectivity, diamond_xr);
}

DiamondDualMeshBuilder::DiamondDualMeshBuilder(const IMesh& i_mesh)
{
  switch (i_mesh.dimension()) {
  case 2: {
    this->_buildDualDiamondMeshFrom<2>(i_mesh);
    break;
  }
  case 3: {
    this->_buildDualDiamondMeshFrom<3>(i_mesh);
    break;
  }
    // LCOV_EXCL_START
  default: {
    throw UnexpectedError("invalid mesh dimension: " + stringify(i_mesh.dimension()));
  }
    // LCOV_EXCL_STOP
  }
}
