#include <language/ast/ASTNodeDataType.hpp>

#include <language/PEGGrammar.hpp>
#include <language/ast/ASTNode.hpp>
#include <utils/PugsAssert.hpp>

ASTNodeDataType
getVectorDataType(const ASTNode& type_node)
{
  if (not(type_node.is_type<language::vector_type>() and (type_node.children.size() == 2))) {
    throw parse_error("unexpected node type", type_node.begin());
  }
  ASTNode& dimension_node = *type_node.children[1];
  if (not dimension_node.is_type<language::integer>()) {
    throw parse_error("unexpected non integer constant dimension", dimension_node.begin());
  }
  const size_t dimension = std::stol(dimension_node.string());
  return ASTNodeDataType{ASTNodeDataType::vector_t, dimension};
}

std::string
dataTypeName(const ASTNodeDataType& data_type)
{
  std::string name;
  switch (data_type) {
  case ASTNodeDataType::undefined_t:
    name = "undefined";
    break;
  case ASTNodeDataType::bool_t:
    name = "B";
    break;
  case ASTNodeDataType::unsigned_int_t:
    name = "N";
    break;
  case ASTNodeDataType::int_t:
    name = "Z";
    break;
  case ASTNodeDataType::double_t:
    name = "R";
    break;
  case ASTNodeDataType::vector_t:
    name = "R^" + std::to_string(data_type.dimension());
    break;
  case ASTNodeDataType::tuple_t:
    name = "tuple(" + dataTypeName(data_type.contentType()) + ')';
    break;
  case ASTNodeDataType::list_t:
    name = "list";
    break;
  case ASTNodeDataType::string_t:
    name = "string";
    break;
  case ASTNodeDataType::typename_t:
    name = "typename";
    break;
  case ASTNodeDataType::type_name_id_t:
    name = "type_name_id";
    break;
  case ASTNodeDataType::type_id_t:
    name = data_type.nameOfTypeId();
    break;
  case ASTNodeDataType::function_t:
    name = "function";
    break;
  case ASTNodeDataType::builtin_function_t:
    name = "builtin_function";
    break;
  case ASTNodeDataType::void_t:
    name = "void";
    break;
  }
  return name;
}

ASTNodeDataType
dataTypePromotion(const ASTNodeDataType& data_type_1, const ASTNodeDataType& data_type_2)
{
  if (data_type_1 == data_type_2) {
    return data_type_1;
  } else if ((std::max(data_type_1, data_type_2) <= ASTNodeDataType::double_t) and
             (std::min(data_type_1, data_type_2) >= ASTNodeDataType::bool_t)) {
    return std::max(data_type_1, data_type_2);
  } else if ((data_type_1 == ASTNodeDataType::string_t) and (data_type_2 < ASTNodeDataType::string_t) and
             (data_type_2 >= ASTNodeDataType::bool_t)) {
    return data_type_1;
  } else if ((data_type_1 >= ASTNodeDataType::bool_t) and (data_type_1 <= ASTNodeDataType::double_t) and
             (data_type_2 == ASTNodeDataType::vector_t)) {
    return data_type_2;
  } else {
    return ASTNodeDataType::undefined_t;
  }
}

bool
isNaturalConversion(const ASTNodeDataType& data_type, const ASTNodeDataType& target_data_type)
{
  if (target_data_type == data_type) {
    if (data_type == ASTNodeDataType::type_id_t) {
      return (data_type.nameOfTypeId() == target_data_type.nameOfTypeId());
    } else if (data_type == ASTNodeDataType::vector_t) {
      return (data_type.dimension() == target_data_type.dimension());
    } else {
      return true;
    }
  } else if (target_data_type == ASTNodeDataType::bool_t) {
    return false;
  } else if (target_data_type == ASTNodeDataType::unsigned_int_t) {
    return ((data_type == ASTNodeDataType::int_t) or (data_type == ASTNodeDataType::bool_t));
  } else if (target_data_type == ASTNodeDataType::int_t) {
    return ((data_type == ASTNodeDataType::unsigned_int_t) or (data_type == ASTNodeDataType::bool_t));
  } else if (target_data_type == ASTNodeDataType::double_t) {
    return ((data_type == ASTNodeDataType::unsigned_int_t) or (data_type == ASTNodeDataType::int_t) or
            (data_type == ASTNodeDataType::bool_t));
  } else if (target_data_type == ASTNodeDataType::string_t) {
    return ((data_type >= ASTNodeDataType::bool_t) and (data_type < ASTNodeDataType::string_t));
  } else if (target_data_type == ASTNodeDataType::tuple_t) {
    return isNaturalConversion(data_type, target_data_type.contentType());
  }
  return false;
}
