#ifndef CONNECTIVITY_1D_HPP
#define CONNECTIVITY_1D_HPP

#include <Kokkos_Core.hpp>
#include <PastisAssert.hpp>

#include <TinyVector.hpp>
#include <ConnectivityUtils.hpp>

#include <RefId.hpp>
#include <RefNodeList.hpp>

class Connectivity1D
{
public:
  static constexpr size_t dimension = 1;

  ConnectivityMatrix m_cell_to_node_matrix;

  ConnectivityMatrix m_node_to_cell_matrix;
  ConnectivityMatrixShort m_node_to_cell_local_node_matrix;

  // Stores numbering of nodes of each cell.
  //
  // This is different from m_cell_to_node_matrix which return the global id of
  // a local node in a cell
  ConnectivityMatrix m_node_id_per_cell_matrix;

private:
  std::vector<RefNodeList> m_ref_node_list;

  Kokkos::View<double*> m_inv_cell_nb_nodes;

  std::vector<std::vector<unsigned int>>
  _buildConnectivity(const size_t& number_of_cells)
  {
    std::vector<std::vector<unsigned int>> cell_by_node_vector(number_of_cells);
    for (unsigned int j=0; j<number_of_cells; ++j) {
      cell_by_node_vector[j] = {j, j+1};
    }
    return cell_by_node_vector;
  }
public:
  void addRefNodeList(const RefNodeList& ref_node_list)
  {
    m_ref_node_list.push_back(ref_node_list);
  }

  size_t numberOfRefNodeList() const
  {
    return m_ref_node_list.size();
  }

  const RefNodeList& refNodeList(const size_t& i) const
  {
    return m_ref_node_list[i];
  }

  KOKKOS_INLINE_FUNCTION
  size_t numberOfNodes() const
  {
    return m_node_to_cell_matrix.numRows();
  }

  KOKKOS_INLINE_FUNCTION
  size_t numberOfCells() const
  {
    return m_cell_to_node_matrix.numRows();
  }

  const Kokkos::View<const double*> invCellNbNodes() const
  {
    return m_inv_cell_nb_nodes;
  }

  Connectivity1D(const Connectivity1D&) = delete;

  Connectivity1D(const size_t& number_of_cells)
      : Connectivity1D(_buildConnectivity(number_of_cells))
  {
    ;
  }

  Connectivity1D(const std::vector<std::vector<unsigned int>>& cell_by_node_vector)
  {
    m_cell_to_node_matrix
        = Kokkos::create_staticcrsgraph<ConnectivityMatrix>("cell_to_node_matrix",
                                                            cell_by_node_vector);

    Assert(this->numberOfCells()>0);
    {
      Kokkos::View<double*> inv_cell_nb_nodes("inv_cell_nb_nodes", this->numberOfCells());
      Kokkos::parallel_for(this->numberOfCells(), KOKKOS_LAMBDA(const size_t& j) {
          const auto& cell_nodes = m_cell_to_node_matrix.rowConst(j);
          inv_cell_nb_nodes[j] = 1./cell_nodes.length;
        });
      m_inv_cell_nb_nodes = inv_cell_nb_nodes;
    }

    {
      std::vector<std::vector<unsigned int>> node_id_per_cell_vector(this->numberOfCells());
      unsigned int id=0;
      for (unsigned int j=0; j<this->numberOfCells(); ++j) {
        const auto& cell_to_node = m_cell_to_node_matrix.rowConst(j);
        auto& node_id_per_cell = node_id_per_cell_vector[j];
        node_id_per_cell.resize(cell_to_node.length);
        for (size_t r=0; r<cell_to_node.length; ++r) {
          node_id_per_cell[r] = id++;
        }
      }
      m_node_id_per_cell_matrix
          = Kokkos::create_staticcrsgraph<ConnectivityMatrix>("node_id_per_cell_matrix",
                                                              cell_by_node_vector);
    }

    ConnectivityUtils utils;
    utils.computeNodeCellConnectivity(m_cell_to_node_matrix,
                                      m_node_to_cell_matrix,
                                      m_node_to_cell_local_node_matrix);
  }

  ~Connectivity1D()
  {
    ;
  }
};

#endif // CONNECTIVITY_1D_HPP
