#ifndef INTERPOLATE_ITEM_VALUE_HPP
#define INTERPOLATE_ITEM_VALUE_HPP

#include <language/utils/PugsFunctionAdapter.hpp>
#include <mesh/ItemType.hpp>
#include <mesh/ItemValue.hpp>

template <typename T>
class InterpolateItemValue;
template <typename OutputType, typename InputType>
class InterpolateItemValue<OutputType(InputType)> : public PugsFunctionAdapter<OutputType(InputType)>
{
  static constexpr size_t Dimension = OutputType::Dimension;
  using Adapter                     = PugsFunctionAdapter<OutputType(InputType)>;

 public:
  template <ItemType item_type>
  PUGS_INLINE static ItemValue<OutputType, item_type>
  interpolate(const FunctionSymbolId& function_symbol_id, const ItemValue<const InputType, item_type>& position)
  {
    auto& expression    = Adapter::getFunctionExpression(function_symbol_id);
    auto convert_result = Adapter::getResultConverter(expression.m_data_type);

    Array<ExecutionPolicy> context_list = Adapter::getContextList(expression);

    using execution_space = typename Kokkos::DefaultExecutionSpace::execution_space;
    Kokkos::Experimental::UniqueToken<execution_space, Kokkos::Experimental::UniqueTokenScope::Global> tokens;
    const IConnectivity& connectivity = *position.connectivity_ptr();

    ItemValue<OutputType, item_type> value(connectivity);
    using ItemId = ItemIdT<item_type>;

    parallel_for(connectivity.template numberOf<item_type>(), [=, &expression, &tokens](ItemId i) {
      const int32_t t = tokens.acquire();

      auto& execution_policy = context_list[t];

      Adapter::convertArgs(execution_policy.currentContext(), position[i]);
      auto result = expression.execute(execution_policy);
      value[i]    = convert_result(std::move(result));

      tokens.release(t);
    });

    return value;
  }

  template <ItemType item_type>
  PUGS_INLINE static Array<OutputType>
  interpolate(const FunctionSymbolId& function_symbol_id,
              const ItemValue<const InputType, item_type>& position,
              const Array<const ItemIdT<item_type>>& list_of_items)
  {
    auto& expression    = Adapter::getFunctionExpression(function_symbol_id);
    auto convert_result = Adapter::getResultConverter(expression.m_data_type);

    Array<ExecutionPolicy> context_list = Adapter::getContextList(expression);

    using execution_space = typename Kokkos::DefaultExecutionSpace::execution_space;
    Kokkos::Experimental::UniqueToken<execution_space, Kokkos::Experimental::UniqueTokenScope::Global> tokens;

    Array<OutputType> value{list_of_items.size()};
    using ItemId = ItemIdT<item_type>;

    parallel_for(list_of_items.size(), [=, &expression, &tokens](size_t i_item) {
      ItemId item_id  = list_of_items[i_item];
      const int32_t t = tokens.acquire();

      auto& execution_policy = context_list[t];

      Adapter::convertArgs(execution_policy.currentContext(), position[item_id]);
      auto result   = expression.execute(execution_policy);
      value[i_item] = convert_result(std::move(result));

      tokens.release(t);
    });

    return value;
  }

  template <ItemType item_type>
  PUGS_INLINE static Array<OutputType>
  interpolate(const FunctionSymbolId& function_symbol_id,
              const ItemValue<const InputType, item_type>& position,
              const Array<ItemIdT<item_type>>& list_of_items)
  {
    return interpolate(function_symbol_id, position, Array<const ItemIdT<item_type>>{list_of_items});
  }
};

#endif   // INTERPOLATE_ITEM_VALUE_HPP
