#ifndef NODE_BY_CELL_DATA_HPP
#define NODE_BY_CELL_DATA_HPP

#include <Kokkos_StaticCrsGraph.hpp>

typedef Kokkos::StaticCrsGraph<unsigned int, Kokkos::HostSpace> ConnectivityMatrix;

template <typename DataType>
class NodeByCellData
{
#warning should eventually filter const from DataType
 private:
  ConnectivityMatrix m_node_id_per_cell_matrix;

  Kokkos::View<unsigned int*> m_begin;
  Kokkos::View<unsigned int*> m_end;

  Kokkos::View<DataType*> m_values;

 public:
  class SubView
  {
   private:
    DataType* const m_values;
    const size_t m_size;
   public:
    KOKKOS_FORCEINLINE_FUNCTION
    const DataType& operator[](const size_t& i) const
    {
      return m_values[i];
    }

    KOKKOS_FORCEINLINE_FUNCTION
    DataType& operator[](const size_t& i)
    {
      return m_values[i];
    }

    KOKKOS_FORCEINLINE_FUNCTION
    const size_t& size() const
    {
      return m_size;
    }

    KOKKOS_FORCEINLINE_FUNCTION
    SubView(const SubView&) = delete;

    KOKKOS_FORCEINLINE_FUNCTION
    SubView(SubView&&) = default;

    KOKKOS_FORCEINLINE_FUNCTION
    SubView(Kokkos::View<DataType*> values,
            const unsigned int& begin,
            const unsigned int& end)
        : m_values(&(values[begin])),
          m_size(end-begin)
    {

    }
  };

  KOKKOS_FORCEINLINE_FUNCTION
  size_t numberOfValues() const
  {
    return m_values.extent(0);
  }

  KOKKOS_FORCEINLINE_FUNCTION
  DataType& value(const size_t& i)
  {
    return m_values[i];
  }

  KOKKOS_FORCEINLINE_FUNCTION
  const DataType& value(const size_t & i) const
  {
    return m_values[i];
  }

  KOKKOS_FORCEINLINE_FUNCTION
  size_t numberOfEntities() const
  {
    return m_node_id_per_cell_matrix.numRows();
  }

  KOKKOS_FORCEINLINE_FUNCTION
  SubView&& myCellValues(const size_t& i_cell)
  {
    ConnectivityMatrix::size_type cell_begin = m_begin[i_cell];
    ConnectivityMatrix::size_type cell_end = m_begin[i_cell+1];
    return std::move(SubView(m_values, cell_begin, cell_end));
  }

  KOKKOS_FORCEINLINE_FUNCTION
  Kokkos::View<DataType*>
  cellValues(const size_t& i_cell)
  {
    ConnectivityMatrix::size_type cell_begin = m_begin[i_cell];
    ConnectivityMatrix::size_type cell_end = m_begin[i_cell+1];
    // ConnectivityMatrix::size_type cell_begin = m_node_id_per_cell_matrix.row_map(i_cell);
    // ConnectivityMatrix::size_type cell_end = m_node_id_per_cell_matrix.row_map(i_cell+1);
    return Kokkos::subview(m_values, std::make_pair(cell_begin, cell_end));
  }

  KOKKOS_FORCEINLINE_FUNCTION
  Kokkos::View<const DataType*>
  cellValues(const size_t& i_cell) const
  {
    ConnectivityMatrix::size_type cell_begin = m_begin[i_cell];
    ConnectivityMatrix::size_type cell_end = m_end[i_cell+1];
    // ConnectivityMatrix::size_type cell_begin = m_node_id_per_cell_matrix.row_map(i_cell);
    // ConnectivityMatrix::size_type cell_end = m_node_id_per_cell_matrix.row_map(i_cell+1);
    return Kokkos::subview(m_values, std::make_pair(cell_begin, cell_end));
  }

  NodeByCellData(const ConnectivityMatrix& node_id_per_cell_matrix)
      : m_node_id_per_cell_matrix(node_id_per_cell_matrix),
        m_begin("begin", m_node_id_per_cell_matrix.numRows()+1),
        // m_end("end", m_node_id_per_cell_matrix.numRows()),
        m_values("values", m_node_id_per_cell_matrix.entries.extent(0))
  {
    for (unsigned int j=0; j<m_node_id_per_cell_matrix.numRows()+1; ++j) {
      m_begin[j] = m_node_id_per_cell_matrix.row_map(j);
      // m_end[j] = m_node_id_per_cell_matrix.row_map(j+1);
    }
  }

  ~NodeByCellData() = default;
};

#endif // NODE_BY_CELL_DATA_HPP
