#ifndef FINITE_VOLUMES_EULER_UNKNOWNS_HPP
#define FINITE_VOLUMES_EULER_UNKNOWNS_HPP

#include <TinyVector.hpp>
#include <ItemValue.hpp>

template <typename TMeshData>
class FiniteVolumesEulerUnknowns
{
public:
  using MeshDataType = TMeshData;
  using MeshType = typename MeshDataType::MeshType;

  static constexpr size_t Dimension = MeshType::Dimension;
  using Rd = TinyVector<Dimension>;

private:
  const MeshDataType& m_mesh_data;
  const MeshType& m_mesh;

  CellValue<double> m_rhoj;
  CellValue<Rd> m_uj;
  CellValue<double> m_Ej;

  CellValue<double> m_ej;
  CellValue<double> m_pj;
  CellValue<double> m_gammaj;
  CellValue<double> m_cj;
  CellValue<double> m_mj;
  CellValue<double> m_inv_mj;

public:
  CellValue<double>& rhoj()
  {
    return m_rhoj;
  }

  const CellValue<const double> rhoj() const
  {
    return m_rhoj;
  }

  CellValue<Rd>& uj()
  {
    return m_uj;
  }

  const CellValue<const Rd> uj() const
  {
    return m_uj;
  }

  CellValue<double>& Ej()
  {
    return m_Ej;
  }

  const CellValue<const double> Ej() const
  {
    return m_Ej;
  }

  CellValue<double>& ej()
  {
    return m_ej;
  }

  const CellValue<const double> ej() const
  {
    return m_ej;
  }

  CellValue<double>& pj()
  {
    return m_pj;
  }

  const CellValue<const double> pj() const
  {
    return m_pj;
  }

  CellValue<double>& gammaj()
  {
    return m_gammaj;
  }

  const CellValue<const double> gammaj() const
  {
    return m_gammaj;
  }

  CellValue<double>& cj()
  {
    return m_cj;
  }

  const CellValue<const double> cj() const
  {
    return m_cj;
  }

  CellValue<double>& mj()
  {
    return m_mj;
  }

  const CellValue<const double> mj() const
  {
    return m_mj;
  }

  CellValue<double>& invMj()
  {
    return m_inv_mj;
  }

  const CellValue<const double> invMj() const
  {
    return m_inv_mj;
  }

  void initializeSod()
  {
    const CellValue<const Rd>& xj = m_mesh_data.xj();

    parallel_for(m_mesh.numberOfCells(), PASTIS_LAMBDA(const CellId& j){
        if (xj[j][0]<0.5) {
          m_rhoj[j]=1;
        } else {
          m_rhoj[j]=0.125;
        }
      });

    parallel_for(m_mesh.numberOfCells(), PASTIS_LAMBDA(const CellId& j){
        if (xj[j][0]<0.5) {
          m_pj[j]=1;
        } else {
          m_pj[j]=0.1;
        }
      });

    parallel_for(m_mesh.numberOfCells(), PASTIS_LAMBDA(const CellId& j){
        m_uj[j] = zero;
      });

    parallel_for(m_mesh.numberOfCells(), PASTIS_LAMBDA(const CellId& j){
        m_gammaj[j] = 1.4;
      });

    BlockPerfectGas block_eos(m_rhoj, m_ej, m_pj, m_gammaj, m_cj);
    block_eos.updateEandCFromRhoP();

    parallel_for(m_mesh.numberOfCells(), PASTIS_LAMBDA(const CellId& j){
        m_Ej[j] = m_ej[j]+0.5*(m_uj[j],m_uj[j]);
      });

    const CellValue<const double>& Vj = m_mesh_data.Vj();
    parallel_for(m_mesh.numberOfCells(), PASTIS_LAMBDA(const CellId& j){
        m_mj[j] = m_rhoj[j] * Vj[j];
      });

    parallel_for(m_mesh.numberOfCells(), PASTIS_LAMBDA(const CellId& j){
        m_inv_mj[j] = 1./m_mj[j];
      });
  }

  FiniteVolumesEulerUnknowns(const MeshDataType& mesh_data)
    : m_mesh_data(mesh_data),
      m_mesh(m_mesh_data.mesh()),
      m_rhoj(m_mesh.connectivity()),
      m_uj(m_mesh.connectivity()),
      m_Ej(m_mesh.connectivity()),
      m_ej(m_mesh.connectivity()),
      m_pj(m_mesh.connectivity()),
      m_gammaj(m_mesh.connectivity()),
      m_cj(m_mesh.connectivity()),
      m_mj(m_mesh.connectivity()),
      m_inv_mj(m_mesh.connectivity())
  {
    ;
  }

};

#endif // FINITE_VOLUMES_EULER_UNKNOWNS_HPP
