#ifndef CONNECTIVITY_DISPATCHER_HPP
#define CONNECTIVITY_DISPATCHER_HPP

#include <Mesh.hpp>
#include <ItemValue.hpp>
#include <ItemValueUtils.hpp>

#include <unordered_map>

template <int Dimension>
class ConnectivityDispatcher
{
 public:
  using ConnectivityType = Connectivity<Dimension>;

 private:
  const ConnectivityType& m_connectivity;

  ConnectivityDescriptor m_new_descriptor;

  std::shared_ptr<ConnectivityType> m_dispatched_connectivity;

  CellValue<const int> m_cell_new_owner;
  FaceValue<const int> m_face_new_owner;
  NodeValue<const int> m_node_new_owner;

  const std::vector<Array<const CellId>> m_cell_list_to_send_by_proc;
#warning use a const sementic too
  std::vector<Array<const NodeId>> m_node_list_to_send_by_proc;

  Array<const int> m_nb_cell_to_send_by_proc;
  Array<const int> m_nb_cell_to_recv_by_proc;

  Array<const unsigned int> m_nb_node_to_recv_by_proc;

  std::vector<Array<const NodeId>> m_recv_node_id_correspondance_by_proc;

  std::vector<Array<int>> m_recv_cell_number_by_proc;
  std::unordered_map<int, int> m_node_number_id_map;

  CellValue<int> _getCellNewOwner();
  FaceValue<int> _getFaceNewOwner();
  NodeValue<int> _getNodeNewOwner();

  std::vector<Array<const CellId>> _buildCellListToSend() const;

  Array<int> _buildNbCellToSend();

  void _dispatchFaces();
 public:
  std::shared_ptr<const ConnectivityType>
  dispatchedConnectivity() const
  {
    return m_dispatched_connectivity;
  }

  PASTIS_INLINE
  const CellValue<const int>& cellNewOwner() const
  {
    return m_cell_new_owner;
  }

  PASTIS_INLINE
  const FaceValue<const int>& faceNewOwner() const
  {
    return m_face_new_owner;
  }

  PASTIS_INLINE
  const NodeValue<const int>& nodeNewOwner() const
  {
    return m_node_new_owner;
  }

  template<typename DataType, typename ConnectivityPtr>
  std::vector<Array<std::remove_const_t<DataType>>>
  exchange(ItemValue<DataType, ItemType::cell, ConnectivityPtr> cell_value) const
  {
    using MutableDataType = std::remove_const_t<DataType>;
    std::vector<Array<DataType>> cell_value_to_send_by_proc(parallel::size());
    for (size_t i=0; i<parallel::size(); ++i) {
      const Array<const CellId>& cell_list = m_cell_list_to_send_by_proc[i];
      Array<MutableDataType> cell_value_list(cell_list.size());
      parallel_for (cell_list.size(), PASTIS_LAMBDA(const CellId& cell_id) {
          cell_value_list[cell_id] = cell_value[cell_list[cell_id]];
        });
      cell_value_to_send_by_proc[i] = cell_value_list;
    }

    std::vector<Array<MutableDataType>> recv_cell_value_by_proc(parallel::size());
    for (size_t i=0; i<parallel::size(); ++i) {
      recv_cell_value_by_proc[i] = Array<MutableDataType>(m_nb_cell_to_recv_by_proc[i]);
    }

    parallel::exchange(cell_value_to_send_by_proc, recv_cell_value_by_proc);
    return recv_cell_value_by_proc;
  }

#warning Should write this function generically w.r. to the item type
  template<typename DataType, typename ConnectivityPtr>
  ItemValue<DataType, ItemType::node, ConnectivityPtr>
  dispatch(ItemValue<DataType, ItemType::node, ConnectivityPtr> node_value) const
  {
    Assert(m_dispatched_connectivity.use_count()> 0,
           "cannot dispatch quantity before connectivity");
    using MutableDataType = std::remove_const_t<DataType>;
    std::vector<Array<DataType>> node_value_to_send_by_proc(parallel::size());
    for (size_t i=0; i<parallel::size(); ++i) {
      const Array<const NodeId>& node_list = m_node_list_to_send_by_proc[i];
      Array<MutableDataType> node_value_list(node_list.size());
      parallel_for (node_list.size(), PASTIS_LAMBDA(const NodeId& node_id) {
          node_value_list[node_id] = node_value[node_list[node_id]];
        });
      node_value_to_send_by_proc[i] = node_value_list;
    }

    std::vector<Array<MutableDataType>> recv_node_value_by_proc(parallel::size());
    for (size_t i=0; i<parallel::size(); ++i) {
      recv_node_value_by_proc[i] = Array<MutableDataType>(m_nb_node_to_recv_by_proc[i]);
    }

    parallel::exchange(node_value_to_send_by_proc, recv_node_value_by_proc);

    NodeValue<MutableDataType> new_node_value(*m_dispatched_connectivity);
    for (size_t i_rank=0; i_rank<parallel::size(); ++i_rank) {
      for (size_t r=0; r<recv_node_value_by_proc[i_rank].size(); ++r) {
        const NodeId& node_id = m_recv_node_id_correspondance_by_proc[i_rank][r];
        new_node_value[node_id] = recv_node_value_by_proc[i_rank][r];
      }
    }
    return new_node_value;
  }

  ConnectivityDispatcher(const ConnectivityType& mesh);
  ConnectivityDispatcher(const ConnectivityDispatcher&) = delete;
  ~ConnectivityDispatcher() = default;
};


#endif // CONNECTIVITY_DISPATCHER_HPP
