#ifndef BOUNDARY_CONDITION_HANDLER_HPP
#define BOUNDARY_CONDITION_HANDLER_HPP

#include <utils/Array.hpp>

#include <mesh/MeshNodeBoundary.hpp>
#include <mesh/RefItemList.hpp>

#include <memory>
#include <vector>

class BoundaryCondition   // clazy:exclude=copyable-polymorphic
{
 public:
  enum Type
  {
    velocity,
    normal_velocity,
    pressure,
    symmetry
  };

  const Type m_type;

 protected:
  BoundaryCondition(Type type) : m_type(type)
  {
    ;
  }

 public:
  const Type&
  type() const
  {
    return m_type;
  }

  virtual ~BoundaryCondition() = default;
};

template <size_t Dimension>
class PressureBoundaryCondition final : public BoundaryCondition   // clazy:exclude=copyable-polymorphic
{
 private:
  const Array<const double> m_value_list;
  const Array<const FaceId> m_face_list;

 public:
  const Array<const FaceId>&
  faceList() const
  {
    return m_face_list;
  }

  const Array<const double>&
  valueList() const
  {
    return m_value_list;
  }

  PressureBoundaryCondition(const Array<const FaceId>& face_list, Array<const double> value_list)
    : BoundaryCondition{BoundaryCondition::pressure}, m_value_list{value_list}, m_face_list{face_list}
  {}

  ~PressureBoundaryCondition() = default;
};

template <>
class PressureBoundaryCondition<1> final : public BoundaryCondition   // clazy:exclude=copyable-polymorphic
{
 private:
  const Array<const double> m_value_list;
  const Array<const NodeId> m_face_list;

 public:
  const Array<const NodeId>&
  faceList() const
  {
    return m_face_list;
  }

  const Array<const double>&
  valueList() const
  {
    return m_value_list;
  }

  PressureBoundaryCondition(const Array<const NodeId>& face_list, Array<const double> value_list)
    : BoundaryCondition{BoundaryCondition::pressure}, m_value_list{value_list}, m_face_list{face_list}
  {}

  ~PressureBoundaryCondition() = default;
};

template <size_t dimension>
class SymmetryBoundaryCondition : public BoundaryCondition
{
 public:
  using Rd = TinyVector<dimension, double>;

 private:
  const MeshFlatNodeBoundary<dimension> m_mesh_flat_node_boundary;

 public:
  const Rd&
  outgoingNormal() const
  {
    return m_mesh_flat_node_boundary.outgoingNormal();
  }

  size_t
  numberOfNodes() const
  {
    return m_mesh_flat_node_boundary.nodeList().size();
  }

  const Array<const NodeId>&
  nodeList() const
  {
    return m_mesh_flat_node_boundary.nodeList();
  }

  SymmetryBoundaryCondition(const MeshFlatNodeBoundary<dimension>& mesh_flat_node_boundary)
    : BoundaryCondition(BoundaryCondition::symmetry), m_mesh_flat_node_boundary(mesh_flat_node_boundary)
  {
    ;
  }

  ~SymmetryBoundaryCondition() = default;
};

class BoundaryConditionHandler
{
 private:
  std::shared_ptr<BoundaryCondition> m_boundary_condition;

 public:
  const BoundaryCondition&
  boundaryCondition() const
  {
    return *m_boundary_condition;
  }

  PUGS_INLINE
  BoundaryConditionHandler& operator=(BoundaryConditionHandler&) = default;

  PUGS_INLINE
  BoundaryConditionHandler(const BoundaryConditionHandler&) = default;

  PUGS_INLINE
  BoundaryConditionHandler(BoundaryConditionHandler&&) = default;

  PUGS_INLINE
  BoundaryConditionHandler(std::shared_ptr<BoundaryCondition> boundary_condition)
    : m_boundary_condition(boundary_condition)
  {}

  ~BoundaryConditionHandler() = default;
};

#endif   // BOUNDARY_CONDITION_HANDLER_HPP
