#include <utils/PugsUtils.hpp>

#include <utils/BuildInfo.hpp>
#include <utils/RevisionInfo.hpp>

#include <utils/Messenger.hpp>

#include <utils/ConsoleManager.hpp>
#include <utils/FPEManager.hpp>
#include <utils/SignalManager.hpp>

#include <rang.hpp>

#include <Kokkos_Core.hpp>

#include <CLI/CLI.hpp>

#include <iostream>

std::string
initialize(int& argc, char* argv[])
{
  parallel::Messenger::create(argc, argv);

  std::cout << "Pugs version: " << rang::style::bold << RevisionInfo::version() << rang::style::reset << '\n';

  std::cout << "-------------------- " << rang::fg::green << "git info" << rang::fg::reset
            << " -------------------------" << '\n';
  std::cout << "tag:  " << rang::style::bold << RevisionInfo::gitTag() << rang::style::reset << '\n';
  std::cout << "HEAD: " << rang::style::bold << RevisionInfo::gitHead() << rang::style::reset << '\n';
  std::cout << "hash: " << rang::style::bold << RevisionInfo::gitHash() << rang::style::reset << "  (";

  if (RevisionInfo::gitIsClean()) {
    std::cout << rang::fgB::green << "clean" << rang::fg::reset;
  } else {
    std::cout << rang::fgB::red << "dirty" << rang::fg::reset;
  }
  std::cout << ")\n";
  std::cout << "-------------------- " << rang::fg::green << "build info" << rang::fg::reset
            << " -----------------------" << '\n';
  std::cout << "type:     " << rang::style::bold << BuildInfo::type() << rang::style::reset << '\n';
  std::cout << "compiler: " << rang::style::bold << BuildInfo::compiler() << rang::style::reset << '\n';
  std::cout << "kokkos:   " << rang::style::bold << BuildInfo::kokkosDevices() << rang::style::reset << '\n';
  std::cout << "mpi:      " << rang::style::bold << BuildInfo::mpiLibrary() << rang::style::reset << '\n';
  std::cout << "-------------------------------------------------------\n";

  std::string filename;
  {
    CLI::App app{"Pugs help"};

    app.add_option("filename,-f,--filename", filename, "gmsh file");

    int threads = -1;
    app.add_option("--threads", threads, "Number of Kokkos threads")
      ->check(CLI::Range(1, std::numeric_limits<decltype(threads)>::max()));

    bool enable_color = true;
    app.add_flag("--color,!--no-color", enable_color, "Colorize console output [default: true]");

    bool enable_fpe = true;
    app.add_flag("--fpe,!--no-fpe", enable_fpe, "Trap floating point exceptions [default: true]");

    bool enable_signals = true;
    app.add_flag("--signal,!--no-signal", enable_signals, "Catches signals [default: true]");

    bool pause_on_error = false;
    app.add_flag("-p,--pause-on-error", pause_on_error, "Pause for debugging on unexpected error [default: false]");

    std::atexit([]() { std::cout << rang::style::reset; });
    try {
      app.parse(argc, argv);
    }
    catch (const CLI::ParseError& e) {
      parallel::Messenger::destroy();
      std::exit(app.exit(e, std::cout, std::cerr));
    }

    ConsoleManager::init(enable_color);
    FPEManager::init(enable_fpe);
    SignalManager::setPauseForDebug(pause_on_error);
    SignalManager::init(enable_signals);
  }

  Kokkos::initialize(argc, argv);
  std::cout << "-------------------- " << rang::fg::green << "exec info" << rang::fg::reset
            << " ------------------------" << '\n';

  std::cout << rang::style::bold;
  Kokkos::DefaultExecutionSpace::print_configuration(std::cout);
  std::cout << rang::style::reset;
  std::cout << "-------------------------------------------------------\n";

  return filename;
}

void
finalize()
{
  Kokkos::finalize();
  parallel::Messenger::destroy();
}
