#ifndef AST_NODE_DATA_TYPE_HPP
#define AST_NODE_DATA_TYPE_HPP

#include <utils/PugsAssert.hpp>

#include <limits>
#include <memory>
#include <string>

class ASTNode;

class ASTNodeDataType
{
 public:
  enum DataType : int32_t
  {
    undefined_t        = -1,
    bool_t             = 0,
    int_t              = 1,
    unsigned_int_t     = 2,
    double_t           = 3,
    vector_t           = 4,
    tuple_t            = 5,
    list_t             = 6,
    string_t           = 7,
    typename_t         = 10,
    type_name_id_t     = 11,
    type_id_t          = 21,
    function_t         = 22,
    builtin_function_t = 23,
    void_t             = std::numeric_limits<int32_t>::max()
  };

 private:
  DataType m_data_type;
  std::shared_ptr<ASTNodeDataType> m_content_type;
  size_t m_dimension;
  std::string m_name_of_type_id;

 public:
  PUGS_INLINE
  size_t
  dimension() const
  {
    return m_dimension;
  }

  PUGS_INLINE
  const ASTNodeDataType&
  contentType() const
  {
    Assert(m_content_type);
    return *m_content_type;
  }

  PUGS_INLINE
  const std::string&
  nameOfTypeId() const
  {
    return m_name_of_type_id;
  }

  PUGS_INLINE
  operator const DataType&() const
  {
    return m_data_type;
  }

  ASTNodeDataType& operator=(const ASTNodeDataType&) = default;
  ASTNodeDataType& operator=(ASTNodeDataType&&) = default;

  ASTNodeDataType(DataType data_type)
    : m_data_type{data_type}, m_content_type{nullptr}, m_dimension{1}, m_name_of_type_id{"unknown"}
  {}

  ASTNodeDataType(DataType data_type, const ASTNodeDataType& content_type)
    : m_data_type{data_type},
      m_content_type{std::make_shared<ASTNodeDataType>(content_type)},
      m_dimension{1},
      m_name_of_type_id{"unknown"}
  {}

  ASTNodeDataType(DataType data_type, size_t dimension)
    : m_data_type{data_type}, m_content_type{nullptr}, m_dimension{dimension}, m_name_of_type_id{"unknown"}
  {}

  ASTNodeDataType(DataType data_type, const std::string& type_name)
    : m_data_type{data_type}, m_content_type{nullptr}, m_dimension{1}, m_name_of_type_id{type_name}
  {}

  ASTNodeDataType(const ASTNodeDataType&) = default;

  ASTNodeDataType(ASTNodeDataType&&) = default;

  ~ASTNodeDataType() = default;
};

ASTNodeDataType getVectorDataType(const ASTNode& type_node);

std::string dataTypeName(const ASTNodeDataType& data_type);

ASTNodeDataType dataTypePromotion(const ASTNodeDataType& data_type_1, const ASTNodeDataType& data_type_2);

bool isNaturalConversion(const ASTNodeDataType& data_type, const ASTNodeDataType& target_data_type);

#endif   // AST_NODE_DATA_TYPE_HPP
