#ifndef POLYNOMIAL_CENTERED_CANONICAL_BASIS_VIEW_HPP
#define POLYNOMIAL_CENTERED_CANONICAL_BASIS_VIEW_HPP

#include <algebra/TinyVector.hpp>
#include <utils/Array.hpp>
#include <utils/Exceptions.hpp>

template <size_t Dimension, typename DataType>
class PolynomialCenteredCanonicalBasisView
{
 public:
  class CoefficientsView
  {
   private:
    DataType* const m_values;
    const size_t m_size;

   public:
    [[nodiscard]] PUGS_INLINE size_t
    size() const
    {
      return m_size;
    }

    [[nodiscard]] PUGS_INLINE DataType&
    operator[](size_t i) const
    {
      Assert(i < m_size, "invalid index");
      return m_values[i];
    }

    CoefficientsView& operator=(const CoefficientsView&) = delete;
    CoefficientsView& operator=(CoefficientsView&&)      = delete;

    CoefficientsView(DataType* values, size_t size) : m_values{values}, m_size{size}
    {
      ;
    }

    // To try to keep these views close to the initial array one
    // forbids copy constructor and take benefits of C++-17 copy
    // elisions.
    CoefficientsView(const CoefficientsView&) = delete;

    CoefficientsView()  = delete;
    ~CoefficientsView() = default;
  };

 private:
  static_assert((Dimension > 0) and (Dimension <= 3), "invalid dimension");

  const size_t m_degree;

  // Coefficients are stored in the following order given the shifting
  // value is omitted here for simplicity (ie: xj=0).
  //
  // ----------------------------------------------------
  // For a polynomial of degree n depending on x, one has
  // 1, x, x^2, ..., x^n
  //
  // ----------------------------------------------------
  // For a polynomial of degree n depending on x and y, one has
  // 1,   x,   x^2, ...,   x^{n-1}, x^n
  // y, x y, x^2 y, ..., x^{n-1} y
  // ...
  // y^{n-1}, x y^{n-1}
  // y^n
  //
  // ----------------------------------------------------
  // For a polynomial of degree n depending on x, y and z, one has
  // 1,   x,   x^2, ...,   x^{n-1}, x^n
  // y, x y, x^2 y, ..., x^{n-1} y
  // ...
  // y^{n-1}, x y^{n-1}
  // y^n
  //
  //   z,   x z,   x^2 z, ...,   x^{n-2} z, x^{n-1} z
  // y z, x y z, x^2 y z, ..., x^{n-2} y z
  // ...
  // y^{n-2}  z, x y^{n-2} z
  // y^{n -1} z
  // ...
  // ...
  // z^{n-2},      x z^{n-2}, x^2 z^{n-2}
  // y z^{n-2},  x y z^{n-2}
  // y^2 z^{n-2}
  //
  // z^{n-1},      x z^{n-1}
  // y z^{n-1}
  //
  // z^n

  CoefficientsView m_coefficients;
  const TinyVector<Dimension>& m_xj;

 public:
  static size_t
  dimensionFromDegree(size_t degree)
  {
    if constexpr (Dimension == 1) {
      return degree + 1;
    } else if constexpr (Dimension == 2) {
      return ((degree + 2) * (degree + 1)) / 2;
    } else {   // Dimension == 3
      return ((degree + 3) * (degree + 2) * (degree + 1)) / 6;
    }
  }

  static size_t
  degreeFromDimension(size_t polynomial_basis_dimension)
  {
    Assert(polynomial_basis_dimension > 0);
    if constexpr (Dimension == 1) {
      return polynomial_basis_dimension - 1;
    } else {
      // No need fir an explicit formula
      // - degree is small and integer
      // - do not need the use of sqrt
      for (size_t degree = 0; degree < polynomial_basis_dimension; ++degree) {
        size_t dimension_from_degree = dimensionFromDegree(degree);
        if (dimension_from_degree == polynomial_basis_dimension) {
          return degree;
        } else if (dimension_from_degree > polynomial_basis_dimension) {
          break;
        }
      }
      throw NormalError("incorrect polynomial basis dimension");
    }
  }

  DataType
  operator()(const TinyVector<Dimension>& x) const
  {
    if constexpr (Dimension == 1) {
      const double x_xj = (x - m_xj)[0];

      std::remove_const_t<DataType> result = m_coefficients[m_degree];
      for (ssize_t i = m_degree - 1; i >= 0; --i) {
        result = x_xj * result + m_coefficients[i];
      }

      return result;
    } else if constexpr (Dimension == 2) {
      const TinyVector X_Xj = x - m_xj;
      const double& x_xj    = X_Xj[0];
      const double& y_yj    = X_Xj[1];

      size_t i = m_coefficients.size() - 1;

      std::remove_const_t<DataType> result = m_coefficients[i--];
      for (ssize_t i_y = m_degree - 1; i_y >= 0; --i_y) {
        std::remove_const_t<DataType> x_result = m_coefficients[i--];
        for (ssize_t i_x = m_degree - i_y - 1; i_x >= 0; --i_x) {
          x_result = x_xj * x_result + m_coefficients[i--];
        }
        result = y_yj * result + x_result;
      }

      return result;
    } else {   // Dimension == 3
      const TinyVector X_Xj = x - m_xj;
      const double& x_xj    = X_Xj[0];
      const double& y_yj    = X_Xj[1];
      const double& z_zj    = X_Xj[2];

      size_t i = m_coefficients.size() - 1;

      std::remove_const_t<DataType> result = m_coefficients[i--];
      for (ssize_t i_z = m_degree - 1; i_z >= 0; --i_z) {
        std::remove_const_t<DataType> y_result = m_coefficients[i--];
        for (ssize_t i_y = m_degree - i_z - 1; i_y >= 0; --i_y) {
          std::remove_const_t<DataType> x_result = m_coefficients[i--];
          for (ssize_t i_x = m_degree - i_z - i_y - 1; i_x >= 0; --i_x) {
            x_result = x_xj * x_result + m_coefficients[i--];
          }
          y_result = y_yj * y_result + x_result;
        }
        result = z_zj * result + y_result;
      }

      return result;
    }
  }

  template <typename ArrayType>
  PolynomialCenteredCanonicalBasisView(const size_t degree,
                                       ArrayType& coefficient_list,
                                       const TinyVector<Dimension>& xj)
    : m_degree{degree}, m_coefficients{&(coefficient_list[0]), coefficient_list.size()}, m_xj{xj}
  {}

  template <typename ArrayType>
  PolynomialCenteredCanonicalBasisView(const size_t degree,
                                       const ArrayType& coefficient_list,
                                       const TinyVector<Dimension>& xj)
    : m_degree{degree}, m_coefficients{&(coefficient_list[0]), coefficient_list.size()}, m_xj{xj}
  {}

  PolynomialCenteredCanonicalBasisView(const PolynomialCenteredCanonicalBasisView&) = delete;
  PolynomialCenteredCanonicalBasisView(PolynomialCenteredCanonicalBasisView&&)      = default;
  PolynomialCenteredCanonicalBasisView()                                            = delete;
  ~PolynomialCenteredCanonicalBasisView()                                           = default;
};

#endif   // POLYNOMIAL_CENTERED_CANONICAL_BASIS_VIEW_HPP
