#ifndef DISCRETE_FUNCTION_DPK_VARIANT_HPP
#define DISCRETE_FUNCTION_DPK_VARIANT_HPP

#include <scheme/DiscreteFunctionDPk.hpp>
#include <scheme/DiscreteFunctionDPkVector.hpp>

#include <memory>
#include <variant>

class DiscreteFunctionDPkVariant
{
 public:
  using Variant = std::variant<DiscreteFunctionDPk<1, const double>,
                               DiscreteFunctionDPk<1, const TinyVector<1>>,
                               DiscreteFunctionDPk<1, const TinyVector<2>>,
                               DiscreteFunctionDPk<1, const TinyVector<3>>,
                               DiscreteFunctionDPk<1, const TinyMatrix<1>>,
                               DiscreteFunctionDPk<1, const TinyMatrix<2>>,
                               DiscreteFunctionDPk<1, const TinyMatrix<3>>,

                               DiscreteFunctionDPk<2, const double>,
                               DiscreteFunctionDPk<2, const TinyVector<1>>,
                               DiscreteFunctionDPk<2, const TinyVector<2>>,
                               DiscreteFunctionDPk<2, const TinyVector<3>>,
                               DiscreteFunctionDPk<2, const TinyMatrix<1>>,
                               DiscreteFunctionDPk<2, const TinyMatrix<2>>,
                               DiscreteFunctionDPk<2, const TinyMatrix<3>>,

                               DiscreteFunctionDPk<3, const double>,
                               DiscreteFunctionDPk<3, const TinyVector<1>>,
                               DiscreteFunctionDPk<3, const TinyVector<2>>,
                               DiscreteFunctionDPk<3, const TinyVector<3>>,
                               DiscreteFunctionDPk<3, const TinyMatrix<1>>,
                               DiscreteFunctionDPk<3, const TinyMatrix<2>>,
                               DiscreteFunctionDPk<3, const TinyMatrix<3>>,

                               DiscreteFunctionDPkVector<1, const double>,
                               DiscreteFunctionDPkVector<1, const TinyVector<1>>,
                               DiscreteFunctionDPkVector<1, const TinyVector<2>>,
                               DiscreteFunctionDPkVector<1, const TinyVector<3>>,
                               DiscreteFunctionDPkVector<1, const TinyMatrix<1>>,
                               DiscreteFunctionDPkVector<1, const TinyMatrix<2>>,
                               DiscreteFunctionDPkVector<1, const TinyMatrix<3>>,

                               DiscreteFunctionDPkVector<2, const double>,
                               DiscreteFunctionDPkVector<2, const TinyVector<1>>,
                               DiscreteFunctionDPkVector<2, const TinyVector<2>>,
                               DiscreteFunctionDPkVector<2, const TinyVector<3>>,
                               DiscreteFunctionDPkVector<2, const TinyMatrix<1>>,
                               DiscreteFunctionDPkVector<2, const TinyMatrix<2>>,
                               DiscreteFunctionDPkVector<2, const TinyMatrix<3>>,

                               DiscreteFunctionDPkVector<3, const double>,
                               DiscreteFunctionDPkVector<3, const TinyVector<1>>,
                               DiscreteFunctionDPkVector<3, const TinyVector<2>>,
                               DiscreteFunctionDPkVector<3, const TinyVector<3>>,
                               DiscreteFunctionDPkVector<3, const TinyMatrix<1>>,
                               DiscreteFunctionDPkVector<3, const TinyMatrix<2>>,
                               DiscreteFunctionDPkVector<3, const TinyMatrix<3>>>;

 private:
  Variant m_discrete_function_dpk;

 public:
  PUGS_INLINE
  const Variant&
  discreteFunctionDPk() const
  {
    return m_discrete_function_dpk;
  }

  template <typename DiscreteFunctionDPkT>
  PUGS_INLINE auto
  get() const
  {
    static_assert(is_discrete_function_dpk_v<DiscreteFunctionDPkT>, "invalid template argument");
    using DataType = typename DiscreteFunctionDPkT::data_type;
    static_assert(std::is_const_v<DataType>, "data type of extracted discrete function must be const");

#ifndef NDEBUG
    if (not std::holds_alternative<DiscreteFunctionDPkT>(this->m_discrete_function_dpk)) {
      std::ostringstream error_msg;
      error_msg << "invalid discrete function type\n";
      error_msg << "- required " << rang::fgB::red << demangle<DiscreteFunctionDPkT>() << rang::fg::reset << '\n';
      error_msg << "- contains " << rang::fgB::yellow
                << std::visit([](auto&& f) -> std::string { return demangle<decltype(f)>(); },
                              this->m_discrete_function_dpk)
                << rang::fg::reset;
      throw NormalError(error_msg.str());
    }
#endif   // NDEBUG

    return std::get<DiscreteFunctionDPkT>(this->discreteFunctionDPk());
  }

  template <size_t Dimension, typename DataType>
  DiscreteFunctionDPkVariant(const DiscreteFunctionDPk<Dimension, DataType>& discrete_function_dpk)
    : m_discrete_function_dpk{DiscreteFunctionDPk<Dimension, const DataType>{discrete_function_dpk}}
  {
    static_assert(std::is_same_v<std::remove_const_t<DataType>, double> or                       //
                    std::is_same_v<std::remove_const_t<DataType>, TinyVector<1, double>> or      //
                    std::is_same_v<std::remove_const_t<DataType>, TinyVector<2, double>> or      //
                    std::is_same_v<std::remove_const_t<DataType>, TinyVector<3, double>> or      //
                    std::is_same_v<std::remove_const_t<DataType>, TinyMatrix<1, 1, double>> or   //
                    std::is_same_v<std::remove_const_t<DataType>, TinyMatrix<2, 2, double>> or   //
                    std::is_same_v<std::remove_const_t<DataType>, TinyMatrix<3, 3, double>>,
                  "DiscreteFunctionDPk with this DataType is not allowed in variant");
  }

  template <size_t Dimension, typename DataType>
  DiscreteFunctionDPkVariant(const DiscreteFunctionDPkVector<Dimension, DataType>& discrete_function_dpk)
    : m_discrete_function_dpk{DiscreteFunctionDPkVector<Dimension, const DataType>{discrete_function_dpk}}
  {
    static_assert(std::is_same_v<std::remove_const_t<DataType>, double> or                       //
                    std::is_same_v<std::remove_const_t<DataType>, TinyVector<1, double>> or      //
                    std::is_same_v<std::remove_const_t<DataType>, TinyVector<2, double>> or      //
                    std::is_same_v<std::remove_const_t<DataType>, TinyVector<3, double>> or      //
                    std::is_same_v<std::remove_const_t<DataType>, TinyMatrix<1, 1, double>> or   //
                    std::is_same_v<std::remove_const_t<DataType>, TinyMatrix<2, 2, double>> or   //
                    std::is_same_v<std::remove_const_t<DataType>, TinyMatrix<3, 3, double>>,
                  "DiscreteFunctionDPkVector with this DataType is not allowed in variant");
  }

  DiscreteFunctionDPkVariant& operator=(DiscreteFunctionDPkVariant&&)      = default;
  DiscreteFunctionDPkVariant& operator=(const DiscreteFunctionDPkVariant&) = default;

  DiscreteFunctionDPkVariant(const DiscreteFunctionDPkVariant&) = default;
  DiscreteFunctionDPkVariant(DiscreteFunctionDPkVariant&&)      = default;

  DiscreteFunctionDPkVariant()  = delete;
  ~DiscreteFunctionDPkVariant() = default;
};

#endif   // DISCRETE_FUNCTION_DPK_VARIANT_HPP
