#ifndef ARRAY_HPP
#define ARRAY_HPP

#include <PastisMacros.hpp>
#include <Kokkos_Core.hpp>

#include <PastisAssert.hpp>

template <typename DataType>
class Array
{
 public:
  using data_type = DataType;
  using index_type = size_t;

 private:
  Kokkos::View<DataType*> m_values;

  // Allows const version to access our data
  friend Array<std::add_const_t<DataType>>;

 public:
  PASTIS_INLINE
  size_t size() const
  {
    return m_values.extent(0);
  }

  PASTIS_INLINE
  DataType& operator[](const index_type& i) const
  {
    Assert(i<m_values.extent(0));
    return m_values[i];
  }

  PASTIS_INLINE
  Array(const size_t& size)
      : m_values("anonymous", size)
  {
    static_assert(not std::is_const<DataType>(),
                  "Cannot allocate Array of const data: only view is supported");
  }

  template <typename DataType2>
  PASTIS_INLINE
  Array& operator=(const Array<DataType2>& array)
  {
    // ensures that DataType is the same as source DataType2
    static_assert(std::is_same<std::remove_const_t<DataType>, std::remove_const_t<DataType2>>(),
                  "Cannot assign Array of different type");
    // ensures that const is not lost through copy
    static_assert(((std::is_const<DataType2>() and std::is_const<DataType>())
                   or not std::is_const<DataType2>()),
                  "Cannot assign Array of const to  Array of non-const");
    m_values = array.m_values;
    return *this;
  }

  PASTIS_INLINE
  Array& operator=(const Array&) = default;

  PASTIS_INLINE
  Array& operator=(Array&&) = default;

  PASTIS_INLINE
  Array() = default;

  template <typename DataType2>
  PASTIS_INLINE
  Array(const Array<DataType2>& array)
  {
    this->operator=(array);
  }

  PASTIS_INLINE
  Array(Array&&) = default;

  PASTIS_INLINE
  Array(const Array&) = default;

  PASTIS_INLINE
  ~Array() = default;
};

#endif // ARRAY_HPP
