#include <language/utils/EmbeddedIDiscreteFunctionOperators.hpp>

#include <language/node_processor/BinaryExpressionProcessor.hpp>
#include <scheme/DiscreteFunctionP0.hpp>
#include <scheme/IDiscreteFunction.hpp>
#include <utils/Exceptions.hpp>

template <typename T>
PUGS_INLINE std::string
name(const T&)
{
  return dataTypeName(ast_node_data_type_from<T>);
}

template <>
PUGS_INLINE std::string
name(const IDiscreteFunction& f)
{
  return "Vh(" + dataTypeName(f.dataType()) + ")";
}

template <>
PUGS_INLINE std::string
name(const std::shared_ptr<const IDiscreteFunction>& f)
{
  return "Vh(" + dataTypeName(f->dataType()) + ")";
}

template <typename LHS_T, typename RHS_T>
PUGS_INLINE std::string
invalid_operands(const LHS_T& f, const RHS_T& g)
{
  std::ostringstream os;
  os << "undefined binary operator\n";
  os << "note: incompatible operand types " << name(f) << " and " << name(g);
  return os.str();
}

template <typename BinOperatorT, typename DiscreteFunctionT>
std::shared_ptr<const IDiscreteFunction>
innerCompositionLaw(const DiscreteFunctionT& lhs, const DiscreteFunctionT& rhs)
{
  Assert(lhs.mesh() == rhs.mesh());
  using data_type = typename DiscreteFunctionT::data_type;
  if constexpr ((std::is_same_v<language::multiply_op, BinOperatorT> and is_tiny_vector_v<data_type>) or
                (std::is_same_v<language::divide_op, BinOperatorT> and not std::is_arithmetic_v<data_type>)) {
    throw NormalError(invalid_operands(lhs, rhs));
  } else {
    return std::make_shared<decltype(BinOp<BinOperatorT>{}.eval(lhs, rhs))>(BinOp<BinOperatorT>{}.eval(lhs, rhs));
  }
}

template <typename BinOperatorT, size_t Dimension>
std::shared_ptr<const IDiscreteFunction>
innerCompositionLaw(const std::shared_ptr<const IDiscreteFunction>& f,
                    const std::shared_ptr<const IDiscreteFunction>& g)
{
  Assert(f->mesh() == g->mesh());
  Assert(f->dataType() == g->dataType());

  switch (f->dataType()) {
  case ASTNodeDataType::double_t: {
    auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, double>&>(*f);
    auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, double>&>(*g);

    return innerCompositionLaw<BinOperatorT>(fh, gh);
  }
  case ASTNodeDataType::vector_t: {
    switch (f->dataType().dimension()) {
    case 1: {
      auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<1>>&>(*f);
      auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<1>>&>(*g);

      return innerCompositionLaw<BinOperatorT>(fh, gh);
    }
    case 2: {
      auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<2>>&>(*f);
      auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<2>>&>(*g);

      return innerCompositionLaw<BinOperatorT>(fh, gh);
    }
    case 3: {
      auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<3>>&>(*f);
      auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<3>>&>(*g);

      return innerCompositionLaw<BinOperatorT>(fh, gh);
    }
    default: {
      throw NormalError(invalid_operands(f, g));
    }
    }
  }
  case ASTNodeDataType::matrix_t: {
    Assert(f->dataType().nbRows() == f->dataType().nbColumns());
    switch (f->dataType().nbRows()) {
    case 1: {
      auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<1>>&>(*f);
      auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<1>>&>(*g);

      return innerCompositionLaw<BinOperatorT>(fh, gh);
    }
    case 2: {
      auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<2>>&>(*f);
      auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<2>>&>(*g);

      return innerCompositionLaw<BinOperatorT>(fh, gh);
    }
    case 3: {
      auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<3>>&>(*f);
      auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<3>>&>(*g);

      return innerCompositionLaw<BinOperatorT>(fh, gh);
    }
    default: {
      throw UnexpectedError("invalid data type Vh(" + dataTypeName(g->dataType()) + ")");
    }
    }
  }
  default: {
    throw UnexpectedError("invalid data type Vh(" + dataTypeName(g->dataType()) + ")");
  }
  }
}

template <typename BinOperatorT>
std::shared_ptr<const IDiscreteFunction>
innerCompositionLaw(const std::shared_ptr<const IDiscreteFunction>& f,
                    const std::shared_ptr<const IDiscreteFunction>& g)
{
  if (f->mesh() != g->mesh()) {
    throw NormalError("discrete functions defined on different meshes");
  }
  if (f->dataType() != g->dataType()) {
    throw NormalError(invalid_operands(f, g));
  }

  switch (f->mesh()->dimension()) {
  case 1: {
    return innerCompositionLaw<BinOperatorT, 1>(f, g);
  }
  case 2: {
    return innerCompositionLaw<BinOperatorT, 2>(f, g);
  }
  case 3: {
    return innerCompositionLaw<BinOperatorT, 3>(f, g);
  }
  default: {
    throw UnexpectedError("invalid mesh dimension");
  }
  }
}

template <typename BinOperatorT, typename LeftDiscreteFunctionT, typename RightDiscreteFunctionT>
std::shared_ptr<const IDiscreteFunction>
applyBinaryOperation(const LeftDiscreteFunctionT& lhs, const RightDiscreteFunctionT& rhs)
{
  Assert(lhs.mesh() == rhs.mesh());
  using lhs_data_type = typename LeftDiscreteFunctionT::data_type;
  using rhs_data_type = typename RightDiscreteFunctionT::data_type;

  static_assert(not std::is_same_v<rhs_data_type, lhs_data_type>,
                "use innerCompositionLaw when data types are the same");

  return std::make_shared<decltype(BinOp<BinOperatorT>{}.eval(lhs, rhs))>(BinOp<BinOperatorT>{}.eval(lhs, rhs));
}

template <typename BinOperatorT, size_t Dimension, typename DiscreteFunctionT>
std::shared_ptr<const IDiscreteFunction>
applyBinaryOperation(const DiscreteFunctionT& fh, const std::shared_ptr<const IDiscreteFunction>& g)
{
  Assert(fh.mesh() == g->mesh());
  Assert(fh.dataType() != g->dataType());
  using lhs_data_type = std::decay_t<typename DiscreteFunctionT::data_type>;

  switch (g->dataType()) {
  case ASTNodeDataType::double_t: {
    if constexpr (not std::is_same_v<lhs_data_type, double>) {
      if constexpr (not is_tiny_matrix_v<lhs_data_type>) {
        auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, double>&>(*g);

        return applyBinaryOperation<BinOperatorT>(fh, gh);
      } else {
        throw NormalError(invalid_operands(fh, g));
      }
    } else {
      throw UnexpectedError("should have called innerCompositionLaw");
    }
  }
  case ASTNodeDataType::vector_t: {
    if constexpr (std::is_same_v<language::multiply_op, BinOperatorT>) {
      switch (g->dataType().dimension()) {
      case 1: {
        if constexpr (not is_tiny_vector_v<lhs_data_type> and
                      (std::is_same_v<lhs_data_type, TinyMatrix<1>> or std::is_same_v<lhs_data_type, double>)) {
          auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<1>>&>(*g);

          return applyBinaryOperation<BinOperatorT>(fh, gh);
        } else {
          throw NormalError(invalid_operands(fh, g));
        }
      }
      case 2: {
        if constexpr (not is_tiny_vector_v<lhs_data_type> and
                      (std::is_same_v<lhs_data_type, TinyMatrix<2>> or std::is_same_v<lhs_data_type, double>)) {
          auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<2>>&>(*g);

          return applyBinaryOperation<BinOperatorT>(fh, gh);
        } else {
          throw NormalError(invalid_operands(fh, g));
        }
      }
      case 3: {
        if constexpr (not is_tiny_vector_v<lhs_data_type> and
                      (std::is_same_v<lhs_data_type, TinyMatrix<3>> or std::is_same_v<lhs_data_type, double>)) {
          auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<3>>&>(*g);

          return applyBinaryOperation<BinOperatorT>(fh, gh);
        } else {
          throw NormalError(invalid_operands(fh, g));
        }
      }
      default: {
        throw UnexpectedError("invalid rhs data type Vh(" + dataTypeName(g->dataType()) + ")");
      }
      }
    } else {
      throw NormalError(invalid_operands(fh, g));
    }
  }
  case ASTNodeDataType::matrix_t: {
    Assert(g->dataType().nbRows() == g->dataType().nbColumns());
    if constexpr (std::is_same_v<lhs_data_type, double> and std::is_same_v<language::multiply_op, BinOperatorT>) {
      switch (g->dataType().nbRows()) {
      case 1: {
        auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<1>>&>(*g);

        return applyBinaryOperation<BinOperatorT>(fh, gh);
      }
      case 2: {
        auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<2>>&>(*g);

        return applyBinaryOperation<BinOperatorT>(fh, gh);
      }
      case 3: {
        auto gh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<3>>&>(*g);

        return applyBinaryOperation<BinOperatorT>(fh, gh);
      }
      default: {
        throw UnexpectedError("invalid rhs data type Vh(" + dataTypeName(g->dataType()) + ")");
      }
      }
    } else {
      throw NormalError(invalid_operands(fh, g));
    }
  }
  default: {
    throw UnexpectedError("invalid rhs data type Vh(" + dataTypeName(g->dataType()) + ")");
  }
  }
}

template <typename BinOperatorT, size_t Dimension>
std::shared_ptr<const IDiscreteFunction>
applyBinaryOperation(const std::shared_ptr<const IDiscreteFunction>& f,
                     const std::shared_ptr<const IDiscreteFunction>& g)
{
  Assert(f->mesh() == g->mesh());
  Assert(f->dataType() != g->dataType());

  switch (f->dataType()) {
  case ASTNodeDataType::double_t: {
    auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, double>&>(*f);

    return applyBinaryOperation<BinOperatorT, Dimension>(fh, g);
  }
  case ASTNodeDataType::matrix_t: {
    Assert(f->dataType().nbRows() == f->dataType().nbColumns());
    switch (f->dataType().nbRows()) {
    case 1: {
      auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<1>>&>(*f);

      return applyBinaryOperation<BinOperatorT, Dimension>(fh, g);
    }
    case 2: {
      auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<2>>&>(*f);

      return applyBinaryOperation<BinOperatorT, Dimension>(fh, g);
    }
    case 3: {
      auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<3>>&>(*f);

      return applyBinaryOperation<BinOperatorT, Dimension>(fh, g);
    }
    default: {
      throw UnexpectedError("invalid lhs data type Vh(" + dataTypeName(f->dataType()) + ")");
    }
    }
  }
  default: {
    throw NormalError(invalid_operands(f, g));
  }
  }
}

template <typename BinOperatorT>
std::shared_ptr<const IDiscreteFunction>
applyBinaryOperation(const std::shared_ptr<const IDiscreteFunction>& f,
                     const std::shared_ptr<const IDiscreteFunction>& g)
{
  if (f->mesh() != g->mesh()) {
    throw NormalError("functions defined on different meshes");
  }

  Assert(f->dataType() != g->dataType(), "should call inner composition instead");

  switch (f->mesh()->dimension()) {
  case 1: {
    return applyBinaryOperation<BinOperatorT, 1>(f, g);
  }
  case 2: {
    return applyBinaryOperation<BinOperatorT, 2>(f, g);
  }
  case 3: {
    return applyBinaryOperation<BinOperatorT, 3>(f, g);
  }
  default: {
    throw UnexpectedError("invalid mesh dimension");
  }
  }
}

std::shared_ptr<const IDiscreteFunction>
operator+(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  return innerCompositionLaw<language::plus_op>(f, g);
}

std::shared_ptr<const IDiscreteFunction>
operator-(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  return innerCompositionLaw<language::minus_op>(f, g);
}

std::shared_ptr<const IDiscreteFunction>
operator*(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  if (f->dataType() == g->dataType()) {
    return innerCompositionLaw<language::multiply_op>(f, g);
  } else {
    return applyBinaryOperation<language::multiply_op>(f, g);
  }
}

std::shared_ptr<const IDiscreteFunction>
operator/(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  if (f->dataType() == g->dataType()) {
    return innerCompositionLaw<language::divide_op>(f, g);
  } else {
    return applyBinaryOperation<language::divide_op>(f, g);
  }
}

template <typename BinOperatorT, typename DataType, typename DiscreteFunctionT>
std::shared_ptr<const IDiscreteFunction>
applyBinaryOperationWithLeftConstant(const DataType& a, const DiscreteFunctionT& f)
{
  using lhs_data_type = std::decay_t<DataType>;
  using rhs_data_type = std::decay_t<typename DiscreteFunctionT::data_type>;

  if constexpr (std::is_same_v<language::multiply_op, BinOperatorT>) {
    if constexpr (std::is_same_v<lhs_data_type, double>) {
      return std::make_shared<decltype(BinOp<BinOperatorT>{}.eval(a, f))>(BinOp<BinOperatorT>{}.eval(a, f));
    } else if constexpr (is_tiny_matrix_v<lhs_data_type> and
                         (is_tiny_matrix_v<rhs_data_type> or is_tiny_vector_v<rhs_data_type>)) {
      return std::make_shared<decltype(BinOp<BinOperatorT>{}.eval(a, f))>(BinOp<BinOperatorT>{}.eval(a, f));
    } else {
      throw NormalError(invalid_operands(a, f));
    }
  } else {
    throw NormalError(invalid_operands(a, f));
  }
}

template <typename BinOperatorT, size_t Dimension, typename DataType>
std::shared_ptr<const IDiscreteFunction>
applyBinaryOperationWithLeftConstant(const DataType& a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  switch (f->dataType()) {
  case ASTNodeDataType::bool_t:
  case ASTNodeDataType::unsigned_int_t:
  case ASTNodeDataType::int_t:
  case ASTNodeDataType::double_t: {
    auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, double>&>(*f);
    return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
  }
  case ASTNodeDataType::vector_t: {
    if constexpr (is_tiny_matrix_v<DataType>) {
      switch (f->dataType().dimension()) {
      case 1: {
        if constexpr (std::is_same_v<DataType, TinyMatrix<1>>) {
          auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<1>>&>(*f);
          return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
        } else {
          throw NormalError(invalid_operands(a, f));
        }
      }
      case 2: {
        if constexpr (std::is_same_v<DataType, TinyMatrix<2>>) {
          auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<2>>&>(*f);
          return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
        } else {
          throw NormalError(invalid_operands(a, f));
        }
      }
      case 3: {
        if constexpr (std::is_same_v<DataType, TinyMatrix<3>>) {
          auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<3>>&>(*f);
          return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
        } else {
          throw NormalError(invalid_operands(a, f));
        }
      }
      default: {
        throw UnexpectedError("invalid lhs data type Vh(" + dataTypeName(f->dataType()) + ")");
      }
      }
    } else {
      switch (f->dataType().dimension()) {
      case 1: {
        auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<1>>&>(*f);
        return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
      }
      case 2: {
        auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<2>>&>(*f);
        return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
      }
      case 3: {
        auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyVector<3>>&>(*f);
        return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
      }
      default: {
        throw UnexpectedError("invalid lhs data type Vh(" + dataTypeName(f->dataType()) + ")");
      }
      }
    }
  }
  case ASTNodeDataType::matrix_t: {
    Assert(f->dataType().nbRows() == f->dataType().nbColumns());
    if constexpr (is_tiny_matrix_v<DataType>) {
      switch (f->dataType().nbRows()) {
      case 1: {
        if constexpr (std::is_same_v<DataType, TinyMatrix<1>>) {
          auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<1>>&>(*f);
          return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
        } else {
          throw NormalError(invalid_operands(a, f));
        }
      }
      case 2: {
        if constexpr (std::is_same_v<DataType, TinyMatrix<2>>) {
          auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<2>>&>(*f);
          return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
        } else {
          throw NormalError(invalid_operands(a, f));
        }
      }
      case 3: {
        if constexpr (std::is_same_v<DataType, TinyMatrix<3>>) {
          auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<3>>&>(*f);
          return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
        } else {
          throw NormalError(invalid_operands(a, f));
        }
      }
      default: {
        throw UnexpectedError("invalid lhs data type Vh(" + dataTypeName(f->dataType()) + ")");
      }
      }
    } else {
      switch (f->dataType().nbRows()) {
      case 1: {
        auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<1>>&>(*f);
        return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
      }
      case 2: {
        auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<2>>&>(*f);
        return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
      }
      case 3: {
        auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<3>>&>(*f);
        return applyBinaryOperationWithLeftConstant<BinOperatorT>(a, fh);
      }
      default: {
        throw UnexpectedError("invalid lhs data type Vh(" + dataTypeName(f->dataType()) + ")");
      }
      }
    }
  }
  default: {
    throw NormalError(invalid_operands(a, f));
  }
  }
}

template <typename BinOperatorT, typename DataType>
std::shared_ptr<const IDiscreteFunction>
applyBinaryOperationWithLeftConstant(const DataType& a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  switch (f->mesh()->dimension()) {
  case 1: {
    return applyBinaryOperationWithLeftConstant<BinOperatorT, 1>(a, f);
  }
  case 2: {
    return applyBinaryOperationWithLeftConstant<BinOperatorT, 2>(a, f);
  }
  case 3: {
    return applyBinaryOperationWithLeftConstant<BinOperatorT, 3>(a, f);
  }
  default: {
    throw UnexpectedError("invalid mesh dimension");
  }
  }
}

template <typename BinOperatorT, typename DataType, typename DiscreteFunctionT>
std::shared_ptr<const IDiscreteFunction>
applyBinaryOperationWithRightConstant(const DiscreteFunctionT& f, const DataType& a)
{
  using lhs_data_type = std::decay_t<typename DiscreteFunctionT::data_type>;
  using rhs_data_type = std::decay_t<DataType>;

  if constexpr (std::is_same_v<language::multiply_op, BinOperatorT>) {
    if constexpr (is_tiny_matrix_v<lhs_data_type> and is_tiny_matrix_v<rhs_data_type>) {
      return std::make_shared<decltype(BinOp<BinOperatorT>{}.eval(f, a))>(BinOp<BinOperatorT>{}.eval(f, a));
    } else if constexpr (std::is_same_v<lhs_data_type, double> and
                         (is_tiny_matrix_v<rhs_data_type> or is_tiny_vector_v<rhs_data_type>)) {
      return std::make_shared<decltype(BinOp<BinOperatorT>{}.eval(f, a))>(BinOp<BinOperatorT>{}.eval(f, a));
    } else {
      throw NormalError(invalid_operands(f, a));
    }
  } else {
    throw NormalError(invalid_operands(f, a));
  }
}

template <typename BinOperatorT, size_t Dimension, typename DataType>
std::shared_ptr<const IDiscreteFunction>
applyBinaryOperationWithRightConstant(const std::shared_ptr<const IDiscreteFunction>& f, const DataType& a)
{
  switch (f->dataType()) {
  case ASTNodeDataType::bool_t:
  case ASTNodeDataType::unsigned_int_t:
  case ASTNodeDataType::int_t:
  case ASTNodeDataType::double_t: {
    auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, double>&>(*f);
    return applyBinaryOperationWithRightConstant<BinOperatorT>(fh, a);
  }
  case ASTNodeDataType::matrix_t: {
    Assert(f->dataType().nbRows() == f->dataType().nbColumns());
    if constexpr (is_tiny_matrix_v<DataType>) {
      switch (f->dataType().nbRows()) {
      case 1: {
        if constexpr (std::is_same_v<DataType, TinyMatrix<1>>) {
          auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<1>>&>(*f);
          return applyBinaryOperationWithRightConstant<BinOperatorT>(fh, a);
        } else {
          throw NormalError(invalid_operands(f, a));
        }
      }
      case 2: {
        if constexpr (std::is_same_v<DataType, TinyMatrix<2>>) {
          auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<2>>&>(*f);
          return applyBinaryOperationWithRightConstant<BinOperatorT>(fh, a);
        } else {
          throw NormalError(invalid_operands(f, a));
        }
      }
      case 3: {
        if constexpr (std::is_same_v<DataType, TinyMatrix<3>>) {
          auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<3>>&>(*f);
          return applyBinaryOperationWithRightConstant<BinOperatorT>(fh, a);
        } else {
          throw NormalError(invalid_operands(f, a));
        }
      }
      default: {
        throw UnexpectedError("invalid lhs data type Vh(" + dataTypeName(f->dataType()) + ")");
      }
      }
    } else {
      switch (f->dataType().nbRows()) {
      case 1: {
        auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<1>>&>(*f);
        return applyBinaryOperationWithRightConstant<BinOperatorT>(fh, a);
      }
      case 2: {
        auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<2>>&>(*f);
        return applyBinaryOperationWithRightConstant<BinOperatorT>(fh, a);
      }
      case 3: {
        auto fh = dynamic_cast<const DiscreteFunctionP0<Dimension, TinyMatrix<3>>&>(*f);
        return applyBinaryOperationWithRightConstant<BinOperatorT>(fh, a);
      }
      default: {
        throw UnexpectedError("invalid lhs data type Vh(" + dataTypeName(f->dataType()) + ")");
      }
      }
    }
  }
  default: {
    throw NormalError(invalid_operands(f, a));
  }
  }
}

template <typename BinOperatorT, typename DataType>
std::shared_ptr<const IDiscreteFunction>
applyBinaryOperationWithRightConstant(const std::shared_ptr<const IDiscreteFunction>& f, const DataType& a)
{
  switch (f->mesh()->dimension()) {
  case 1: {
    return applyBinaryOperationWithRightConstant<BinOperatorT, 1>(f, a);
  }
  case 2: {
    return applyBinaryOperationWithRightConstant<BinOperatorT, 2>(f, a);
  }
  case 3: {
    return applyBinaryOperationWithRightConstant<BinOperatorT, 3>(f, a);
  }
  default: {
    throw UnexpectedError("invalid mesh dimension");
  }
  }
}

std::shared_ptr<const IDiscreteFunction>
operator*(const double& a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  return applyBinaryOperationWithLeftConstant<language::multiply_op>(a, f);
}

std::shared_ptr<const IDiscreteFunction>
operator*(const TinyMatrix<1>& A, const std::shared_ptr<const IDiscreteFunction>& B)
{
  return applyBinaryOperationWithLeftConstant<language::multiply_op>(A, B);
}

std::shared_ptr<const IDiscreteFunction>
operator*(const TinyMatrix<2>& A, const std::shared_ptr<const IDiscreteFunction>& B)
{
  return applyBinaryOperationWithLeftConstant<language::multiply_op>(A, B);
}

std::shared_ptr<const IDiscreteFunction>
operator*(const TinyMatrix<3>& A, const std::shared_ptr<const IDiscreteFunction>& B)
{
  return applyBinaryOperationWithLeftConstant<language::multiply_op>(A, B);
}

std::shared_ptr<const IDiscreteFunction>
operator*(const std::shared_ptr<const IDiscreteFunction>& a, const TinyVector<1>& u)
{
  return applyBinaryOperationWithRightConstant<language::multiply_op>(a, u);
}

std::shared_ptr<const IDiscreteFunction>
operator*(const std::shared_ptr<const IDiscreteFunction>& a, const TinyVector<2>& u)
{
  return applyBinaryOperationWithRightConstant<language::multiply_op>(a, u);
}

std::shared_ptr<const IDiscreteFunction>
operator*(const std::shared_ptr<const IDiscreteFunction>& a, const TinyVector<3>& u)
{
  return applyBinaryOperationWithRightConstant<language::multiply_op>(a, u);
}

std::shared_ptr<const IDiscreteFunction>
operator*(const std::shared_ptr<const IDiscreteFunction>& a, const TinyMatrix<1>& A)
{
  return applyBinaryOperationWithRightConstant<language::multiply_op>(a, A);
}

std::shared_ptr<const IDiscreteFunction>
operator*(const std::shared_ptr<const IDiscreteFunction>& a, const TinyMatrix<2>& A)
{
  return applyBinaryOperationWithRightConstant<language::multiply_op>(a, A);
}

std::shared_ptr<const IDiscreteFunction>
operator*(const std::shared_ptr<const IDiscreteFunction>& a, const TinyMatrix<3>& A)
{
  return applyBinaryOperationWithRightConstant<language::multiply_op>(a, A);
}
