#ifndef MESH_HPP
#define MESH_HPP

#include <Kokkos_Core.hpp>
#include <TinyVector.hpp>

#include <memory>

struct IMesh
{
  virtual const size_t meshDimension() const = 0;
  ~IMesh() = default;
};

template <typename ConnectivityType>
class Mesh final : public IMesh
{
public:
  typedef ConnectivityType Connectivity;
  static constexpr size_t dimension = ConnectivityType::dimension;
  typedef TinyVector<dimension> Rd;

private:
  const std::shared_ptr<Connectivity> m_connectivity;
  Kokkos::View<Rd*> m_xr;

public:

  const size_t meshDimension() const
  {
    return dimension;
  }

  const Connectivity& connectivity() const
  {
    return *m_connectivity;
  }

  const size_t& numberOfNodes() const
  {
    return m_connectivity->numberOfNodes();
  }

  const size_t& numberOfFaces() const
  {
    return m_connectivity->numberOfFaces();
  }

  const size_t& numberOfCells() const
  {
    return m_connectivity->numberOfCells();
  }

  Kokkos::View<Rd*> xr() const
  {
    return m_xr;
  }

  KOKKOS_INLINE_FUNCTION
  Mesh(const std::shared_ptr<Connectivity>& connectivity)
    : m_connectivity(connectivity),
      m_xr("xr", connectivity->numberOfNodes())
  {
    static_assert (Connectivity::dimension ==1,"no automatic calculation of vertices in dim>1");
    const double delta_x = 1./connectivity->numberOfCells();
    Kokkos::parallel_for(connectivity->numberOfNodes(), KOKKOS_LAMBDA(const int& r){
	m_xr[r][0] = r*delta_x;
      });
  }

  KOKKOS_INLINE_FUNCTION
  Mesh(const std::shared_ptr<Connectivity>& connectivity,
       Kokkos::View<Rd*>& xr)
    : m_connectivity(connectivity),
      m_xr(xr)
  {
    ;
  }

  Mesh() = delete;

  ~Mesh()
  {
    ;
  }
};

#endif // MESH_HPP
