#ifndef SUB_ARRAY_HPP
#define SUB_ARRAY_HPP

#include <utils/Array.hpp>
#include <utils/PugsAssert.hpp>
#include <utils/PugsMacros.hpp>
#include <utils/PugsUtils.hpp>

#include <algorithm>

template <typename DataType>
class [[nodiscard]] SubArray
{
 public:
  using data_type  = DataType;
  using index_type = size_t;

 private:
  PUGS_RESTRICT DataType* const m_sub_values;
  const size_t m_size;

  // Allows const version to access our data
  friend SubArray<std::add_const_t<DataType>>;

 public:
  PUGS_INLINE size_t size() const noexcept
  {
    return m_size;
  }

  PUGS_INLINE DataType& operator[](index_type i) const noexcept(NO_ASSERT)
  {
    Assert(i < m_size);
    return m_sub_values[i];
  }

  PUGS_INLINE
  void fill(const DataType& data) const
  {
    static_assert(not std::is_const<DataType>(), "Cannot modify SubArray of const");

    // could consider to use std::fill
    parallel_for(
      this->size(), PUGS_LAMBDA(index_type i) { m_sub_values[i] = data; });
  }

  PUGS_INLINE
  SubArray& operator=(const SubArray&) = delete;

  PUGS_INLINE
  SubArray& operator=(SubArray&&) = delete;

  PUGS_INLINE
  explicit SubArray(const Array<DataType>& array, size_t begin, size_t size)
    : m_sub_values{&array[0] + begin}, m_size{size}
  {
    Assert(begin + size <= array.size(), "SubView is not contained in the source Array");
  }

  PUGS_INLINE
  explicit SubArray(DataType* const raw_array, size_t begin, size_t size)
    : m_sub_values{raw_array + begin}, m_size{size}
  {}

  PUGS_INLINE
  SubArray() = delete;

  PUGS_INLINE
  SubArray(const SubArray&) = delete;

  PUGS_INLINE
  SubArray(SubArray &&) = delete;

  PUGS_INLINE
  ~SubArray() = default;
};

#endif   // SUB_ARRAY_HPP
