#include <language/modules/UtilsModule.hpp>

#include <language/utils/ASTDotPrinter.hpp>
#include <language/utils/ASTPrinter.hpp>
#include <language/utils/BuiltinFunctionEmbedder.hpp>
#include <language/utils/SymbolTable.hpp>
#include <utils/PugsUtils.hpp>

extern const std::unique_ptr<ASTNode>* p_root_node;

UtilsModule::UtilsModule()
{
  this->_addBuiltinFunction("getPugsVersion", std::make_shared<BuiltinFunctionEmbedder<std::string(void)>>(

                                                []() -> std::string { return pugsVersion(); }

                                                ));

  this->_addBuiltinFunction("getPugsBuildInfo", std::make_shared<BuiltinFunctionEmbedder<std::string(void)>>(

                                                  []() -> std::string { return pugsBuildInfo(); }

                                                  ));

  this->_addBuiltinFunction("getAST", std::make_shared<BuiltinFunctionEmbedder<std::string(void)>>(

                                        []() -> std::string {
                                          Assert(p_root_node != nullptr, "unable to find AST's root node");

                                          const auto& root_node = *p_root_node;
                                          std::ostringstream os;
                                          os << ASTPrinter{*root_node};

                                          return os.str();
                                        }

                                        ));

  this->_addBuiltinFunction("saveASTDot", std::make_shared<BuiltinFunctionEmbedder<void(const std::string&)>>(

                                            [](const std::string& dot_filename) -> void {
                                              Assert(p_root_node != nullptr, "unable to find AST's root node");

                                              const auto& root_node = *p_root_node;

                                              std::ofstream fout(dot_filename);

                                              if (not fout) {
                                                std::ostringstream os;
                                                os << "could not create file '" << dot_filename << "'\n";
                                                throw NormalError(os.str());
                                              }

                                              ASTDotPrinter dot_printer{*root_node};
                                              fout << dot_printer;

                                              if (not fout) {
                                                std::ostringstream os;
                                                os << "could not write AST to '" << dot_filename << "'\n";
                                                throw NormalError(os.str());
                                              }
                                            }

                                            ));

  this->_addBuiltinFunction("getFunctionAST",
                            std::make_shared<BuiltinFunctionEmbedder<std::string(const FunctionSymbolId&)>>(

                              [](const FunctionSymbolId& function_symbol_id) -> std::string {
                                auto& function_table = function_symbol_id.symbolTable().functionTable();

                                const auto& function_descriptor = function_table[function_symbol_id.id()];

                                std::ostringstream os;
                                os << function_descriptor.name() << ": domain mapping\n";
                                os << ASTPrinter(function_descriptor.domainMappingNode());
                                os << function_descriptor.name() << ": definition\n";
                                os << ASTPrinter(function_descriptor.definitionNode());

                                return os.str();
                              }

                              ));
}
