#include <language/modules/ModuleRepository.hpp>

#include <language/ast/ASTNode.hpp>
#include <language/modules/LinearSolverModule.hpp>
#include <language/modules/MathModule.hpp>
#include <language/modules/MeshModule.hpp>
#include <language/modules/SchemeModule.hpp>
#include <language/modules/UtilsModule.hpp>
#include <language/modules/VTKModule.hpp>
#include <language/utils/BuiltinFunctionEmbedder.hpp>
#include <language/utils/ParseError.hpp>
#include <language/utils/SymbolTable.hpp>
#include <utils/PugsAssert.hpp>

void
ModuleRepository::_subscribe(std::unique_ptr<IModule> m)
{
  auto [i_module, success] = m_module_set.emplace(m->name(), std::move(m));
  Assert(success, "module has already been subscribed");
}

ModuleRepository::ModuleRepository()
{
  this->_subscribe(std::make_unique<LinearSolverModule>());
  this->_subscribe(std::make_unique<MathModule>());
  this->_subscribe(std::make_unique<MeshModule>());
  this->_subscribe(std::make_unique<SchemeModule>());
  this->_subscribe(std::make_unique<UtilsModule>());
  this->_subscribe(std::make_unique<VTKModule>());
}

template <typename NameEmbedderMapT, typename EmbedderTableT>
void
ModuleRepository::_populateEmbedderTableT(const ASTNode& module_name_node,
                                          const NameEmbedderMapT& name_embedder_map,
                                          const ASTNodeDataType& data_type,
                                          SymbolTable& symbol_table,
                                          EmbedderTableT& embedder_table)
{
  const std::string& module_name = module_name_node.string();

  for (auto [symbol_name, embedded] : name_embedder_map) {
    auto [i_symbol, success] = symbol_table.add(symbol_name, module_name_node.begin());

    if (not success) {
      std::ostringstream error_message;
      error_message << "importing module '" << module_name << "', cannot add symbol '" << symbol_name
                    << "', it is already defined!";
      throw ParseError(error_message.str(), module_name_node.begin());
    }

    i_symbol->attributes().setDataType(data_type);
    i_symbol->attributes().setIsInitialized();
    i_symbol->attributes().value() = embedder_table.size();

    embedder_table.add(embedded);
  }
}

void
ModuleRepository::populateSymbolTable(const ASTNode& module_name_node, SymbolTable& symbol_table)
{
  const std::string& module_name = module_name_node.string();

  auto i_module = m_module_set.find(module_name);
  if (i_module != m_module_set.end()) {
    const IModule& populating_module = *i_module->second;

    this->_populateEmbedderTableT(module_name_node, populating_module.getNameBuiltinFunctionMap(),
                                  ASTNodeDataType::build<ASTNodeDataType::builtin_function_t>(), symbol_table,
                                  symbol_table.builtinFunctionEmbedderTable());

    this->_populateEmbedderTableT(module_name_node, populating_module.getNameTypeMap(),
                                  ASTNodeDataType::build<ASTNodeDataType::type_name_id_t>(), symbol_table,
                                  symbol_table.typeEmbedderTable());
  } else {
    throw ParseError(std::string{"could not find module "} + module_name, std::vector{module_name_node.begin()});
  }
}
