#ifndef MESSENGER_HPP
#define MESSENGER_HPP

#include <PastisMacros.hpp>
#include <PastisAssert.hpp>

#include <Array.hpp>

#warning REMOVE
enum class CellType : unsigned short;

class Messenger
{
 private:
  static Messenger* m_instance;
  Messenger(int& argc, char* argv[]);

  int m_rank{0};
  int m_size{1};

  Array<int> _allGather(int& data) const;

  int _broadcast(int& data, int root_rank) const;
  Array<int> _broadcast(Array<int>& array, int root_rank) const;
  Array<int> _allToAll(const Array<int>& sent_array, Array<int>& recv_array) const;

  template <typename DataType>
  void _exchange(const std::vector<Array<DataType>>& sent_array_list,
                 std::vector<Array<std::remove_const_t<DataType>>>& recv_array_list) const;

 public:
  static void create(int& argc, char* argv[]);
  static void destroy();

  PASTIS_INLINE
  static Messenger& getInstance()
  {
    Assert(m_instance != nullptr);
    return *m_instance;
  }

  PASTIS_INLINE
  const int& rank() const
  {
    return m_rank;
  }

  PASTIS_INLINE
  const int& size() const
  {
    return m_size;
  }

  void barrier() const;

  template <typename DataType>
  PASTIS_INLINE
  Array<DataType> allGather(const DataType& data) const
  {
    if constexpr(std::is_same<DataType, int>()) {
      int int_data = data;
      return _allGather(int_data);
    } else {
      static_assert(std::is_same<DataType, int>(), "unexpected type of data");
    }
  }

  template <typename DataType>
  PASTIS_INLINE
  Array<DataType> allToAll(const Array<DataType>& sent_array) const
  {
    Assert(sent_array.size() == static_cast<size_t>(m_size));
    if constexpr(std::is_same<DataType, int>()) {
      Array<int> recv_array(m_size);
      return _allToAll(sent_array, recv_array);
    } else {
      static_assert(std::is_same<DataType, int>(), "unexpected type of data");
    }
  }

  template <typename DataType>
  PASTIS_INLINE
  DataType broadcast(const DataType& data, int root_rank) const
  {
    if constexpr(std::is_same<DataType, int>()) {
      int int_data = data;
      return _broadcast(int_data, root_rank);
    } else {
      static_assert(std::is_same<DataType, int>(), "unexpected type of data");
    }
  }

  template <typename DataType>
  PASTIS_INLINE
  Array<DataType> broadcast(const Array<DataType>& array, int root_rank) const
  {
    if constexpr(std::is_same<DataType, int>()) {
      Array<int> int_array = array;
      return _broadcast(int_array, root_rank);
    } else {
      static_assert(std::is_same<DataType, int>(), "unexpected type of data");
    }
  }

  template <typename SendDataType,
            typename RecvDataType>
  PASTIS_INLINE
  void exchange(const std::vector<Array<SendDataType>>& sent_array_list,
                std::vector<Array<RecvDataType>>& recv_array_list) const
  {
    static_assert(std::is_same_v<std::remove_const_t<SendDataType>,RecvDataType>,
                  "send and receive data type do not match");
    static_assert(not std::is_const_v<RecvDataType>,
                  "receive data type cannot be const");

    if constexpr(std::is_same<RecvDataType, int>()) {
      _exchange(sent_array_list, recv_array_list);
    } else if constexpr(std::is_same<RecvDataType, CellType>()) {
      _exchange(sent_array_list, recv_array_list);
    } else {
      static_assert(std::is_same<RecvDataType, int>(), "unexpected type of data");
    }
  }

  Messenger(const Messenger&) = delete;
  ~Messenger();
};

PASTIS_INLINE
const Messenger& messenger()
{
  return Messenger::getInstance();
}

[[deprecated("use better name")]]
PASTIS_INLINE
const int& commRank()
{
  return messenger().rank();
}

[[deprecated("use better name")]]
PASTIS_INLINE
const int& commSize()
{
  return messenger().size();
}

PASTIS_INLINE
void barrier()
{
  return messenger().barrier();
}

template <typename DataType>
PASTIS_INLINE
DataType broadcast(const DataType& data, int root_rank)
{
  return messenger().broadcast(data, root_rank);
}

template <typename DataType>
PASTIS_INLINE
Array<DataType> allGather(const DataType& data)
{
  return messenger().allGather(data);
}

template <typename DataType>
PASTIS_INLINE
Array<DataType> allToAll(const Array<DataType>& array)
{
  return messenger().allToAll(array);
}

template <typename DataType>
PASTIS_INLINE
Array<DataType> broadcast(const Array<DataType>& array, int root_rank)
{
  return messenger().broadcast(array, root_rank);
}

template <typename SendDataType,
          typename RecvDataType>
PASTIS_INLINE
void exchange(const std::vector<Array<SendDataType>>& sent_array_list,
              std::vector<Array<RecvDataType>>& recv_array_list)
{
  static_assert(std::is_same_v<std::remove_const_t<SendDataType>,RecvDataType>,
                "send and receive data type do not match");
  static_assert(not std::is_const_v<RecvDataType>,
                "receive data type cannot be const");

  messenger().exchange(sent_array_list, recv_array_list);
}

#endif // MESSENGER_HPP
