#ifndef MESH_HPP
#define MESH_HPP

#include <ItemValue.hpp>
#include <TinyVector.hpp>

#include <CSRGraph.hpp>

#include <memory>

struct IMesh
{
  virtual const size_t meshDimension() const = 0;
  virtual CSRGraph cellToCellGraph() const = 0;
  ~IMesh() = default;
};

template <typename ConnectivityType>
class Mesh final : public IMesh
{
public:
  using Connectivity = ConnectivityType;

  static constexpr size_t dimension = ConnectivityType::dimension;
  using Rd = TinyVector<dimension>;

private:
  const std::shared_ptr<Connectivity> m_connectivity;
  NodeValue<const Rd> m_xr;
  NodeValue<Rd> m_mutable_xr;

public:
  PASTIS_INLINE
  CSRGraph cellToCellGraph() const final
  {
    return m_connectivity->cellToCellGraph();
  }

  PASTIS_INLINE
  const size_t meshDimension() const
  {
    return dimension;
  }

  PASTIS_INLINE
  const Connectivity& connectivity() const
  {
    return *m_connectivity;
  }

  PASTIS_INLINE
  size_t numberOfNodes() const
  {
    return m_connectivity->numberOfNodes();
  }

  PASTIS_INLINE
  size_t numberOfFaces() const
  {
    return m_connectivity->numberOfFaces();
  }

  PASTIS_INLINE
  size_t numberOfCells() const
  {
    return m_connectivity->numberOfCells();
  }

  PASTIS_INLINE
  const NodeValue<const Rd>& xr() const
  {
    return m_xr;
  }

  [[deprecated("should rework this class: quite ugly")]]
  PASTIS_INLINE
  NodeValue<Rd> mutableXr() const
  {
    return m_mutable_xr;
  }

  PASTIS_INLINE
  Mesh(const std::shared_ptr<Connectivity>& connectivity,
       NodeValue<Rd>& xr)
      : m_connectivity{connectivity},
        m_xr{xr},
        m_mutable_xr{xr}
  {
    ;
  }

  Mesh() = delete;

  ~Mesh()
  {
    ;
  }
};

#endif // MESH_HPP
