#ifndef DATA_VARIANT_HPP
#define DATA_VARIANT_HPP

#include <algebra/TinyVector.hpp>
#include <language/utils/EmbeddedData.hpp>
#include <language/utils/FunctionSymbolId.hpp>
#include <utils/PugsAssert.hpp>

#include <tuple>
#include <variant>
#include <vector>

class AggregateDataVariant;

using DataVariant = std::variant<std::monostate,
                                 bool,
                                 uint64_t,
                                 int64_t,
                                 double,
                                 std::string,
                                 EmbeddedData,
                                 AggregateDataVariant,
                                 FunctionSymbolId,
                                 TinyVector<1>,
                                 TinyVector<2>,
                                 TinyVector<3>>;

class AggregateDataVariant   // LCOV_EXCL_LINE
{
 private:
  std::vector<DataVariant> m_data_vector;

  std::ostream&
  _printComponent(std::ostream& os, const DataVariant& value) const
  {
    std::visit(
      [&](auto&& v) {
        if constexpr (std::is_same_v<std::decay_t<decltype(v)>, std::monostate>) {
          os << " -- ";
        } else {
          os << v;
        }
      },
      value);
    return os;
  }

  std::ostream&
  _print(std::ostream& os) const
  {
    Assert(m_data_vector.size() > 0, "unexpected compound data size");
    os << '(';
    this->_printComponent(os, m_data_vector[0]);
    for (size_t i = 1; i < m_data_vector.size(); ++i) {
      os << ", ";
      this->_printComponent(os, m_data_vector[i]);
    }
    os << ')';
    return os;
  }

 public:
  friend std::ostream&
  operator<<(std::ostream& os, const AggregateDataVariant& compound)
  {
    return compound._print(os);
  }

  PUGS_INLINE
  size_t
  size() const
  {
    return m_data_vector.size();
  }

  PUGS_INLINE
  DataVariant& operator[](size_t i)
  {
    Assert(i < m_data_vector.size());
    return m_data_vector[i];
  }

  PUGS_INLINE
  const DataVariant& operator[](size_t i) const
  {
    Assert(i < m_data_vector.size());
    return m_data_vector[i];
  }

  AggregateDataVariant& operator=(const AggregateDataVariant&) = default;
  AggregateDataVariant& operator=(AggregateDataVariant&&) = default;

  AggregateDataVariant(std::vector<DataVariant>&& data_vector) : m_data_vector(data_vector) {}

  AggregateDataVariant(const AggregateDataVariant&) = default;
  AggregateDataVariant(AggregateDataVariant&&)      = default;

  AggregateDataVariant() = default;
};

#endif   // DATA_VARIANT_HPP
