#include <language/utils/EmbeddedIDiscreteFunctionMathFunctions.hpp>

#include <language/utils/EmbeddedIDiscreteFunctionUtils.hpp>
#include <mesh/IMesh.hpp>
#include <scheme/DiscreteFunctionP0.hpp>
#include <scheme/DiscreteFunctionUtils.hpp>
#include <scheme/IDiscreteFunction.hpp>
#include <scheme/IDiscreteFunctionDescriptor.hpp>

#define DISCRETE_FUNCTION_CALL(FUNCTION, ARG)                                                                         \
  if (ARG->dataType() == ASTNodeDataType::double_t and ARG->descriptor().type() == DiscreteFunctionType::P0) {        \
    switch (ARG->mesh()->dimension()) {                                                                               \
    case 1: {                                                                                                         \
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;                                                     \
      return std::make_shared<const DiscreteFunctionType>(FUNCTION(dynamic_cast<const DiscreteFunctionType&>(*ARG))); \
    }                                                                                                                 \
    case 2: {                                                                                                         \
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;                                                     \
      return std::make_shared<const DiscreteFunctionType>(FUNCTION(dynamic_cast<const DiscreteFunctionType&>(*ARG))); \
    }                                                                                                                 \
    case 3: {                                                                                                         \
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;                                                     \
      return std::make_shared<const DiscreteFunctionType>(FUNCTION(dynamic_cast<const DiscreteFunctionType&>(*ARG))); \
    }                                                                                                                 \
    default: {                                                                                                        \
      throw UnexpectedError("invalid mesh dimension");                                                                \
    }                                                                                                                 \
    }                                                                                                                 \
  } else {                                                                                                            \
    throw NormalError("invalid operand type " + operand_type_name(ARG));                                              \
  }

std::shared_ptr<const IDiscreteFunction>
sqrt(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(sqrt, f);
}

std::shared_ptr<const IDiscreteFunction>
abs(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(abs, f);
}

std::shared_ptr<const IDiscreteFunction>
sin(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(sin, f);
}

std::shared_ptr<const IDiscreteFunction>
cos(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(cos, f);
}

std::shared_ptr<const IDiscreteFunction>
tan(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(tan, f);
}

std::shared_ptr<const IDiscreteFunction>
asin(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(asin, f);
}

std::shared_ptr<const IDiscreteFunction>
acos(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(acos, f);
}

std::shared_ptr<const IDiscreteFunction>
atan(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(atan, f);
}

std::shared_ptr<const IDiscreteFunction>
atan2(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  if ((f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (g->dataType() == ASTNodeDataType::double_t and g->descriptor().type() == DiscreteFunctionType::P0)) {
    std::shared_ptr mesh = getCommonMesh({f, g});

    if (mesh.use_count() == 0) {
      throw NormalError("operands are defined on different meshes");
    }

    switch (mesh->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(
        atan2(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(
        atan2(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(
        atan2(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(f) << " and " << operand_type_name(g);
    throw NormalError(os.str());
  }
}

std::shared_ptr<const IDiscreteFunction>
atan2(const double a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(atan2(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(atan2(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(atan2(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(a) << " and " << operand_type_name(f);
    throw NormalError(os.str());
  }
}

std::shared_ptr<const IDiscreteFunction>
atan2(const std::shared_ptr<const IDiscreteFunction>& f, const double a)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(atan2(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(atan2(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(atan2(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(f) << " and " << operand_type_name(a);
    throw NormalError(os.str());
  }
}

std::shared_ptr<const IDiscreteFunction>
sinh(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(sinh, f);
}

std::shared_ptr<const IDiscreteFunction>
cosh(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(cosh, f);
}

std::shared_ptr<const IDiscreteFunction>
tanh(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(tanh, f);
}

std::shared_ptr<const IDiscreteFunction>
asinh(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(asinh, f);
}

std::shared_ptr<const IDiscreteFunction>
acosh(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(acosh, f);
}

std::shared_ptr<const IDiscreteFunction>
atanh(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(atanh, f);
}

std::shared_ptr<const IDiscreteFunction>
exp(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(exp, f);
}

std::shared_ptr<const IDiscreteFunction>
log(const std::shared_ptr<const IDiscreteFunction>& f)
{
  DISCRETE_FUNCTION_CALL(log, f);
}

std::shared_ptr<const IDiscreteFunction>
pow(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  if ((f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (g->dataType() == ASTNodeDataType::double_t and g->descriptor().type() == DiscreteFunctionType::P0)) {
    std::shared_ptr mesh = getCommonMesh({f, g});

    if (mesh.use_count() == 0) {
      throw NormalError("operands are defined on different meshes");
    }

    switch (mesh->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(
        pow(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(
        pow(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(
        pow(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(f) << " and " << operand_type_name(g);
    throw NormalError(os.str());
  }
}

std::shared_ptr<const IDiscreteFunction>
pow(const double a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(pow(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(pow(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(pow(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(a) << " and " << operand_type_name(f);
    throw NormalError(os.str());
  }
}

std::shared_ptr<const IDiscreteFunction>
pow(const std::shared_ptr<const IDiscreteFunction>& f, const double a)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(pow(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(pow(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(pow(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(f) << " and " << operand_type_name(a);
    throw NormalError(os.str());
  }
}

template <size_t Dimension>
std::shared_ptr<const IDiscreteFunction>
dot(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  Assert((f->dataType() == ASTNodeDataType::vector_t and f->descriptor().type() == DiscreteFunctionType::P0) and
         (g->dataType() == ASTNodeDataType::vector_t and g->descriptor().type() == DiscreteFunctionType::P0) and
         (f->dataType().dimension() == g->dataType().dimension()));

  using DiscreteFunctionResultType = DiscreteFunctionP0<Dimension, double>;

  switch (f->dataType().dimension()) {
  case 1: {
    using Rd                   = TinyVector<1>;
    using DiscreteFunctionType = DiscreteFunctionP0<Dimension, Rd>;
    return std::make_shared<const DiscreteFunctionResultType>(
      dot(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
  }
  case 2: {
    using Rd                   = TinyVector<2>;
    using DiscreteFunctionType = DiscreteFunctionP0<Dimension, Rd>;
    return std::make_shared<const DiscreteFunctionResultType>(
      dot(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
  }
  case 3: {
    using Rd                   = TinyVector<3>;
    using DiscreteFunctionType = DiscreteFunctionP0<Dimension, Rd>;
    return std::make_shared<const DiscreteFunctionResultType>(
      dot(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
  }
  default: {
    throw UnexpectedError("invalid data dimension " + operand_type_name(f));
  }
  }
}

std::shared_ptr<const IDiscreteFunction>
dot(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  if ((f->dataType() == ASTNodeDataType::vector_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (g->dataType() == ASTNodeDataType::vector_t and g->descriptor().type() == DiscreteFunctionType::P0) and
      (f->dataType().dimension() == g->dataType().dimension())) {
    std::shared_ptr mesh = getCommonMesh({f, g});

    if (mesh.use_count() == 0) {
      throw NormalError("operands are defined on different meshes");
    }

    switch (mesh->dimension()) {
    case 1: {
      return dot<1>(f, g);
    }
    case 2: {
      return dot<2>(f, g);
    }
    case 3: {
      return dot<3>(f, g);
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(f) << " and " << operand_type_name(g);
    throw NormalError(os.str());
  }
}

template <size_t Dimension, size_t VectorDimension>
std::shared_ptr<const IDiscreteFunction>
dot(const std::shared_ptr<const IDiscreteFunction>& f, const TinyVector<VectorDimension>& a)
{
  Assert((f->dataType() == ASTNodeDataType::vector_t and f->descriptor().type() == DiscreteFunctionType::P0) and
         (f->dataType().dimension() == a.dimension()));

  using DiscreteFunctionResultType = DiscreteFunctionP0<Dimension, double>;
  using DiscreteFunctionType       = DiscreteFunctionP0<Dimension, TinyVector<VectorDimension>>;

  return std::make_shared<const DiscreteFunctionResultType>(dot(dynamic_cast<const DiscreteFunctionType&>(*f), a));
}

template <size_t VectorDimension>
std::shared_ptr<const IDiscreteFunction>
dot(const std::shared_ptr<const IDiscreteFunction>& f, const TinyVector<VectorDimension>& a)
{
  if ((f->dataType() == ASTNodeDataType::vector_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (f->dataType().dimension() == a.dimension())) {
    switch (f->mesh()->dimension()) {
    case 1: {
      return dot<1, VectorDimension>(f, a);
    }
    case 2: {
      return dot<2, VectorDimension>(f, a);
    }
    case 3: {
      return dot<3, VectorDimension>(f, a);
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(f) << " and " << operand_type_name(a);
    throw NormalError(os.str());
  }
}

template <size_t Dimension, size_t VectorDimension>
std::shared_ptr<const IDiscreteFunction>
dot(const TinyVector<VectorDimension>& a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  Assert((f->dataType() == ASTNodeDataType::vector_t and f->descriptor().type() == DiscreteFunctionType::P0) and
         (f->dataType().dimension() == a.dimension()));

  using DiscreteFunctionResultType = DiscreteFunctionP0<Dimension, double>;
  using DiscreteFunctionType       = DiscreteFunctionP0<Dimension, TinyVector<VectorDimension>>;

  return std::make_shared<const DiscreteFunctionResultType>(dot(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
}

template <size_t VectorDimension>
std::shared_ptr<const IDiscreteFunction>
dot(const TinyVector<VectorDimension>& a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  if ((f->dataType() == ASTNodeDataType::vector_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (f->dataType().dimension() == a.dimension())) {
    switch (f->mesh()->dimension()) {
    case 1: {
      return dot<1, VectorDimension>(a, f);
    }
    case 2: {
      return dot<2, VectorDimension>(a, f);
    }
    case 3: {
      return dot<3, VectorDimension>(a, f);
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(a) << " and " << operand_type_name(f);
    throw NormalError(os.str());
  }
}

template std::shared_ptr<const IDiscreteFunction> dot(const std::shared_ptr<const IDiscreteFunction>&,
                                                      const TinyVector<1>&);

template std::shared_ptr<const IDiscreteFunction> dot(const std::shared_ptr<const IDiscreteFunction>&,
                                                      const TinyVector<2>&);

template std::shared_ptr<const IDiscreteFunction> dot(const std::shared_ptr<const IDiscreteFunction>&,
                                                      const TinyVector<3>&);

template std::shared_ptr<const IDiscreteFunction> dot(const TinyVector<1>&,
                                                      const std::shared_ptr<const IDiscreteFunction>&);

template std::shared_ptr<const IDiscreteFunction> dot(const TinyVector<2>&,
                                                      const std::shared_ptr<const IDiscreteFunction>&);

template std::shared_ptr<const IDiscreteFunction> dot(const TinyVector<3>&,
                                                      const std::shared_ptr<const IDiscreteFunction>&);

double
min(const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return min(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return min(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return min(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    throw NormalError("invalid operand type " + operand_type_name(f));
  }
}

std::shared_ptr<const IDiscreteFunction>
min(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  if ((f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (g->dataType() == ASTNodeDataType::double_t and g->descriptor().type() == DiscreteFunctionType::P0)) {
    std::shared_ptr mesh = getCommonMesh({f, g});

    if (mesh.use_count() == 0) {
      throw NormalError("operands are defined on different meshes");
    }

    switch (mesh->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(
        min(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(
        min(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(
        min(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(f) << " and " << operand_type_name(g);
    throw NormalError(os.str());
  }
}

std::shared_ptr<const IDiscreteFunction>
min(const double a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(min(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(min(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(min(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(a) << " and " << operand_type_name(f);
    throw NormalError(os.str());
  }
}

std::shared_ptr<const IDiscreteFunction>
min(const std::shared_ptr<const IDiscreteFunction>& f, const double a)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(min(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(min(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(min(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(f) << " and " << operand_type_name(a);
    throw NormalError(os.str());
  }
}

double
max(const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return max(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return max(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return max(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    throw NormalError("invalid operand type " + operand_type_name(f));
  }
}

std::shared_ptr<const IDiscreteFunction>
max(const std::shared_ptr<const IDiscreteFunction>& f, const std::shared_ptr<const IDiscreteFunction>& g)
{
  if ((f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) and
      (g->dataType() == ASTNodeDataType::double_t and g->descriptor().type() == DiscreteFunctionType::P0)) {
    std::shared_ptr mesh = getCommonMesh({f, g});

    if (mesh.use_count() == 0) {
      throw NormalError("operands are defined on different meshes");
    }

    switch (mesh->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(
        max(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(
        max(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(
        max(dynamic_cast<const DiscreteFunctionType&>(*f), dynamic_cast<const DiscreteFunctionType&>(*g)));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(f) << " and " << operand_type_name(g);
    throw NormalError(os.str());
  }
}

std::shared_ptr<const IDiscreteFunction>
max(const double a, const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(max(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(max(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(max(a, dynamic_cast<const DiscreteFunctionType&>(*f)));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(a) << " and " << operand_type_name(f);
    throw NormalError(os.str());
  }
}

std::shared_ptr<const IDiscreteFunction>
max(const std::shared_ptr<const IDiscreteFunction>& f, const double a)
{
  if (f->dataType() == ASTNodeDataType::double_t and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, double>;
      return std::make_shared<const DiscreteFunctionType>(max(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, double>;
      return std::make_shared<const DiscreteFunctionType>(max(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, double>;
      return std::make_shared<const DiscreteFunctionType>(max(dynamic_cast<const DiscreteFunctionType&>(*f), a));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    std::stringstream os;
    os << "incompatible operand types " << operand_type_name(f) << " and " << operand_type_name(a);
    throw NormalError(os.str());
  }
}

template <typename ValueT>
ValueT
sum_of(const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ast_node_data_type_from<ValueT> and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, ValueT>;
      return sum(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, ValueT>;
      return sum(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, ValueT>;
      return sum(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    throw NormalError("invalid operand type " + operand_type_name(f));
  }
}

template double sum_of<double>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyVector<1> sum_of<TinyVector<1>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyVector<2> sum_of<TinyVector<2>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyVector<3> sum_of<TinyVector<3>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyMatrix<1> sum_of<TinyMatrix<1>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyMatrix<2> sum_of<TinyMatrix<2>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyMatrix<3> sum_of<TinyMatrix<3>>(const std::shared_ptr<const IDiscreteFunction>&);

template <typename ValueT>
ValueT
integral_of(const std::shared_ptr<const IDiscreteFunction>& f)
{
  if (f->dataType() == ast_node_data_type_from<ValueT> and f->descriptor().type() == DiscreteFunctionType::P0) {
    switch (f->mesh()->dimension()) {
    case 1: {
      using DiscreteFunctionType = DiscreteFunctionP0<1, ValueT>;
      return integrate(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 2: {
      using DiscreteFunctionType = DiscreteFunctionP0<2, ValueT>;
      return integrate(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    case 3: {
      using DiscreteFunctionType = DiscreteFunctionP0<3, ValueT>;
      return integrate(dynamic_cast<const DiscreteFunctionType&>(*f));
    }
    default: {
      throw UnexpectedError("invalid mesh dimension");
    }
    }
  } else {
    throw NormalError("invalid operand type " + operand_type_name(f));
  }
}

template double integral_of<double>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyVector<1> integral_of<TinyVector<1>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyVector<2> integral_of<TinyVector<2>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyVector<3> integral_of<TinyVector<3>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyMatrix<1> integral_of<TinyMatrix<1>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyMatrix<2> integral_of<TinyMatrix<2>>(const std::shared_ptr<const IDiscreteFunction>&);

template TinyMatrix<3> integral_of<TinyMatrix<3>>(const std::shared_ptr<const IDiscreteFunction>&);
