#include <catch2/catch.hpp>

#include <language/ast/ASTBuilder.hpp>
#include <language/ast/ASTModulesImporter.hpp>
#include <language/ast/ASTNodeDataTypeBuilder.hpp>
#include <language/ast/ASTNodeDeclarationToAffectationConverter.hpp>
#include <language/ast/ASTNodeExpressionBuilder.hpp>
#include <language/ast/ASTNodeTypeCleaner.hpp>
#include <language/ast/ASTSymbolTableBuilder.hpp>
#include <language/modules/MathModule.hpp>

#define CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, variable_name, expected_value)                \
  {                                                                                                  \
    string_input input{data, "test.pgs"};                                                            \
    auto ast = ASTBuilder::build(input);                                                             \
                                                                                                     \
    ASTModulesImporter{*ast};                                                                        \
    ASTNodeTypeCleaner<language::import_instruction>{*ast};                                          \
                                                                                                     \
    ASTSymbolTableBuilder{*ast};                                                                     \
    ASTNodeDataTypeBuilder{*ast};                                                                    \
                                                                                                     \
    ASTNodeDeclarationToAffectationConverter{*ast};                                                  \
    ASTNodeTypeCleaner<language::var_declaration>{*ast};                                             \
    ASTNodeTypeCleaner<language::fct_declaration>{*ast};                                             \
                                                                                                     \
    ASTNodeExpressionBuilder{*ast};                                                                  \
    ExecutionPolicy exec_policy;                                                                     \
    ast->execute(exec_policy);                                                                       \
                                                                                                     \
    auto symbol_table = ast->m_symbol_table;                                                         \
                                                                                                     \
    using namespace TAO_PEGTL_NAMESPACE;                                                             \
    position use_position{internal::iterator{"fixture"}, "fixture"};                                 \
    use_position.byte    = 10000;                                                                    \
    auto [symbol, found] = symbol_table->find(variable_name, use_position);                          \
                                                                                                     \
    using namespace Catch::Matchers;                                                                 \
                                                                                                     \
    REQUIRE_THAT(found, Predicate<bool>([](bool found) -> bool { return found; },                    \
                                        std::string{"Cannot find symbol '"} + variable_name + "'")); \
                                                                                                     \
    auto attributes = symbol->attributes();                                                          \
    auto value      = std::get<decltype(expected_value)>(attributes.value());                        \
                                                                                                     \
    REQUIRE(value == expected_value);                                                                \
  }

// clazy:excludeall=non-pod-global-static

TEST_CASE("BuiltinFunctionProcessor", "[language]")
{
  SECTION("math module functions")
  {
    // @note HERE we do not use SECTION to be able to count tests and to check
    // that all math functions are actually tested

    std::set<std::string> tested_function_set;
    {   // sqrt
      tested_function_set.insert("sqrt");
      std::string_view data = R"(
import math;
let x:R, x = sqrt(4);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::sqrt(4l)});
    }

    {   // abs
      tested_function_set.insert("abs");
      std::string_view data = R"(
import math;
let x:R, x = abs(-3.4);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::abs(-3.4)});
    }

    {   // sin
      tested_function_set.insert("sin");
      std::string_view data = R"(
import math;
let x:R, x = sin(1.3);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::sin(1.3)});
    }

    {   // cos
      tested_function_set.insert("cos");
      std::string_view data = R"(
import math;
let x:R, x = cos(1.3);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::cos(1.3)});
    }

    {   // tan
      tested_function_set.insert("tan");
      std::string_view data = R"(
import math;
let x:R, x = tan(1.3);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::tan(1.3)});
    }

    {   // asin
      tested_function_set.insert("asin");
      std::string_view data = R"(
import math;
let x:R, x = asin(0.7);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::asin(0.7)});
    }

    {   // acos
      tested_function_set.insert("acos");
      std::string_view data = R"(
import math;
let x:R, x = acos(0.7);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::acos(0.7)});
    }

    {   // atan
      tested_function_set.insert("atan");
      std::string_view data = R"(
import math;
let x:R, x = atan(0.7);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::atan(0.7)});
    }

    {   // atan2
      tested_function_set.insert("atan2");
      std::string_view data = R"(
import math;
let x:R, x = atan2(0.7, 0.4);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::atan2(0.7, 0.4)});
    }

    {   // sinh
      tested_function_set.insert("sinh");
      std::string_view data = R"(
import math;
let x:R, x = sinh(0.6);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::sinh(0.6)});
    }

    {   // cosh
      tested_function_set.insert("cosh");
      std::string_view data = R"(
import math;
let x:R, x = cosh(1.7);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::cosh(1.7)});
    }

    {   // tanh
      tested_function_set.insert("tanh");
      std::string_view data = R"(
import math;
let x:R, x = tanh(0.6);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::tanh(0.6)});
    }

    {   // asinh
      tested_function_set.insert("asinh");
      std::string_view data = R"(
import math;
let x:R, x = asinh(0.6);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::asinh(0.6)});
    }

    {   // acosh
      tested_function_set.insert("acosh");
      std::string_view data = R"(
import math;
let x:R, x = acosh(1.7);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::acosh(1.7)});
    }

    {   // tanh
      tested_function_set.insert("atanh");
      std::string_view data = R"(
import math;
let x:R, x = atanh(0.6);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::atanh(0.6)});
    }

    {   // exp
      tested_function_set.insert("exp");
      std::string_view data = R"(
import math;
let x:R, x = exp(1.7);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::exp(1.7)});
    }

    {   // log
      tested_function_set.insert("log");
      std::string_view data = R"(
import math;
let x:R, x = log(1.6);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::log(1.6)});
    }

    {   // pow
      tested_function_set.insert("pow");
      std::string_view data = R"(
import math;
let x:R, x = pow(1.6, 2.3);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "x", double{std::pow(1.6, 2.3)});
    }

    {   // ceil
      tested_function_set.insert("ceil");
      std::string_view data = R"(
import math;
let z:Z, z = ceil(-1.2);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "z", int64_t{-1});
    }

    {   // floor
      tested_function_set.insert("floor");
      std::string_view data = R"(
import math;
let z:Z, z = floor(-1.2);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "z", int64_t{-2});
    }

    {   // trunc
      tested_function_set.insert("trunc");
      std::string_view data = R"(
import math;
let z:Z, z = trunc(-0.2) + trunc(0.7);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "z", int64_t{0});
    }

    {   // round
      tested_function_set.insert("round");
      std::string_view data = R"(
import math;
let z:Z, z = round(-1.2);
)";
      CHECK_BUILTIN_FUNCTION_EVALUATION_RESULT(data, "z", int64_t{-1});
    }

    MathModule math_module;

    bool missing_test = false;
    for (const auto& [function_name, builtin_function] : math_module.getNameBuiltinFunctionMap()) {
      if (tested_function_set.find(function_name) == tested_function_set.end()) {
        UNSCOPED_INFO("function '" << function_name << "' is NOT tested");
        missing_test = true;
      }
    }
    REQUIRE_FALSE(missing_test);
  }
}
