#include <catch2/catch.hpp>

#include <language/ast/ASTBuilder.hpp>
#include <language/ast/ASTModulesImporter.hpp>
#include <language/ast/ASTNodeAffectationExpressionBuilder.hpp>
#include <language/ast/ASTNodeDataTypeBuilder.hpp>
#include <language/ast/ASTNodeDeclarationToAffectationConverter.hpp>
#include <language/ast/ASTNodeExpressionBuilder.hpp>
#include <language/ast/ASTNodeTypeCleaner.hpp>
#include <language/ast/ASTSymbolTableBuilder.hpp>
#include <language/utils/ASTPrinter.hpp>
#include <utils/Demangle.hpp>

#include <pegtl/string_input.hpp>

#include <sstream>

#define CHECK_EVALUATION_RESULT(data, variable_name, expected_value)          \
  {                                                                           \
    string_input input{data, "test.pgs"};                                     \
    auto ast = ASTBuilder::build(input);                                      \
                                                                              \
    ASTModulesImporter{*ast};                                                 \
    ASTNodeTypeCleaner<language::import_instruction>{*ast};                   \
                                                                              \
    ASTSymbolTableBuilder{*ast};                                              \
    ASTNodeDataTypeBuilder{*ast};                                             \
                                                                              \
    ASTNodeDeclarationToAffectationConverter{*ast};                           \
    ASTNodeTypeCleaner<language::var_declaration>{*ast};                      \
    ASTNodeTypeCleaner<language::fct_declaration>{*ast};                      \
                                                                              \
    ASTNodeExpressionBuilder{*ast};                                           \
    ExecutionPolicy exec_policy;                                              \
    ast->execute(exec_policy);                                                \
                                                                              \
    auto symbol_table = ast->m_symbol_table;                                  \
                                                                              \
    using namespace TAO_PEGTL_NAMESPACE;                                      \
    position use_position{internal::iterator{"fixture"}, "fixture"};          \
    use_position.byte    = 10000;                                             \
    auto [symbol, found] = symbol_table->find(variable_name, use_position);   \
                                                                              \
    auto attributes = symbol->attributes();                                   \
    auto value      = std::get<decltype(expected_value)>(attributes.value()); \
                                                                              \
    REQUIRE(value == expected_value);                                         \
  }

#define CHECK_EVALUATION_THROWS_WITH(data, error_message)     \
  {                                                           \
    auto eval = [&] {                                         \
      string_input input{data, "test.pgs"};                   \
      auto ast = ASTBuilder::build(input);                    \
                                                              \
      ASTModulesImporter{*ast};                               \
      ASTNodeTypeCleaner<language::import_instruction>{*ast}; \
                                                              \
      ASTSymbolTableBuilder{*ast};                            \
      ASTNodeDataTypeBuilder{*ast};                           \
                                                              \
      ASTNodeDeclarationToAffectationConverter{*ast};         \
      ASTNodeTypeCleaner<language::var_declaration>{*ast};    \
      ASTNodeTypeCleaner<language::fct_declaration>{*ast};    \
                                                              \
      ASTNodeExpressionBuilder{*ast};                         \
      ExecutionPolicy exec_policy;                            \
      ast->execute(exec_policy);                              \
    };                                                        \
                                                              \
    REQUIRE_THROWS_WITH(eval(), error_message);               \
  }

// clazy:excludeall=non-pod-global-static

TEST_CASE("ArraySubscriptProcessor", "[language]")
{
  SECTION("R^1 component access")
  {
    std::string_view data = R"(
let x : R^1, x = 1;
let x0: R, x0 = x[0];
)";
    CHECK_EVALUATION_RESULT(data, "x0", double{1});
  }

  SECTION("R^2 component access")
  {
    std::string_view data = R"(
let x : R^2, x = (1,2);
let x0: R, x0 = x[0];
let x1: R, x1 = x[1];
)";
    CHECK_EVALUATION_RESULT(data, "x0", double{1});
    CHECK_EVALUATION_RESULT(data, "x1", double{2});
  }

  SECTION("R^3 component access")
  {
    std::string_view data = R"(
let x : R^3, x = (1,2,3);
let x0 : R, x0 = x[0];
let x1 : R, x1 = x[1];
let x2 : R, x2 = x[2];
)";
    CHECK_EVALUATION_RESULT(data, "x0", double{1});
    CHECK_EVALUATION_RESULT(data, "x1", double{2});
    CHECK_EVALUATION_RESULT(data, "x2", double{3});
  }

  SECTION("R^d component access from integer expression")
  {
    std::string_view data = R"(
let x : R^3, x = (1,2,3);
let x0: R,  x0 = x[3-2-1];

let y : R^2, y = (2,7);
let y1: R,  y1 = y[2/2];

let z : R^1, z = 8;
let z0: R,  z0 = z[(2-2)*1];
)";
    CHECK_EVALUATION_RESULT(data, "x0", double{1});
    CHECK_EVALUATION_RESULT(data, "y1", double{7});
    CHECK_EVALUATION_RESULT(data, "z0", double{8});
  }

  SECTION("error invalid index type")
  {
    SECTION("R index type")
    {
      std::string_view data = R"(
let x : R^3, x = (1,2,3);
let x0: R,  x0 = x[2.3];
)";

      CHECK_EVALUATION_THROWS_WITH(data, std::string{"invalid implicit conversion: R -> Z"});
    }

    SECTION("string index type")
    {
      std::string_view data = R"(
let x : R^3, x = (1,2,3);
let x0: R,  x0 = x["foo"];
)";

      CHECK_EVALUATION_THROWS_WITH(data, std::string{"invalid implicit conversion: string -> Z"});
    }

    SECTION("R^d index type")
    {
      std::string_view data = R"(
let x : R^3, x = (1,2,3);
let x0: R,  x0 = x[x];
)";

      CHECK_EVALUATION_THROWS_WITH(data, std::string{"invalid implicit conversion: R^3 -> Z"});
    }
  }
}
