#include <catch2/catch.hpp>

#include <language/ast/ASTBuilder.hpp>
#include <language/ast/ASTNodeAffectationExpressionBuilder.hpp>
#include <language/ast/ASTNodeDataTypeBuilder.hpp>
#include <language/ast/ASTNodeDeclarationToAffectationConverter.hpp>
#include <language/ast/ASTNodeExpressionBuilder.hpp>
#include <language/ast/ASTNodeTypeCleaner.hpp>
#include <language/ast/ASTSymbolTableBuilder.hpp>
#include <language/utils/ASTPrinter.hpp>
#include <utils/Demangle.hpp>

#include <pegtl/string_input.hpp>

#include <sstream>

#define CHECK_AFFECTATION_RESULT(data, variable_name, expected_value)         \
  {                                                                           \
    string_input input{data, "test.pgs"};                                     \
    auto ast = ASTBuilder::build(input);                                      \
                                                                              \
    ASTSymbolTableBuilder{*ast};                                              \
    ASTNodeDataTypeBuilder{*ast};                                             \
                                                                              \
    ASTNodeDeclarationToAffectationConverter{*ast};                           \
    ASTNodeTypeCleaner<language::var_declaration>{*ast};                      \
                                                                              \
    ASTNodeExpressionBuilder{*ast};                                           \
    ExecutionPolicy exec_policy;                                              \
    ast->execute(exec_policy);                                                \
                                                                              \
    auto symbol_table = ast->m_symbol_table;                                  \
                                                                              \
    using namespace TAO_PEGTL_NAMESPACE;                                      \
    position use_position{internal::iterator{"fixture"}, "fixture"};          \
    use_position.byte    = 10000;                                             \
    auto [symbol, found] = symbol_table->find(variable_name, use_position);   \
                                                                              \
    auto attributes = symbol->attributes();                                   \
    auto value      = std::get<decltype(expected_value)>(attributes.value()); \
                                                                              \
    REQUIRE(value == expected_value);                                         \
  }

// clazy:excludeall=non-pod-global-static

TEST_CASE("ASTAffectationToStringProcessor", "[language]")
{
  SECTION("Affectations")
  {
    CHECK_AFFECTATION_RESULT(R"(let s : string; s = "foo";)", "s", std::string("foo"));
    CHECK_AFFECTATION_RESULT(R"(let n : N, n = 2; let s : string; s = n;)", "s", std::to_string(2ul));
    CHECK_AFFECTATION_RESULT(R"(let s : string; s = -1;)", "s", std::to_string(-1l));
    CHECK_AFFECTATION_RESULT(R"(let s : string; s = true;)", "s", std::to_string(true));
    CHECK_AFFECTATION_RESULT(R"(let s : string; s = 2.3;)", "s", std::to_string(2.3));
    {
      std::ostringstream os;
      os << TinyVector<1>{13} << std::ends;
      CHECK_AFFECTATION_RESULT(R"(let x : R^1, x = 13; let s : string; s = x;)", "s", os.str());
    }
    {
      std::ostringstream os;
      os << TinyVector<2>{2, 3} << std::ends;
      CHECK_AFFECTATION_RESULT(R"(let x : R^2, x = (2,3); let s : string; s = x;)", "s", os.str());
    }
    {
      std::ostringstream os;
      os << TinyVector<3>{1, 2, 3} << std::ends;
      CHECK_AFFECTATION_RESULT(R"(let x : R^3, x = (1,2,3); let s : string; s = x;)", "s", os.str());
    }
  }

  SECTION("+=")
  {
    CHECK_AFFECTATION_RESULT(R"(let s : string, s = "foo"; s += "bar";)", "s", std::string("foobar"));
    CHECK_AFFECTATION_RESULT(R"(let n : N, n = 2; let s : string, s = "foo"; s += n;)", "s",
                             (std::string("foo") + std::to_string(2ul)));
    CHECK_AFFECTATION_RESULT(R"(let s : string, s = "foo"; s += -1;)", "s", (std::string("foo") + std::to_string(-1l)));
    CHECK_AFFECTATION_RESULT(R"(let s : string, s = "foo"; s += true;)", "s",
                             (std::string("foo") + std::to_string(true)));
    CHECK_AFFECTATION_RESULT(R"(let s : string, s = "foo"; s += 2.3;)", "s",
                             (std::string("foo") + std::to_string(2.3)));
    {
      std::ostringstream os;
      os << "foo" << TinyVector<1>{13} << std::ends;
      CHECK_AFFECTATION_RESULT(R"(let x : R^1, x = 13; let s : string, s="foo"; s += x;)", "s", os.str());
    }
    {
      std::ostringstream os;
      os << "foo" << TinyVector<2>{2, 3} << std::ends;
      CHECK_AFFECTATION_RESULT(R"(let x : R^2, x = (2,3); let s : string, s="foo"; s += x;)", "s", os.str());
    }
    {
      std::ostringstream os;
      os << "foo" << TinyVector<3>{1, 2, 3} << std::ends;
      CHECK_AFFECTATION_RESULT(R"(let x : R^3, x = (1,2,3); let s : string, s="foo"; s += x;)", "s", os.str());
    }
  }
}
