#include <catch2/catch.hpp>

#include <language/ast/ASTBuilder.hpp>
#include <language/ast/ASTNodeAffectationExpressionBuilder.hpp>
#include <language/ast/ASTNodeDataTypeBuilder.hpp>
#include <language/ast/ASTNodeDeclarationToAffectationConverter.hpp>
#include <language/ast/ASTNodeExpressionBuilder.hpp>
#include <language/ast/ASTNodeTypeCleaner.hpp>
#include <language/ast/ASTSymbolTableBuilder.hpp>
#include <language/utils/ASTPrinter.hpp>
#include <utils/Demangle.hpp>

#include <pegtl/string_input.hpp>

#include <sstream>

#define CHECK_FOR_PROCESSOR_RESULT(data, variable_name, expected_value)       \
  {                                                                           \
    string_input input{data, "test.pgs"};                                     \
    auto ast = ASTBuilder::build(input);                                      \
                                                                              \
    ASTSymbolTableBuilder{*ast};                                              \
    ASTNodeDataTypeBuilder{*ast};                                             \
                                                                              \
    ASTNodeDeclarationToAffectationConverter{*ast};                           \
    ASTNodeTypeCleaner<language::var_declaration>{*ast};                      \
                                                                              \
    ASTNodeExpressionBuilder{*ast};                                           \
    ExecutionPolicy exec_policy;                                              \
    ast->execute(exec_policy);                                                \
                                                                              \
    auto symbol_table = ast->m_symbol_table;                                  \
                                                                              \
    using namespace TAO_PEGTL_NAMESPACE;                                      \
    position use_position{internal::iterator{"fixture"}, "fixture"};          \
    use_position.byte    = 10000;                                             \
    auto [symbol, found] = symbol_table->find(variable_name, use_position);   \
                                                                              \
    auto attributes = symbol->attributes();                                   \
    auto value      = std::get<decltype(expected_value)>(attributes.value()); \
                                                                              \
    REQUIRE(value == expected_value);                                         \
  }

#define CHECK_FOR_PROCESSOR_THROWS_WITH(data, error_message)          \
  {                                                                   \
    string_input input{data, "test.pgs"};                             \
    auto ast = ASTBuilder::build(input);                              \
                                                                      \
    ASTSymbolTableBuilder{*ast};                                      \
                                                                      \
    REQUIRE_THROWS_WITH(ASTNodeDataTypeBuilder{*ast}, error_message); \
  }

// clazy:excludeall=non-pod-global-static

TEST_CASE("ForProcessor", "[language]")
{
  SECTION("simple for")
  {
    std::string_view data = R"(
let i:N, i = 0;
for(let l:N, l=0; l<10; ++l) {
  i += l;
}
)";
    CHECK_FOR_PROCESSOR_RESULT(data, "i", 45ul);
  }

  SECTION("for with break")
  {
    std::string_view data = R"(
let i:N, i = 0;
for(let l:N, l=0; l<10; ++l) {
  i += l;
  if (i > 30) break;
}
)";
    CHECK_FOR_PROCESSOR_RESULT(data, "i", 36ul);
  }

  SECTION("for with continue")
  {
    std::string_view data = R"(
let i:N, i = 0;
for(let l:N, l=0; l<10; ++l) {
  if (l<3) continue;
  i += l;
}
)";
    CHECK_FOR_PROCESSOR_RESULT(data, "i", 42ul);
  }

  SECTION("errors")
  {
    SECTION("bad test type")
    {
      std::string_view data = R"(
for(let l:N, l=0; l; ++l) {
}
)";

      CHECK_FOR_PROCESSOR_THROWS_WITH(data, "invalid implicit conversion: N -> B");
    }
  }
}
