#include <catch2/catch.hpp>

#include <language/ast/ASTBuilder.hpp>
#include <language/ast/ASTNodeAffectationExpressionBuilder.hpp>
#include <language/ast/ASTNodeDataTypeBuilder.hpp>
#include <language/ast/ASTNodeDeclarationToAffectationConverter.hpp>
#include <language/ast/ASTNodeExpressionBuilder.hpp>
#include <language/ast/ASTNodeTypeCleaner.hpp>
#include <language/ast/ASTSymbolTableBuilder.hpp>
#include <language/utils/ASTPrinter.hpp>
#include <utils/Demangle.hpp>

#include <pegtl/string_input.hpp>

#include <sstream>

#define CHECK_AFFECTATION_RESULT(data, variable_name, expected_value)         \
  {                                                                           \
    string_input input{data, "test.pgs"};                                     \
    auto ast = ASTBuilder::build(input);                                      \
                                                                              \
    ASTSymbolTableBuilder{*ast};                                              \
    ASTNodeDataTypeBuilder{*ast};                                             \
                                                                              \
    ASTNodeDeclarationToAffectationConverter{*ast};                           \
    ASTNodeTypeCleaner<language::var_declaration>{*ast};                      \
                                                                              \
    ASTNodeExpressionBuilder{*ast};                                           \
    ExecutionPolicy exec_policy;                                              \
    ast->execute(exec_policy);                                                \
                                                                              \
    auto symbol_table = ast->m_symbol_table;                                  \
                                                                              \
    using namespace TAO_PEGTL_NAMESPACE;                                      \
    position use_position{internal::iterator{"fixture"}, "fixture"};          \
    use_position.byte    = 10000;                                             \
    auto [symbol, found] = symbol_table->find(variable_name, use_position);   \
                                                                              \
    auto attributes = symbol->attributes();                                   \
    auto value      = std::get<decltype(expected_value)>(attributes.value()); \
                                                                              \
    REQUIRE(value == expected_value);                                         \
  }

#define CHECK_AFFECTATION_THROWS_WITH(data, error_message)              \
  {                                                                     \
    string_input input{data, "test.pgs"};                               \
    auto ast = ASTBuilder::build(input);                                \
                                                                        \
    ASTSymbolTableBuilder{*ast};                                        \
    ASTNodeDataTypeBuilder{*ast};                                       \
                                                                        \
    ASTNodeDeclarationToAffectationConverter{*ast};                     \
    ASTNodeTypeCleaner<language::var_declaration>{*ast};                \
                                                                        \
    REQUIRE_THROWS_WITH(ASTNodeExpressionBuilder{*ast}, error_message); \
  }

// clazy:excludeall=non-pod-global-static

TEST_CASE("AffectationProcessor", "[language]")
{
  SECTION("Affectations")
  {
    SECTION("B")
    {
      CHECK_AFFECTATION_RESULT("let b : B; b = true;", "b", true);
    }

    SECTION("N")
    {
      CHECK_AFFECTATION_RESULT("let n : N, n = 1;", "n", 1ul);
      CHECK_AFFECTATION_RESULT("let m : N, m = 2; let n : N, n = m;", "n", 2ul);
      CHECK_AFFECTATION_RESULT("let n : N, n = true;", "n", 1ul);
      CHECK_AFFECTATION_RESULT("let n : N, n = false;", "n", 0ul);
    }

    SECTION("Z")
    {
      CHECK_AFFECTATION_RESULT("let z : Z, z = -1;", "z", -1l);
      CHECK_AFFECTATION_RESULT("let z : Z, z = true;", "z", 1l);
      CHECK_AFFECTATION_RESULT("let z : Z, z = false;", "z", 0l);
    }

    SECTION("R")
    {
      CHECK_AFFECTATION_RESULT("let r : R, r = -1;", "r", double{-1});
      CHECK_AFFECTATION_RESULT("let r : R, r = true;", "r", double{1});
      CHECK_AFFECTATION_RESULT("let r : R, r = false;", "r", double{0});
      CHECK_AFFECTATION_RESULT("let r : R, r = -2.3;", "r", double{-2.3});
    }

    SECTION("R^1")
    {
      CHECK_AFFECTATION_RESULT("let x : R^1, x = -1;", "x", (TinyVector<1>{-1}));
      CHECK_AFFECTATION_RESULT("let x : R^1, x = true;", "x", (TinyVector<1>{true}));
      CHECK_AFFECTATION_RESULT("let x : R^1, x = false;", "x", (TinyVector<1>{false}));
      CHECK_AFFECTATION_RESULT("let x : R^1, x = -2.3;", "x", (TinyVector<1>{-2.3}));
      CHECK_AFFECTATION_RESULT("let x : R^1; x[0] = -1;", "x", (TinyVector<1>{-1}));
      CHECK_AFFECTATION_RESULT("let x : R^1; x[0] = true;", "x", (TinyVector<1>{true}));
      CHECK_AFFECTATION_RESULT("let x : R^1; x[0] = false;", "x", (TinyVector<1>{false}));
      CHECK_AFFECTATION_RESULT("let x : R^1; x[0] = -2.3;", "x", (TinyVector<1>{-2.3}));

      CHECK_AFFECTATION_RESULT("let x : R^1, x = 0;", "x", (TinyVector<1>{zero}));
      CHECK_AFFECTATION_RESULT("let x : R^1; x = 0;", "x", (TinyVector<1>{zero}));
    }

    SECTION("R^2")
    {
      CHECK_AFFECTATION_RESULT("let x : R^2, x = (-1, true);", "x", (TinyVector<2>{-1, true}));
      CHECK_AFFECTATION_RESULT("let x : R^2, x = (true, false);", "x", (TinyVector<2>{true, false}));
      CHECK_AFFECTATION_RESULT("let x : R^2, x = (-0.3, 12);", "x", (TinyVector<2>{-0.3, 12}));
      CHECK_AFFECTATION_RESULT("let x : R^2; x[0] = -1; x[1] = true;", "x", (TinyVector<2>{-1, true}));
      CHECK_AFFECTATION_RESULT("let x : R^2; x[0] = true; x[1] = false;", "x", (TinyVector<2>{true, false}));
      CHECK_AFFECTATION_RESULT("let x : R^2; x[0] = -0.3; x[1] = 12;", "x", (TinyVector<2>{-0.3, 12}));

      CHECK_AFFECTATION_RESULT("let x : R^2, x = 0;", "x", (TinyVector<2>{zero}));
      CHECK_AFFECTATION_RESULT("let x : R^2; x = 0;", "x", (TinyVector<2>{zero}));
    }

    SECTION("R^3")
    {
      CHECK_AFFECTATION_RESULT("let x : R^3, x = (-1, true, false);", "x", (TinyVector<3>{-1, true, false}));
      CHECK_AFFECTATION_RESULT("let x : R^3, x = (-0.3, 12, 6.2);", "x", (TinyVector<3>{-0.3, 12, 6.2}));
      CHECK_AFFECTATION_RESULT("let x : R^3; x[0] = -1; x[1] = true; x[2] = false;", "x",
                               (TinyVector<3>{-1, true, false}));
      CHECK_AFFECTATION_RESULT("let x : R^3; x[0] = -0.3; x[1] = 12; x[2] = 6.2;", "x", (TinyVector<3>{-0.3, 12, 6.2}));

      CHECK_AFFECTATION_RESULT("let x : R^3, x = 0;", "x", (TinyVector<3>{zero}));
      CHECK_AFFECTATION_RESULT("let x : R^3; x = 0;", "x", (TinyVector<3>{zero}));
    }
  }

  SECTION("+=")
  {
    SECTION("N")
    {
      CHECK_AFFECTATION_RESULT("let n : N, n = 1; n += 3;", "n", 4ul);
      CHECK_AFFECTATION_RESULT("let m : N, m = 2; let n : N, n = 1; n += m;", "n", 3ul);
      CHECK_AFFECTATION_RESULT("let n : N, n = 1; n += true;", "n", 2ul);
      CHECK_AFFECTATION_RESULT("let n : N, n = 3; n += false;", "n", 3ul);
    }

    SECTION("Z")
    {
      CHECK_AFFECTATION_RESULT("let z : Z, z = 1; z += 3;", "z", 4l);
      CHECK_AFFECTATION_RESULT("let m : N, m = 2; let z : Z, z = 1; z += m;", "z", 3l);
      CHECK_AFFECTATION_RESULT("let z : Z, z = 1; z += true;", "z", 2l);
      CHECK_AFFECTATION_RESULT("let z : Z, z = 3; z += false;", "z", 3l);
    }

    SECTION("R")
    {
      CHECK_AFFECTATION_RESULT("let r : R, r = 1.2; r += 2.3;", "r", 3.5);
      CHECK_AFFECTATION_RESULT("let m : N, m = 2; let r : R, r = 1.3; r += m;", "r", 3.3);
      CHECK_AFFECTATION_RESULT("let r : R, r = 1.1; r += true;", "r", 2.1);
      CHECK_AFFECTATION_RESULT("let r : R, r = 3.3; r += false;", "r", 3.3);
      CHECK_AFFECTATION_RESULT("let r : R, r = 2; r += 1.1;", "r", 3.1);
    }

    SECTION("R^1")
    {
      CHECK_AFFECTATION_RESULT("let x : R^1, x = -1; let y : R^1, y = 1; x += y;", "x",
                               (TinyVector<1>{-1} + TinyVector<1>{1}));
      CHECK_AFFECTATION_RESULT("let x : R^1, x = 2; x[0] += 1;", "x", (TinyVector<1>{2} + TinyVector<1>{1}));
    }

    SECTION("R^2")
    {
      CHECK_AFFECTATION_RESULT("let x : R^2, x = (-1, true); let y : R^2, y = (1,3); x += y;", "x",
                               (TinyVector<2>{-1, true} + TinyVector<2>{1, 3}));
      CHECK_AFFECTATION_RESULT("let x : R^2, x = (-1, true); x[0] += 2; x[1] += 1;", "x",
                               (TinyVector<2>{-1, true} + TinyVector<2>{2, 1}));
    }

    SECTION("R^3")
    {
      CHECK_AFFECTATION_RESULT("let x : R^3, x = (-1, true, false); let y : R^3, y = (1,2,3); x += y;", "x",
                               (TinyVector<3>{-1, true, false} + TinyVector<3>{1, 2, 3}));
      CHECK_AFFECTATION_RESULT("let x : R^3, x = (-0.3, 12, 6.2); x[0] += 1; x[1] += -3; x[2] += 1;", "x",
                               (TinyVector<3>{-0.3, 12, 6.2} + TinyVector<3>{1, -3, 1}));
    }
  }

  SECTION("-=")
  {
    SECTION("N")
    {
      CHECK_AFFECTATION_RESULT("let n : N, n = 3; n -= 2;", "n", 1ul);
      CHECK_AFFECTATION_RESULT("let m : N, m = 2; let n : N, n = 4; n -= m;", "n", 2ul);
      CHECK_AFFECTATION_RESULT("let n : N, n = 1; n -= true;", "n", 0ul);
      CHECK_AFFECTATION_RESULT("let n : N, n = 3; n -= false;", "n", 3ul);
    }

    SECTION("Z")
    {
      CHECK_AFFECTATION_RESULT("let z : Z, z = 1; z -= 3;", "z", -2l);
      CHECK_AFFECTATION_RESULT("let m : N, m = 2; let z : Z, z = 1; z -= m;", "z", -1l);
      CHECK_AFFECTATION_RESULT("let z : Z, z = 1; z -= true;", "z", 0l);
      CHECK_AFFECTATION_RESULT("let z : Z, z = 3; z -= false;", "z", 3l);
    }

    SECTION("R")
    {
      CHECK_AFFECTATION_RESULT("let r : R, r = 1.1; r -= 2;", "r", (1.1 - 2l));
      CHECK_AFFECTATION_RESULT("let m : N, m = 2; let r : R, r = 1.3; r -= m;", "r", (1.3 - 2ul));
      CHECK_AFFECTATION_RESULT("let r : R, r = 1.1; r -= true;", "r", (1.1 - true));
      CHECK_AFFECTATION_RESULT("let r : R, r = 3.3; r -= false;", "r", 3.3);
      CHECK_AFFECTATION_RESULT("let r : R, r = 2; r -= 1.1;", "r", (2. - 1.1));
    }

    SECTION("R^1")
    {
      CHECK_AFFECTATION_RESULT("let x : R^1, x = -1; let y : R^1, y = 1; x -= y;", "x",
                               (TinyVector<1>{-1} - TinyVector<1>{1}));
      CHECK_AFFECTATION_RESULT("let x : R^1, x = 2; x[0] -= 1;", "x", (TinyVector<1>{2} - TinyVector<1>{1}));
    }

    SECTION("R^2")
    {
      CHECK_AFFECTATION_RESULT("let x : R^2, x = (-1, true); let y : R^2, y = (1,3); x -= y;", "x",
                               (TinyVector<2>{-1, true} - TinyVector<2>{1, 3}));
      CHECK_AFFECTATION_RESULT("let x : R^2, x = (-1, true); x[0] -= 2; x[1] -= 1;", "x",
                               (TinyVector<2>{-1, true} - TinyVector<2>{2, 1}));
    }

    SECTION("R^3")
    {
      CHECK_AFFECTATION_RESULT("let x : R^3, x = (-1, true, false); let y : R^3, y = (1,2,3); x-=y;", "x",
                               (TinyVector<3>{-1, true, false} - TinyVector<3>{1, 2, 3}));
      CHECK_AFFECTATION_RESULT("let x : R^3, x = (-0.3, 12, 6.2); x[0] -= 1; x[1] -= -3; x[2] -= 1;", "x",
                               (TinyVector<3>{-0.3, 12, 6.2} - TinyVector<3>{1, -3, 1}));
    }
  }

  SECTION("*=")
  {
    SECTION("N")
    {
      CHECK_AFFECTATION_RESULT("let n : N, n = 3; n *= 2;", "n", 6ul);
      CHECK_AFFECTATION_RESULT("let m : N, m = 2; let n : N, n = 4; n *= m;", "n", 8ul);
      CHECK_AFFECTATION_RESULT("let n : N, n = 1; n *= true;", "n", 1ul);
      CHECK_AFFECTATION_RESULT("let n : N, n = 3; n *= false;", "n", 0ul);
    }

    SECTION("Z")
    {
      CHECK_AFFECTATION_RESULT("let z : Z, z = 1; z *= 3;", "z", 3l);
      CHECK_AFFECTATION_RESULT("let m : N, m = 2; let z : Z, z = -2; z *= m;", "z", -4l);
      CHECK_AFFECTATION_RESULT("let z : Z, z = 1; z *= true;", "z", 1l);
      CHECK_AFFECTATION_RESULT("let z : Z, z = 3; z *= false;", "z", 0l);
    }

    SECTION("R")
    {
      CHECK_AFFECTATION_RESULT("let r : R, r = 1.1; r *= 2;", "r", (1.1 * 2l));
      CHECK_AFFECTATION_RESULT("let m : N, m = 2; let r : R, r = 1.3; r *= m;", "r", (1.3 * 2ul));
      CHECK_AFFECTATION_RESULT("let r : R, r = 1.1; r *= true;", "r", (1.1 * true));
      CHECK_AFFECTATION_RESULT("let r : R, r = 3.3; r *= false;", "r", (3.3 * false));
      CHECK_AFFECTATION_RESULT("let r : R, r = 2; r *= 1.1;", "r", (2. * 1.1));
    }

    SECTION("R^1")
    {
      CHECK_AFFECTATION_RESULT("let x : R^1, x = 2; x *= 2;", "x", (TinyVector<1>{TinyVector<1>{2} *= 2}));
      CHECK_AFFECTATION_RESULT("let x : R^1, x = 2; x[0] *= 1.3;", "x", (TinyVector<1>{2 * 1.3}));
    }

    SECTION("R^2")
    {
      CHECK_AFFECTATION_RESULT("let x : R^2, x = (-1, true);  x *= 3;", "x",
                               (TinyVector<2>{TinyVector<2>{-1, true} *= 3}));
      CHECK_AFFECTATION_RESULT("let x : R^2, x = (-1, true); x[0] *= 2; x[1] *= 3;", "x",
                               (TinyVector<2>{-1 * 2, true * 3}));
    }

    SECTION("R^3")
    {
      CHECK_AFFECTATION_RESULT("let x : R^3, x = (-1, true, false); x*=5.2;", "x",
                               (TinyVector<3>{TinyVector<3>{-1, true, false} *= 5.2}));
      CHECK_AFFECTATION_RESULT("let x : R^3, x = (-0.3, 12, 6.2); x[0] *= -1; x[1] *= -3; x[2] *= 2;", "x",
                               (TinyVector<3>{-0.3 * -1, 12 * -3, 6.2 * 2}));
    }
  }

  SECTION("/=")
  {
    SECTION("N")
    {
      CHECK_AFFECTATION_RESULT("let n : N, n = 4; n /= 2;", "n", 2ul);
      CHECK_AFFECTATION_RESULT("let m : N, m = 2; let n : N, n = 6; n /= m;", "n", 3ul);
      CHECK_AFFECTATION_RESULT("let n : N, n = 1; n /= true;", "n", 1ul);
    }

    SECTION("Z")
    {
      CHECK_AFFECTATION_RESULT("let z : Z, z = 7; z /= -3;", "z", -2l);
      CHECK_AFFECTATION_RESULT("let m : N, m = 3; let z : Z, z = 6; z /= m;", "z", 2l);
      CHECK_AFFECTATION_RESULT("let z : Z, z = 6; z /= true;", "z", 6l);
    }

    SECTION("R")
    {
      CHECK_AFFECTATION_RESULT("let r : R, r = 1.1; r /= 2;", "r", (1.1 / 2l));
      CHECK_AFFECTATION_RESULT("let m : N, m = 2; let r : R, r = 1.3; r /= m;", "r", (1.3 / 2ul));
      CHECK_AFFECTATION_RESULT("let r : R, r = 1.1; r /= true;", "r", (1.1 / true));
      CHECK_AFFECTATION_RESULT("let r : R, r = 2; r /= 1.1;", "r", (2. / 1.1));
    }

    SECTION("R^1")
    {
      CHECK_AFFECTATION_RESULT("let x : R^1, x = 2; x[0] /= 1.3;", "x", (TinyVector<1>{2 / 1.3}));
    }

    SECTION("R^2")
    {
      CHECK_AFFECTATION_RESULT("let x : R^2, x = (-1, true); x[0] /= 2; x[1] /= 3;", "x",
                               (TinyVector<2>{-1. / 2., true / 3.}));
    }

    SECTION("R^3")
    {
      CHECK_AFFECTATION_RESULT("let x : R^3, x = (-0.3, 12, 6.2); x[0] /= -1.2; x[1] /= -3.1; x[2] /= 2.4;", "x",
                               (TinyVector<3>{-0.3 / -1.2, 12 / -3.1, 6.2 / 2.4}));
    }
  }

  SECTION("errors")
  {
    SECTION("invalid implicit conversions")
    {
      SECTION("-> B")
      {
        CHECK_AFFECTATION_THROWS_WITH("let n : N, n = 1; let b : B; b = n;", "invalid implicit conversion: N -> B");
        CHECK_AFFECTATION_THROWS_WITH("let b : B; b = 1;", "invalid implicit conversion: Z -> B");
        CHECK_AFFECTATION_THROWS_WITH("let b : B; b = 2.3;", "invalid implicit conversion: R -> B");
        CHECK_AFFECTATION_THROWS_WITH("let b : B; b = \"foo\";", "invalid implicit conversion: string -> B");
      }

      SECTION("-> N")
      {
        CHECK_AFFECTATION_THROWS_WITH("let n : N, n = 2.3;", "invalid implicit conversion: R -> N");
        CHECK_AFFECTATION_THROWS_WITH("let n : N, n = \"bar\";", "invalid implicit conversion: string -> N");

        CHECK_AFFECTATION_THROWS_WITH("let n : N, n = 2; n += 1.1;", "invalid implicit conversion: R -> N");
        CHECK_AFFECTATION_THROWS_WITH("let n : N, n = 2; n += \"foo\";", "invalid implicit conversion: string -> N");

        CHECK_AFFECTATION_THROWS_WITH("let n : N, n = 2; n -= 1.1;", "invalid implicit conversion: R -> N");
        CHECK_AFFECTATION_THROWS_WITH("let n : N, n = 2; n -= \"bar\";", "invalid implicit conversion: string -> N");

        CHECK_AFFECTATION_THROWS_WITH("let n : N, n = 2; n *= 2.51;", "invalid implicit conversion: R -> N");
        CHECK_AFFECTATION_THROWS_WITH("let n : N, n = 2; n *= \"foobar\";", "invalid implicit conversion: string -> N");

        CHECK_AFFECTATION_THROWS_WITH("let n : N, n = 2; n /= 2.51;", "invalid implicit conversion: R -> N");
        CHECK_AFFECTATION_THROWS_WITH("let n : N, n = 2; n /= \"foo\";", "invalid implicit conversion: string -> N");
      }

      SECTION("-> Z")
      {
        CHECK_AFFECTATION_THROWS_WITH("let z : Z, z = -2.3;", "invalid implicit conversion: R -> Z");
        CHECK_AFFECTATION_THROWS_WITH("let z : Z, z = \"foobar\";", "invalid implicit conversion: string -> Z");

        CHECK_AFFECTATION_THROWS_WITH("let z : Z, z = 2; z += 1.1;", "invalid implicit conversion: R -> Z");
        CHECK_AFFECTATION_THROWS_WITH("let z : Z, z = 2; z += \"foo\";", "invalid implicit conversion: string -> Z");

        CHECK_AFFECTATION_THROWS_WITH("let z : Z, z = 2; z -= 2.1;", "invalid implicit conversion: R -> Z");
        CHECK_AFFECTATION_THROWS_WITH("let z : Z, z = 2; z -= \"bar\";", "invalid implicit conversion: string -> Z");

        CHECK_AFFECTATION_THROWS_WITH("let z : Z, z = 2; z *= -2.51;", "invalid implicit conversion: R -> Z");
        CHECK_AFFECTATION_THROWS_WITH("let z : Z, z = 2; z *= \"foobar\";", "invalid implicit conversion: string -> Z");

        CHECK_AFFECTATION_THROWS_WITH("let z : Z, z = 4; z /= -2.;", "invalid implicit conversion: R -> Z");
        CHECK_AFFECTATION_THROWS_WITH("let z : Z, z = 2; z /= \"foo\";", "invalid implicit conversion: string -> Z");
      }

      SECTION("-> R")
      {
        CHECK_AFFECTATION_THROWS_WITH("let x : R, x = \"foobar\";", "invalid implicit conversion: string -> R");
        CHECK_AFFECTATION_THROWS_WITH("let x : R, x = 2.3; x += \"foo\";", "invalid implicit conversion: string -> R");
        CHECK_AFFECTATION_THROWS_WITH("let x : R, x = 2.1; x -= \"bar\";", "invalid implicit conversion: string -> R");
        CHECK_AFFECTATION_THROWS_WITH("let x : R, x = 1.2; x *= \"foobar\";",
                                      "invalid implicit conversion: string -> R");
        CHECK_AFFECTATION_THROWS_WITH("let x : R, x =-2.3; x /= \"foo\";", "invalid implicit conversion: string -> R");
      }

      SECTION("-> R^n")
      {
        CHECK_AFFECTATION_THROWS_WITH("let x : R^2, x = \"foobar\";", "invalid implicit conversion: string -> R^2");
        CHECK_AFFECTATION_THROWS_WITH("let x : R^3, x = \"foobar\";", "invalid implicit conversion: string -> R^3");

        CHECK_AFFECTATION_THROWS_WITH("let x : R^2, x = 3.2;", "invalid implicit conversion: R -> R^2");
        CHECK_AFFECTATION_THROWS_WITH("let x : R^3, x = 2.3;", "invalid implicit conversion: R -> R^3");

        CHECK_AFFECTATION_THROWS_WITH("let x : R^2, x = 4;", "invalid implicit conversion: Z -> R^2");
        CHECK_AFFECTATION_THROWS_WITH("let x : R^3, x = 3;", "invalid implicit conversion: Z -> R^3");

        CHECK_AFFECTATION_THROWS_WITH("let x : R^1, x = 0; let y : R^2, y = x;",
                                      "incompatible dimensions in affectation");
        CHECK_AFFECTATION_THROWS_WITH("let x : R^1, x = 0; let y : R^3, y = x;",
                                      "incompatible dimensions in affectation");

        CHECK_AFFECTATION_THROWS_WITH("let x : R^2, x = 0; let y : R^1, y = x;",
                                      "incompatible dimensions in affectation");
        CHECK_AFFECTATION_THROWS_WITH("let x : R^2, x = 0; let y : R^3, y = x;",
                                      "incompatible dimensions in affectation");

        CHECK_AFFECTATION_THROWS_WITH("let x : R^3, x = 0; let y : R^1, y = x;",
                                      "incompatible dimensions in affectation");
        CHECK_AFFECTATION_THROWS_WITH("let x : R^3, x = 0; let y : R^2, y = x;",
                                      "incompatible dimensions in affectation");
      }
    }
  }
}
