#include <catch2/catch.hpp>

#include <language/ast/ASTBuilder.hpp>
#include <language/ast/ASTSymbolTableBuilder.hpp>

#include <pegtl/string_input.hpp>

// clazy:excludeall=non-pod-global-static

TEST_CASE("ASTSymbolTableBuilder", "[language]")
{
  SECTION("Build symbols")
  {
    std::string_view data = R"(
let n:N, n = 2;
{
 let m:N, m = n;
 let n:R, n = m/3.;
}
)";

    string_input input{data, "test.pgs"};
    auto ast = ASTBuilder::build(input);

    ASTSymbolTableBuilder{*ast};
  }

  SECTION("Populate symbol table")
  {
    std::string_view data = R"(
let b:B;
let n:N;
let z:Z;
let x:R;
let (c0,c1,c2,c3):R*Z*N*B;
let f: R*Z*B->R, (x,n,z) -> x+n;
)";

    string_input input{data, "test.pgs"};
    auto ast = ASTBuilder::build(input);

    ASTSymbolTableBuilder{*ast};

    auto root_st = ast->m_symbol_table;

    std::stringstream st_output;
    st_output << '\n' << *root_st;

    std::stringstream expected_output;
    expected_output << '\n'
                    << "-- Symbol table state -- parent : " << static_cast<SymbolTable*>(nullptr) << '\n'
                    << " b: undefined:--\n"
                    << " n: undefined:--\n"
                    << " z: undefined:--\n"
                    << " x: undefined:--\n"
                    << " c0: undefined:--\n"
                    << " c1: undefined:--\n"
                    << " c2: undefined:--\n"
                    << " c3: undefined:--\n"
                    << " f: undefined:0\n"
                    << "------------------------\n";

    REQUIRE(st_output.str() == expected_output.str());
  }

  SECTION("errors")
  {
    SECTION("Undeclared symbol")
    {
      std::string_view data = R"(
let n:N, n = a;
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      REQUIRE_THROWS_AS(ASTSymbolTableBuilder{*ast}, parse_error);
    }

    SECTION("Re-declared symbol")
    {
      std::string_view data = R"(
let n:N, n = 0;
let n:N, n = 1;
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      REQUIRE_THROWS_AS(ASTSymbolTableBuilder{*ast}, parse_error);
    }

    SECTION("Re-declared symbol (function)")
    {
      std::string_view data = R"(
let f:N;
let f : R -> R, x -> 1;
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      REQUIRE_THROWS_AS(ASTSymbolTableBuilder{*ast}, parse_error);
    }

    SECTION("Re-declared parameter (function)")
    {
      std::string_view data = R"(
let f : R*R*N -> R, (x,y,x) -> 1;
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      REQUIRE_THROWS_AS(ASTSymbolTableBuilder{*ast}, parse_error);
    }
  }
}
