#include <catch2/catch.hpp>

#include <language/ast/ASTBuilder.hpp>
#include <language/ast/ASTNodeDataTypeBuilder.hpp>
#include <language/ast/ASTNodeJumpPlacementChecker.hpp>
#include <language/ast/ASTSymbolTableBuilder.hpp>

#include <pegtl/string_input.hpp>

// clazy:excludeall=non-pod-global-static

TEST_CASE("ASTNodeJumpPlacementChecker", "[language]")
{
  SECTION("break")
  {
    SECTION("in for loop")
    {
      std::string_view data = R"(
for(;;) {
  break;
}
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);
      ASTSymbolTableBuilder{*ast};
      ASTNodeDataTypeBuilder{*ast};

      REQUIRE_NOTHROW(ASTNodeJumpPlacementChecker{*ast});
    }

    SECTION("in while loop")
    {
      std::string_view data = R"(
while(true) {
  break;
}
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);
      ASTSymbolTableBuilder{*ast};
      ASTNodeDataTypeBuilder{*ast};

      REQUIRE_NOTHROW(ASTNodeJumpPlacementChecker{*ast});
    }

    SECTION("in do while loop")
    {
      std::string_view data = R"(
do {
  break;
} while(true);
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);
      ASTSymbolTableBuilder{*ast};
      ASTNodeDataTypeBuilder{*ast};

      REQUIRE_NOTHROW(ASTNodeJumpPlacementChecker{*ast});
    }

    SECTION("misplaced")
    {
      std::string_view data = R"(
{
  break;
}
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);
      ASTSymbolTableBuilder{*ast};
      ASTNodeDataTypeBuilder{*ast};

      ast->children[0]->m_data_type = ASTNodeDataType::undefined_t;

      REQUIRE_THROWS_AS(ASTNodeJumpPlacementChecker{*ast}, parse_error);
    }
  }

  SECTION("continue")
  {
    SECTION("in for loop")
    {
      std::string_view data = R"(
for(;;) {
  continue;
}
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);
      ASTSymbolTableBuilder{*ast};
      ASTNodeDataTypeBuilder{*ast};

      REQUIRE_NOTHROW(ASTNodeJumpPlacementChecker{*ast});
    }

    SECTION("in while loop")
    {
      std::string_view data = R"(
while(true) {
  continue;
}
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);
      ASTSymbolTableBuilder{*ast};
      ASTNodeDataTypeBuilder{*ast};

      REQUIRE_NOTHROW(ASTNodeJumpPlacementChecker{*ast});
    }

    SECTION("in do while loop")
    {
      std::string_view data = R"(
do {
  continue;
} while(true);
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);
      ASTSymbolTableBuilder{*ast};
      ASTNodeDataTypeBuilder{*ast};

      REQUIRE_NOTHROW(ASTNodeJumpPlacementChecker{*ast});
    }

    SECTION("misplaced")
    {
      std::string_view data = R"(
{
  continue;
}
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);
      ASTSymbolTableBuilder{*ast};
      ASTNodeDataTypeBuilder{*ast};

      ast->children[0]->m_data_type = ASTNodeDataType::undefined_t;

      REQUIRE_THROWS_AS(ASTNodeJumpPlacementChecker{*ast}, parse_error);
    }
  }
}
