#include <catch2/catch.hpp>

#include <language/ast/ASTBuilder.hpp>
#include <language/ast/ASTModulesImporter.hpp>
#include <language/ast/ASTNodeDataTypeBuilder.hpp>
#include <language/ast/ASTNodeExpressionBuilder.hpp>
#include <language/ast/ASTNodeFunctionEvaluationExpressionBuilder.hpp>
#include <language/ast/ASTNodeFunctionExpressionBuilder.hpp>
#include <language/ast/ASTNodeTypeCleaner.hpp>
#include <language/ast/ASTSymbolTableBuilder.hpp>
#include <language/utils/ASTPrinter.hpp>
#include <language/utils/BuiltinFunctionEmbedder.hpp>
#include <utils/Demangle.hpp>

#include <pegtl/string_input.hpp>

#include <memory>
#include <unordered_map>

namespace test_only
{
class BuiltinFunctionRegister
{
 private:
  std::unordered_map<std::string, std::shared_ptr<IBuiltinFunctionEmbedder>> m_name_builtin_function_map;

  void
  _populateNameBuiltinFunctionMap()
  {
    m_name_builtin_function_map.insert(std::make_pair("RtoR", std::make_shared<BuiltinFunctionEmbedder<double(double)>>(
                                                                [](double x) -> double { return x + 1; })));

    m_name_builtin_function_map.insert(
      std::make_pair("ZtoR", std::make_shared<BuiltinFunctionEmbedder<double(int64_t)>>(
                               [](int64_t z) -> double { return 0.5 * z; })));

    m_name_builtin_function_map.insert(
      std::make_pair("NtoR", std::make_shared<BuiltinFunctionEmbedder<double(uint64_t)>>(
                               [](uint64_t n) -> double { return 0.5 * n; })));

    m_name_builtin_function_map.insert(std::make_pair("BtoR", std::make_shared<BuiltinFunctionEmbedder<double(bool)>>(
                                                                [](bool b) -> double { return b; })));

    m_name_builtin_function_map.insert(
      std::make_pair("R2toB", std::make_shared<BuiltinFunctionEmbedder<bool(double, double)>>(
                                [](double x, double y) -> bool { return x > y; })));

    m_name_builtin_function_map.insert(
      std::make_pair("StoB", std::make_shared<BuiltinFunctionEmbedder<bool(std::string)>>(
                               [](const std::string& s) -> bool { return s.size() > 0; })));
  }

 public:
  BuiltinFunctionRegister(ASTNode& root_node)
  {
    SymbolTable& symbol_table = *root_node.m_symbol_table;

    auto& builtin_function_embedder_table = symbol_table.builtinFunctionEmbedderTable();

    this->_populateNameBuiltinFunctionMap();

    for (const auto& [symbol_name, builtin_function] : m_name_builtin_function_map) {
      auto [i_symbol, success] = symbol_table.add(symbol_name, root_node.begin());

      if (not success) {
        std::ostringstream error_message;
        error_message << "cannot add symbol '" << symbol_name << "' it is already defined";
        throw parse_error(error_message.str(), root_node.begin());
      }

      i_symbol->attributes().setDataType(ASTNodeDataType::builtin_function_t);
      i_symbol->attributes().setIsInitialized();
      i_symbol->attributes().value() = builtin_function_embedder_table.size();

      builtin_function_embedder_table.add(builtin_function);
    }
  }
};
}   // namespace test_only

#define CHECK_AST(data, expected_output)                                                            \
  {                                                                                                 \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>);                  \
    static_assert((std::is_same_v<std::decay_t<decltype(expected_output)>, std::string_view>) or    \
                  (std::is_same_v<std::decay_t<decltype(expected_output)>, std::string>));          \
                                                                                                    \
    string_input input{data, "test.pgs"};                                                           \
    auto ast = ASTBuilder::build(input);                                                            \
                                                                                                    \
    test_only::BuiltinFunctionRegister{*ast};                                                       \
                                                                                                    \
    ASTSymbolTableBuilder{*ast};                                                                    \
    ASTNodeDataTypeBuilder{*ast};                                                                   \
                                                                                                    \
    ASTNodeTypeCleaner<language::var_declaration>{*ast};                                            \
    ASTNodeExpressionBuilder{*ast};                                                                 \
                                                                                                    \
    std::stringstream ast_output;                                                                   \
    ast_output << '\n' << ASTPrinter{*ast, ASTPrinter::Format::raw, {ASTPrinter::Info::exec_type}}; \
                                                                                                    \
    REQUIRE(ast_output.str() == expected_output);                                                   \
  }

#define CHECK_AST_THROWS_WITH(data, expected_error)                                                 \
  {                                                                                                 \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>);                  \
    static_assert((std::is_same_v<std::decay_t<decltype(expected_error)>, std::string_view>) or     \
                  (std::is_same_v<std::decay_t<decltype(expected_error)>, std::string>));           \
                                                                                                    \
    string_input input{data, "test.pgs"};                                                           \
    auto ast = ASTBuilder::build(input);                                                            \
                                                                                                    \
    test_only::BuiltinFunctionRegister{*ast};                                                       \
                                                                                                    \
    ASTSymbolTableBuilder{*ast};                                                                    \
    ASTNodeDataTypeBuilder{*ast};                                                                   \
                                                                                                    \
    ASTNodeTypeCleaner<language::var_declaration>{*ast};                                            \
    REQUIRE_THROWS_WITH(ASTNodeExpressionBuilder{*ast}, Catch::Matchers::Contains(expected_error)); \
  }

// clazy:excludeall=non-pod-global-static

TEST_CASE("ASTNodeBuiltinFunctionExpressionBuilder", "[language]")
{
  SECTION("R -> R")
  {
    SECTION("from R")
    {
      std::string_view data = R"(
RtoR(1.);
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:RtoR:NameProcessor)
     `-(language::real:1.:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("from Z")
    {
      std::string_view data = R"(
RtoR(1);
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:RtoR:NameProcessor)
     `-(language::integer:1:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("from N")
    {
      std::string_view data = R"(
let n : N, n = 1;
RtoR(n);
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:RtoR:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("from B")
    {
      std::string_view data = R"(
RtoR(true);
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:RtoR:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("Z -> R")
  {
    SECTION("from Z")
    {
      std::string_view data = R"(
ZtoR(1);
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:ZtoR:NameProcessor)
     `-(language::integer:1:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("from N")
    {
      std::string_view data = R"(
let n : N, n = 1;
ZtoR(n);
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:ZtoR:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("from B")
    {
      std::string_view data = R"(
ZtoR(true);
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:ZtoR:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("N -> R")
  {
    SECTION("from Z")
    {
      std::string_view data = R"(
NtoR(1);
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:NtoR:NameProcessor)
     `-(language::integer:1:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("from N")
    {
      std::string_view data = R"(
let n : N, n = 1;
NtoR(n);
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:NtoR:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("from B")
    {
      std::string_view data = R"(
NtoR(true);
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:NtoR:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("B -> R")
  {
    SECTION("from B")
    {
      std::string_view data = R"(
BtoR(true);
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:BtoR:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("R2 -> B")
  {
    std::string_view data = R"(
R2toB(1., 0.);
)";

    std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:R2toB:NameProcessor)
     `-(language::function_argument_list:ASTNodeExpressionListProcessor)
         +-(language::real:1.:ValueProcessor)
         `-(language::real:0.:ValueProcessor)
)";

    CHECK_AST(data, result);
  }

  SECTION("string -> B")
  {
    std::string_view data = R"(
StoB("foo");
)";

    std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::function_evaluation:BuiltinFunctionProcessor)
     +-(language::name:StoB:NameProcessor)
     `-(language::literal:"foo":ValueProcessor)
)";

    CHECK_AST(data, result);
  }

  SECTION("errors")
  {
    SECTION("bad number of arguments")
    {
      std::string_view data = R"(
BtoR(true, false);
)";
      CHECK_AST_THROWS_WITH(data, std::string{"bad number of arguments:"});
    }

    SECTION("bad number of arguments 2")
    {
      std::string_view data = R"(
R2toB(3);
)";
      CHECK_AST_THROWS_WITH(data, std::string{"bad number of arguments:"});
    }

    SECTION("invalid argument type")
    {
      std::string_view data = R"(
RtoR("foo");
)";
      CHECK_AST_THROWS_WITH(data, std::string{"invalid implicit conversion: string -> R"});
    }
  }
}
