#include <catch2/catch.hpp>

#include <language/ast/ASTBuilder.hpp>
#include <language/ast/ASTNodeBinaryOperatorExpressionBuilder.hpp>
#include <language/ast/ASTNodeDataTypeBuilder.hpp>
#include <language/ast/ASTNodeDeclarationToAffectationConverter.hpp>
#include <language/ast/ASTNodeExpressionBuilder.hpp>
#include <language/ast/ASTNodeTypeCleaner.hpp>
#include <language/ast/ASTSymbolTableBuilder.hpp>
#include <language/utils/ASTPrinter.hpp>
#include <utils/Demangle.hpp>

#include <pegtl/string_input.hpp>

#define CHECK_AST(data, expected_output)                                                            \
  {                                                                                                 \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>);                  \
    static_assert(std::is_same_v<std::decay_t<decltype(expected_output)>, std::string_view> or      \
                  std::is_same_v<std::decay_t<decltype(expected_output)>, std::string>);            \
                                                                                                    \
    string_input input{data, "test.pgs"};                                                           \
    auto ast = ASTBuilder::build(input);                                                            \
                                                                                                    \
    ASTSymbolTableBuilder{*ast};                                                                    \
    ASTNodeDataTypeBuilder{*ast};                                                                   \
                                                                                                    \
    ASTNodeDeclarationToAffectationConverter{*ast};                                                 \
    ASTNodeTypeCleaner<language::var_declaration>{*ast};                                            \
                                                                                                    \
    ASTNodeExpressionBuilder{*ast};                                                                 \
                                                                                                    \
    std::stringstream ast_output;                                                                   \
    ast_output << '\n' << ASTPrinter{*ast, ASTPrinter::Format::raw, {ASTPrinter::Info::exec_type}}; \
                                                                                                    \
    REQUIRE(ast_output.str() == expected_output);                                                   \
  }

#define REQUIRE_AST_THROWS_WITH(data, expected_output)                             \
  {                                                                                \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>); \
                                                                                   \
    string_input input{data, "test.pgs"};                                          \
    auto ast = ASTBuilder::build(input);                                           \
                                                                                   \
    ASTSymbolTableBuilder{*ast};                                                   \
    ASTNodeDataTypeBuilder{*ast};                                                  \
                                                                                   \
    ASTNodeDeclarationToAffectationConverter{*ast};                                \
    ASTNodeTypeCleaner<language::var_declaration>{*ast};                           \
                                                                                   \
    REQUIRE_THROWS_WITH(ASTNodeExpressionBuilder{*ast}, expected_output);          \
  }

// clazy:excludeall=non-pod-global-static

TEST_CASE("ASTNodeBinaryOperatorExpressionBuilder", "[language]")
{
  SECTION("multiply")
  {
    SECTION("B")
    {
      std::string_view data = R"(
let b : B;
b*true;
false*b*true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:ValueProcessor)
 `-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, long, bool>)
     +-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, bool, bool>)
     |   +-(language::false_kw:ValueProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
let n : N;
let m : N;
n*m*n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, unsigned long, unsigned long>)
     +-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, unsigned long, unsigned long>)
     |   +-(language::name:n:NameProcessor)
     |   `-(language::name:m:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
let a : Z;
a*3*a;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, long, long>)
     +-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, long, long>)
     |   +-(language::name:a:NameProcessor)
     |   `-(language::integer:3:ValueProcessor)
     `-(language::name:a:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3*1.2*2*false;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, double, bool>)
     +-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, double, long>)
     |   +-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, double, double>)
     |   |   +-(language::real:2.3:ValueProcessor)
     |   |   `-(language::real:1.2:ValueProcessor)
     |   `-(language::integer:2:ValueProcessor)
     `-(language::false_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^1")
    {
      std::string_view data = R"(
let x : R^1, x = 3.7;
2*x;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, double>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::real:3.7:ValueProcessor)
 `-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, long, TinyVector<1ul, double> >)
     +-(language::integer:2:ValueProcessor)
     `-(language::name:x:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^2")
    {
      std::string_view data = R"(
let x : R^2, x = (3.2,6);
2*x;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<2ul, double> >)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::real:3.2:ValueProcessor)
 |       `-(language::integer:6:ValueProcessor)
 `-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, long, TinyVector<2ul, double> >)
     +-(language::integer:2:ValueProcessor)
     `-(language::name:x:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^3")
    {
      std::string_view data = R"(
let x : R^3, x = (3.2,6,1.2);
2*x;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<3ul, double> >)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::real:3.2:ValueProcessor)
 |       +-(language::integer:6:ValueProcessor)
 |       `-(language::real:1.2:ValueProcessor)
 `-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, long, TinyVector<3ul, double> >)
     +-(language::integer:2:ValueProcessor)
     `-(language::name:x:NameProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("divide")
  {
    SECTION("B")
    {
      std::string_view data = R"(
let b : B;
b/true;
false/b/true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::divide_op:BinaryExpressionProcessor<language::divide_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:ValueProcessor)
 `-(language::divide_op:BinaryExpressionProcessor<language::divide_op, long, bool>)
     +-(language::divide_op:BinaryExpressionProcessor<language::divide_op, bool, bool>)
     |   +-(language::false_kw:ValueProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
let n : N;
let m : N;
n/m/n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::divide_op:BinaryExpressionProcessor<language::divide_op, unsigned long, unsigned long>)
     +-(language::divide_op:BinaryExpressionProcessor<language::divide_op, unsigned long, unsigned long>)
     |   +-(language::name:n:NameProcessor)
     |   `-(language::name:m:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
let a : Z;
a/3/a;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::divide_op:BinaryExpressionProcessor<language::divide_op, long, long>)
     +-(language::divide_op:BinaryExpressionProcessor<language::divide_op, long, long>)
     |   +-(language::name:a:NameProcessor)
     |   `-(language::integer:3:ValueProcessor)
     `-(language::name:a:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3/1.2/2/false;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::divide_op:BinaryExpressionProcessor<language::divide_op, double, bool>)
     +-(language::divide_op:BinaryExpressionProcessor<language::divide_op, double, long>)
     |   +-(language::divide_op:BinaryExpressionProcessor<language::divide_op, double, double>)
     |   |   +-(language::real:2.3:ValueProcessor)
     |   |   `-(language::real:1.2:ValueProcessor)
     |   `-(language::integer:2:ValueProcessor)
     `-(language::false_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("plus")
  {
    SECTION("B")
    {
      std::string_view data = R"(
let b : B;
b+true;
false+b+true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::plus_op:BinaryExpressionProcessor<language::plus_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:ValueProcessor)
 `-(language::plus_op:BinaryExpressionProcessor<language::plus_op, long, bool>)
     +-(language::plus_op:BinaryExpressionProcessor<language::plus_op, bool, bool>)
     |   +-(language::false_kw:ValueProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
let n : N;
let m : N;
n+m+n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:BinaryExpressionProcessor<language::plus_op, unsigned long, unsigned long>)
     +-(language::plus_op:BinaryExpressionProcessor<language::plus_op, unsigned long, unsigned long>)
     |   +-(language::name:n:NameProcessor)
     |   `-(language::name:m:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
let a : Z;
a+3+a;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:BinaryExpressionProcessor<language::plus_op, long, long>)
     +-(language::plus_op:BinaryExpressionProcessor<language::plus_op, long, long>)
     |   +-(language::name:a:NameProcessor)
     |   `-(language::integer:3:ValueProcessor)
     `-(language::name:a:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3+1.2+2+false;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:BinaryExpressionProcessor<language::plus_op, double, bool>)
     +-(language::plus_op:BinaryExpressionProcessor<language::plus_op, double, long>)
     |   +-(language::plus_op:BinaryExpressionProcessor<language::plus_op, double, double>)
     |   |   +-(language::real:2.3:ValueProcessor)
     |   |   `-(language::real:1.2:ValueProcessor)
     |   `-(language::integer:2:ValueProcessor)
     `-(language::false_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^1")
    {
      std::string_view data = R"(
let x : R^1, x = 1;
let y : R^1, y = 2;
x+y;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, long>)
 |   +-(language::name:y:NameProcessor)
 |   `-(language::integer:2:ValueProcessor)
 `-(language::plus_op:BinaryExpressionProcessor<language::plus_op, TinyVector<1ul, double>, TinyVector<1ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^2")
    {
      std::string_view data = R"(
let x : R^2, x = (1,2);
let y : R^2, y = (0.3,0.7);
x+y;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<2ul, double> >)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::integer:1:ValueProcessor)
 |       `-(language::integer:2:ValueProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<2ul, double> >)
 |   +-(language::name:y:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::real:0.3:ValueProcessor)
 |       `-(language::real:0.7:ValueProcessor)
 `-(language::plus_op:BinaryExpressionProcessor<language::plus_op, TinyVector<2ul, double>, TinyVector<2ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^3")
    {
      std::string_view data = R"(
let x : R^3, x = (1,2,3);
let y : R^3, y = (4,3,2);
x+y;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<3ul, double> >)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::integer:1:ValueProcessor)
 |       +-(language::integer:2:ValueProcessor)
 |       `-(language::integer:3:ValueProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<3ul, double> >)
 |   +-(language::name:y:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::integer:4:ValueProcessor)
 |       +-(language::integer:3:ValueProcessor)
 |       `-(language::integer:2:ValueProcessor)
 `-(language::plus_op:BinaryExpressionProcessor<language::plus_op, TinyVector<3ul, double>, TinyVector<3ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string concatenate bool")
    {
      std::string_view data = R"(
"foo"+true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:ConcatExpressionProcessor<bool>)
     +-(language::literal:"foo":ValueProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string concatenate N")
    {
      std::string_view data = R"(
let n : N, n=0;
"foo"+n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, unsigned long, long>)
 |   +-(language::name:n:NameProcessor)
 |   `-(language::integer:0:ValueProcessor)
 `-(language::plus_op:ConcatExpressionProcessor<unsigned long>)
     +-(language::literal:"foo":ValueProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string concatenate Z")
    {
      std::string_view data = R"(
"foo"+1;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:ConcatExpressionProcessor<long>)
     +-(language::literal:"foo":ValueProcessor)
     `-(language::integer:1:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string concatenate R")
    {
      std::string_view data = R"(
"foo"+1.2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:ConcatExpressionProcessor<double>)
     +-(language::literal:"foo":ValueProcessor)
     `-(language::real:1.2:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string concatenate string")
    {
      std::string_view data = R"(
"foo"+"bar";
)";

      std::string string_name = demangle(typeid(std::string{}).name());

      std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:ConcatExpressionProcessor<)" +
                           string_name + R"( >)
     +-(language::literal:"foo":ValueProcessor)
     `-(language::literal:"bar":ValueProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("minus")
  {
    SECTION("B")
    {
      std::string_view data = R"(
let b : B;
b-true;
false-b-true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::minus_op:BinaryExpressionProcessor<language::minus_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:ValueProcessor)
 `-(language::minus_op:BinaryExpressionProcessor<language::minus_op, long, bool>)
     +-(language::minus_op:BinaryExpressionProcessor<language::minus_op, bool, bool>)
     |   +-(language::false_kw:ValueProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
let n : N;
let m : N;
n-m-n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::minus_op:BinaryExpressionProcessor<language::minus_op, unsigned long, unsigned long>)
     +-(language::minus_op:BinaryExpressionProcessor<language::minus_op, unsigned long, unsigned long>)
     |   +-(language::name:n:NameProcessor)
     |   `-(language::name:m:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
let a : Z;
a-3-a;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::minus_op:BinaryExpressionProcessor<language::minus_op, long, long>)
     +-(language::minus_op:BinaryExpressionProcessor<language::minus_op, long, long>)
     |   +-(language::name:a:NameProcessor)
     |   `-(language::integer:3:ValueProcessor)
     `-(language::name:a:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3-1.2-2-false;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::minus_op:BinaryExpressionProcessor<language::minus_op, double, bool>)
     +-(language::minus_op:BinaryExpressionProcessor<language::minus_op, double, long>)
     |   +-(language::minus_op:BinaryExpressionProcessor<language::minus_op, double, double>)
     |   |   +-(language::real:2.3:ValueProcessor)
     |   |   `-(language::real:1.2:ValueProcessor)
     |   `-(language::integer:2:ValueProcessor)
     `-(language::false_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^1")
    {
      std::string_view data = R"(
let x : R^1, x = 1;
let y : R^1, y = 2;
x-y;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, long>)
 |   +-(language::name:y:NameProcessor)
 |   `-(language::integer:2:ValueProcessor)
 `-(language::minus_op:BinaryExpressionProcessor<language::minus_op, TinyVector<1ul, double>, TinyVector<1ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^2")
    {
      std::string_view data = R"(
let x : R^2, x = (1,2);
let y : R^2, y = (0.3,0.7);
x-y;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<2ul, double> >)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::integer:1:ValueProcessor)
 |       `-(language::integer:2:ValueProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<2ul, double> >)
 |   +-(language::name:y:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::real:0.3:ValueProcessor)
 |       `-(language::real:0.7:ValueProcessor)
 `-(language::minus_op:BinaryExpressionProcessor<language::minus_op, TinyVector<2ul, double>, TinyVector<2ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^3")
    {
      std::string_view data = R"(
let x : R^3, x = (1,2,3);
let y : R^3, y = (4,3,2);
x-y;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<3ul, double> >)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::integer:1:ValueProcessor)
 |       +-(language::integer:2:ValueProcessor)
 |       `-(language::integer:3:ValueProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<3ul, double> >)
 |   +-(language::name:y:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::integer:4:ValueProcessor)
 |       +-(language::integer:3:ValueProcessor)
 |       `-(language::integer:2:ValueProcessor)
 `-(language::minus_op:BinaryExpressionProcessor<language::minus_op, TinyVector<3ul, double>, TinyVector<3ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("or")
  {
    SECTION("B")
    {
      std::string_view data = R"(
let b : B;
b or true;
false or b or true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::or_op:BinaryExpressionProcessor<language::or_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:ValueProcessor)
 `-(language::or_op:BinaryExpressionProcessor<language::or_op, bool, bool>)
     +-(language::or_op:BinaryExpressionProcessor<language::or_op, bool, bool>)
     |   +-(language::false_kw:ValueProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("and")
  {
    SECTION("B")
    {
      std::string_view data = R"(
let b : B;
b and true;
false and b and true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::and_op:BinaryExpressionProcessor<language::and_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:ValueProcessor)
 `-(language::and_op:BinaryExpressionProcessor<language::and_op, bool, bool>)
     +-(language::and_op:BinaryExpressionProcessor<language::and_op, bool, bool>)
     |   +-(language::false_kw:ValueProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("xor")
  {
    SECTION("B")
    {
      std::string_view data = R"(
let b : B;
b xor true;
false xor b xor true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::xor_op:BinaryExpressionProcessor<language::xor_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:ValueProcessor)
 `-(language::xor_op:BinaryExpressionProcessor<language::xor_op, bool, bool>)
     +-(language::xor_op:BinaryExpressionProcessor<language::xor_op, bool, bool>)
     |   +-(language::false_kw:ValueProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("greater")
  {
    SECTION("B")
    {
      std::string_view data = R"(
let b : B;
b > true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_op:BinaryExpressionProcessor<language::greater_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
let n : N;
let m : N;
n > m;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_op:BinaryExpressionProcessor<language::greater_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
let a : Z;
a > 3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_op:BinaryExpressionProcessor<language::greater_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3 > 1.2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_op:BinaryExpressionProcessor<language::greater_op, double, double>)
     +-(language::real:2.3:ValueProcessor)
     `-(language::real:1.2:ValueProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("lesser")
  {
    SECTION("B")
    {
      std::string_view data = R"(
let b : B;
b < true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_op:BinaryExpressionProcessor<language::lesser_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
let n : N;
let m : N;
n < m;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_op:BinaryExpressionProcessor<language::lesser_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
let a : Z;
a < 3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_op:BinaryExpressionProcessor<language::lesser_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3 < 1.2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_op:BinaryExpressionProcessor<language::lesser_op, double, double>)
     +-(language::real:2.3:ValueProcessor)
     `-(language::real:1.2:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("greater or equal")
    {
      SECTION("B")
      {
        std::string_view data = R"(
let b : B;
b >= true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_or_eq_op:BinaryExpressionProcessor<language::greater_or_eq_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N")
      {
        std::string_view data = R"(
let n : N;
let m : N;
n >= m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_or_eq_op:BinaryExpressionProcessor<language::greater_or_eq_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z")
      {
        std::string_view data = R"(
let a : Z;
a >= 3;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_or_eq_op:BinaryExpressionProcessor<language::greater_or_eq_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R")
      {
        std::string_view data = R"(
2.3 >= 1.2;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_or_eq_op:BinaryExpressionProcessor<language::greater_or_eq_op, double, double>)
     +-(language::real:2.3:ValueProcessor)
     `-(language::real:1.2:ValueProcessor)
)";

        CHECK_AST(data, result);
      }
    }

    SECTION("lesser or equal")
    {
      SECTION("B")
      {
        std::string_view data = R"(
let b : B;
b <= true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_or_eq_op:BinaryExpressionProcessor<language::lesser_or_eq_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N")
      {
        std::string_view data = R"(
let n : N;
let m : N;
n <= m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_or_eq_op:BinaryExpressionProcessor<language::lesser_or_eq_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z")
      {
        std::string_view data = R"(
let a : Z;
a <= 3;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_or_eq_op:BinaryExpressionProcessor<language::lesser_or_eq_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R")
      {
        std::string_view data = R"(
2.3 <= 1.2;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_or_eq_op:BinaryExpressionProcessor<language::lesser_or_eq_op, double, double>)
     +-(language::real:2.3:ValueProcessor)
     `-(language::real:1.2:ValueProcessor)
)";

        CHECK_AST(data, result);
      }
    }

    SECTION("equal equal")
    {
      SECTION("B")
      {
        std::string_view data = R"(
let b : B;
b == true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N")
      {
        std::string_view data = R"(
let n : N;
let m : N;
n == m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z")
      {
        std::string_view data = R"(
let a : Z;
a == 3;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R")
      {
        std::string_view data = R"(
2.3 == 1.2;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, double, double>)
     +-(language::real:2.3:ValueProcessor)
     `-(language::real:1.2:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("string == string")
      {
        std::string_view data = R"(
"foo" == "bar";
)";

        std::string string_name = demangle(typeid(std::string{}).name());

        std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, )" +
                             string_name + ", " + string_name + R"( >)
     +-(language::literal:"foo":ValueProcessor)
     `-(language::literal:"bar":ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^1")
      {
        std::string_view data = R"(
let x : R^1, x = 1;
let y : R^1, y = 2;
x==y;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, long>)
 |   +-(language::name:y:NameProcessor)
 |   `-(language::integer:2:ValueProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, TinyVector<1ul, double>, TinyVector<1ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^2")
      {
        std::string_view data = R"(
let x : R^2, x = (1,2);
let y : R^2, y = (0.3,0.7);
x==y;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<2ul, double> >)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::integer:1:ValueProcessor)
 |       `-(language::integer:2:ValueProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<2ul, double> >)
 |   +-(language::name:y:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::real:0.3:ValueProcessor)
 |       `-(language::real:0.7:ValueProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, TinyVector<2ul, double>, TinyVector<2ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^3")
      {
        std::string_view data = R"(
let x : R^3, x = (1,2,3);
let y : R^3, y = (4,3,2);
x==y;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<3ul, double> >)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::integer:1:ValueProcessor)
 |       +-(language::integer:2:ValueProcessor)
 |       `-(language::integer:3:ValueProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<3ul, double> >)
 |   +-(language::name:y:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::integer:4:ValueProcessor)
 |       +-(language::integer:3:ValueProcessor)
 |       `-(language::integer:2:ValueProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, TinyVector<3ul, double>, TinyVector<3ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

        CHECK_AST(data, result);
      }
    }

    SECTION("not equal")
    {
      SECTION("B")
      {
        std::string_view data = R"(
let b : B;
b != true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N")
      {
        std::string_view data = R"(
let n : N;
let m : N;
n != m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z")
      {
        std::string_view data = R"(
let a : Z;
a != 3;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R")
      {
        std::string_view data = R"(
2.3 != 1.2;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, double, double>)
     +-(language::real:2.3:ValueProcessor)
     `-(language::real:1.2:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^1")
      {
        std::string_view data = R"(
let x : R^1, x = 1;
let y : R^1, y = 2;
x!=y;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, long>)
 |   +-(language::name:y:NameProcessor)
 |   `-(language::integer:2:ValueProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, TinyVector<1ul, double>, TinyVector<1ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^2")
      {
        std::string_view data = R"(
let x : R^2, x = (1,2);
let y : R^2, y = (0.3,0.7);
x!=y;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<2ul, double> >)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::integer:1:ValueProcessor)
 |       `-(language::integer:2:ValueProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<2ul, double> >)
 |   +-(language::name:y:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::real:0.3:ValueProcessor)
 |       `-(language::real:0.7:ValueProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, TinyVector<2ul, double>, TinyVector<2ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^3")
      {
        std::string_view data = R"(
let x : R^3, x = (1,2,3);
let y : R^3, y = (4,3,2);
x!=y;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<3ul, double> >)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::integer:1:ValueProcessor)
 |       +-(language::integer:2:ValueProcessor)
 |       `-(language::integer:3:ValueProcessor)
 +-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<3ul, double> >)
 |   +-(language::name:y:NameProcessor)
 |   `-(language::expression_list:ASTNodeExpressionListProcessor)
 |       +-(language::integer:4:ValueProcessor)
 |       +-(language::integer:3:ValueProcessor)
 |       `-(language::integer:2:ValueProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, TinyVector<3ul, double>, TinyVector<3ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("string != string")
      {
        std::string_view data = R"(
"foo" != "bar";
)";

        std::string string_name = demangle(typeid(std::string{}).name());

        std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, )" +
                             string_name + ", " + string_name + R"( >)
     +-(language::literal:"foo":ValueProcessor)
     `-(language::literal:"bar":ValueProcessor)
)";

        CHECK_AST(data, result);
      }
    }
  }

  SECTION("Errors")
  {
    SECTION("Invalid binary operator type")
    {
      auto ast = std::make_unique<ASTNode>();

      REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "unexpected error: undefined binary operator");
    }

    SECTION("Invalid string binary operators")
    {
      SECTION("lhs bad multiply")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::multiply_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string multiply")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::multiply_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string multiply")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::multiply_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad divide")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::divide_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string divide")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::divide_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string divide")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::divide_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad plus")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::plus_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string plus bad rhs")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::plus_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::void_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string plus")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::plus_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad minus")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::minus_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string minus")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::minus_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string minus")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::minus_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad or")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::or_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string or")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::or_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string or")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::or_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad and")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::and_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string and")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::and_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string and")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::and_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad xor")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::xor_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string xor")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::xor_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string xor")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::xor_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad >")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::greater_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string >")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::greater_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string >")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::greater_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad <")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::lesser_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string <")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::lesser_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string <")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::lesser_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad >=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::greater_or_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string >=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::greater_or_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string >=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::greater_or_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad <=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::lesser_or_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string <=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::lesser_or_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string <=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::lesser_or_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad ==")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::eqeq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string ==")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::eqeq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string ==")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::eqeq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad !=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::not_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string ==")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::not_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string ==")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::not_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }
    }

    SECTION("invalid R^d operators")
    {
      SECTION("invalid operator R^1 > R^1")
      {
        std::string_view data = R"(
let x : R^1, x = 0;
let y : R^1, y = 0;
x > y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^1 >= R^1")
      {
        std::string_view data = R"(
let x : R^1, x = 0;
let y : R^1, y = 0;
x >= y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^1 < R^1")
      {
        std::string_view data = R"(
let x : R^1, x = 0;
let y : R^1, y = 0;
x < y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^1 <= R^1")
      {
        std::string_view data = R"(
let x : R^1, x = 0;
let y : R^1, y = 1;
x <= y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^1 * R^1")
      {
        std::string_view data = R"(
let x : R^1, x = 0;
let y : R^1, y = 0;
x * y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^1 / R^1")
      {
        std::string_view data = R"(
let x : R^1, x = 0;
let y : R^1, y = 0;
x / y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^2 > R^2")
      {
        std::string_view data = R"(
let x : R^2, x = 0;
let y : R^2, y = 0;
x > y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^2 >= R^2")
      {
        std::string_view data = R"(
let x : R^2, x = 0;
let y : R^2, y = 0;
x >= y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^2 < R^2")
      {
        std::string_view data = R"(
let x : R^2, x = 0;
let y : R^2, y = 0;
x < y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^2 <= R^2")
      {
        std::string_view data = R"(
let x : R^2, x = 0;
let y : R^2, y = 0;
x <= y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^2 * R^2")
      {
        std::string_view data = R"(
let x : R^2, x = 0;
let y : R^2, y = 0;
x * y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^2 / R^2")
      {
        std::string_view data = R"(
let x : R^2, x = 0;
let y : R^2, y = 0;
x / y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^3 > R^3")
      {
        std::string_view data = R"(
let x : R^3, x = 0;
let y : R^3, y = 0;
x > y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^3 >= R^3")
      {
        std::string_view data = R"(
let x : R^3, x = 0;
let y : R^3, y = 0;
x >= y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^3 < R^3")
      {
        std::string_view data = R"(
let x : R^3, x = 0;
let y : R^3, y = 0;
x < y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^3 <= R^3")
      {
        std::string_view data = R"(
let x : R^3, x = 0;
let y : R^3, y = 0;
x <= y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^3 * R^3")
      {
        std::string_view data = R"(
let x : R^3, x = 0;
let y : R^3, y = 0;
x * y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }

      SECTION("invalid operator R^3 / R^3")
      {
        std::string_view data = R"(
let x : R^3, x = 0;
let y : R^3, y = 0;
x / y;
)";

        REQUIRE_AST_THROWS_WITH(data, "invalid binary operator");
      }
    }

    SECTION("invalid operand types")
    {
      SECTION("incompatible operand dimensions")
      {
        std::string_view data = R"(
let x : R^3, x = 0;
let y : R^1, y = 0;
x + y;
)";

        REQUIRE_AST_THROWS_WITH(data, "incompatible dimensions of operands");
      }

      SECTION("incompatible operand dimensions")
      {
        std::string_view data = R"(
let x : R^1, x = 0;
let y : R^2, y = 0;
x - y;
)";

        REQUIRE_AST_THROWS_WITH(data, "incompatible dimensions of operands");
      }

      SECTION("incompatible operand dimensions")
      {
        std::string_view data = R"(
let x : R^3, x = 0;
let y : R^2, y = 0;
x == y;
)";

        REQUIRE_AST_THROWS_WITH(data, "incompatible dimensions of operands");
      }

      SECTION("incompatible operand dimensions")
      {
        std::string_view data = R"(
let x : R^1, x = 0;
let y : R^2, y = 0;
x != y;
)";

        REQUIRE_AST_THROWS_WITH(data, "incompatible dimensions of operands");
      }
    }
  }
}
