#include <catch2/catch.hpp>

#include <language/ast/ASTBuilder.hpp>
#include <language/ast/ASTNodeAffectationExpressionBuilder.hpp>
#include <language/ast/ASTNodeDataTypeBuilder.hpp>
#include <language/ast/ASTNodeDeclarationToAffectationConverter.hpp>
#include <language/ast/ASTNodeExpressionBuilder.hpp>
#include <language/ast/ASTNodeTypeCleaner.hpp>
#include <language/ast/ASTSymbolInitializationChecker.hpp>
#include <language/ast/ASTSymbolTableBuilder.hpp>
#include <language/utils/ASTPrinter.hpp>
#include <utils/Demangle.hpp>

#include <pegtl/string_input.hpp>

#define CHECK_AST(data, expected_output)                                                            \
  {                                                                                                 \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>);                  \
    static_assert(std::is_same_v<std::decay_t<decltype(expected_output)>, std::string_view> or      \
                  std::is_same_v<std::decay_t<decltype(expected_output)>, std::string>);            \
                                                                                                    \
    string_input input{data, "test.pgs"};                                                           \
    auto ast = ASTBuilder::build(input);                                                            \
                                                                                                    \
    ASTSymbolTableBuilder{*ast};                                                                    \
    ASTNodeDataTypeBuilder{*ast};                                                                   \
                                                                                                    \
    ASTNodeDeclarationToAffectationConverter{*ast};                                                 \
    ASTNodeTypeCleaner<language::var_declaration>{*ast};                                            \
                                                                                                    \
    ASTNodeExpressionBuilder{*ast};                                                                 \
                                                                                                    \
    std::stringstream ast_output;                                                                   \
    ast_output << '\n' << ASTPrinter{*ast, ASTPrinter::Format::raw, {ASTPrinter::Info::exec_type}}; \
                                                                                                    \
    REQUIRE(ast_output.str() == expected_output);                                                   \
  }

#define CHECK_AST_THROWS_WITH(data, expected_error)                                           \
  {                                                                                           \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>);            \
    static_assert(std::is_same_v<std::decay_t<decltype(expected_error)>, std::string_view> or \
                  std::is_same_v<std::decay_t<decltype(expected_error)>, std::string>);       \
                                                                                              \
    string_input input{data, "test.pgs"};                                                     \
    auto ast = ASTBuilder::build(input);                                                      \
                                                                                              \
    ASTSymbolTableBuilder{*ast};                                                              \
    ASTNodeDataTypeBuilder{*ast};                                                             \
                                                                                              \
    ASTNodeDeclarationToAffectationConverter{*ast};                                           \
    ASTNodeTypeCleaner<language::var_declaration>{*ast};                                      \
                                                                                              \
    REQUIRE_THROWS_WITH(ASTNodeExpressionBuilder{*ast}, expected_error);                      \
  }

// clazy:excludeall=non-pod-global-static

TEST_CASE("ASTNodeAffectationExpressionBuilder", "[language]")
{
  const std::string demangled_stdstring = demangle(typeid(std::string{}).name());

  SECTION("Affectations")
  {
    SECTION("boolean affectation")
    {
      SECTION("B <- B")
      {
        std::string_view data = R"(
let b:B, b=true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

        CHECK_AST(data, result);
      }
    }

    SECTION("unsigned integer affectation")
    {
      SECTION("N <- B")
      {
        std::string_view data = R"(
let n:N, n=true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, unsigned long, bool>)
     +-(language::name:n:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N <- N")
      {
        std::string_view data = R"(
let m : N; let n:N, n=m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N <- Z")
      {
        std::string_view data = R"(
let z:Z; let n :N, n=z;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, unsigned long, long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:z:NameProcessor)
)";

        CHECK_AST(data, result);
      }
    }

    SECTION("integer affectation")
    {
      SECTION("Z <- B")
      {
        std::string_view data = R"(
let z : Z, z=true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, long, bool>)
     +-(language::name:z:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z <- N")
      {
        std::string_view data = R"(
let m : N; let z : Z, z=m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, long, unsigned long>)
     +-(language::name:z:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z <- Z")
      {
        std::string_view data = R"(
let q : Z; let z : Z, z=q;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, long, long>)
     +-(language::name:z:NameProcessor)
     `-(language::name:q:NameProcessor)
)";

        CHECK_AST(data, result);
      }
    }

    SECTION("double affectation")
    {
      SECTION("R <- B")
      {
        std::string_view data = R"(
let r : R, r=true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, double, bool>)
     +-(language::name:r:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R <- N")
      {
        std::string_view data = R"(
let m : N; let r : R, r=m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, double, unsigned long>)
     +-(language::name:r:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R <- Z")
      {
        std::string_view data = R"(
let z : Z; let r : R, r=z;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
     +-(language::name:r:NameProcessor)
     `-(language::name:z:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R <- R")
      {
        std::string_view data = R"(
let s : R; let r : R, r=s;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, double, double>)
     +-(language::name:r:NameProcessor)
     `-(language::name:s:NameProcessor)
)";

        CHECK_AST(data, result);
      }
    }

    SECTION("R^d affectation")
    {
      SECTION("R^1 <- R^1")
      {
        std::string_view data = R"(
let x : R^1;
let y : R^1, y = x;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, TinyVector<1ul, double> >)
     +-(language::name:y:NameProcessor)
     `-(language::name:x:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^1 <- R")
      {
        std::string_view data = R"(
let x : R^1, x = 1.3;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, double>)
     +-(language::name:x:NameProcessor)
     `-(language::real:1.3:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^1 <- Z")
      {
        std::string_view data = R"(
let x : R^1, x = -1;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, long>)
     +-(language::name:x:NameProcessor)
     `-(language::unary_minus:UnaryExpressionProcessor<language::unary_minus, long, long>)
         `-(language::integer:1:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^1 <- N")
      {
        std::string_view data = R"(
let n : N;
let x : R^1, x = n;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, unsigned long>)
     +-(language::name:x:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^1 <- B")
      {
        std::string_view data = R"(
let b : B;
let x : R^1, x = b;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<1ul, double>, bool>)
     +-(language::name:x:NameProcessor)
     `-(language::name:b:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^1 <- 0")
      {
        std::string_view data = R"(
let x : R^1, x = 0;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationFromZeroProcessor<TinyVector<1ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::integer:0:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^2 <- R^2")
      {
        std::string_view data = R"(
let x : R^2;
let y : R^2, y = x;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<2ul, double>, TinyVector<2ul, double> >)
     +-(language::name:y:NameProcessor)
     `-(language::name:x:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^2 <- (.,.)")
      {
        std::string_view data = R"(
let y : R^2, y = (0,1);
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<2ul, double> >)
     +-(language::name:y:NameProcessor)
     `-(language::expression_list:ASTNodeExpressionListProcessor)
         +-(language::integer:0:ValueProcessor)
         `-(language::integer:1:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^2 <- 0")
      {
        std::string_view data = R"(
let x : R^2, x = 0;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationFromZeroProcessor<TinyVector<2ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::integer:0:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^3 <- R^3")
      {
        std::string_view data = R"(
let x : R^3;
let y : R^3, y = x;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, TinyVector<3ul, double>, TinyVector<3ul, double> >)
     +-(language::name:y:NameProcessor)
     `-(language::name:x:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^3 <- 0")
      {
        std::string_view data = R"(
let x : R^3, x = 0;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationFromZeroProcessor<TinyVector<3ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::integer:0:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R^3 <- (.,.)")
      {
        std::string_view data = R"(
let y : R^3, y = (1,2,3);
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationToTinyVectorFromListProcessor<language::eq_op, TinyVector<3ul, double> >)
     +-(language::name:y:NameProcessor)
     `-(language::expression_list:ASTNodeExpressionListProcessor)
         +-(language::integer:1:ValueProcessor)
         +-(language::integer:2:ValueProcessor)
         `-(language::integer:3:ValueProcessor)
)";

        CHECK_AST(data, result);
      }
    }

    SECTION("string affectation")
    {
      SECTION("string <- B")
      {
        std::string_view data = R"(
let s : string, s=true;
)";

        std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, )" +
                             demangled_stdstring + R"(, bool>)
     +-(language::name:s:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("string <- N")
      {
        std::string_view data = R"(
let n : N; let s : string, s=n;
)";

        std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, )" +
                             demangled_stdstring + R"(, unsigned long>)
     +-(language::name:s:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("string <- Z")
      {
        std::string_view data = R"(
let z : Z; let s : string, s=z;
)";

        std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, )" +
                             demangled_stdstring +
                             R"(, long>)
     +-(language::name:s:NameProcessor)
     `-(language::name:z:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("string <- R")
      {
        std::string_view data = R"(
let r : R; let s : string, s=r;
)";

        std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, )" +
                             demangled_stdstring +
                             R"(, double>)
     +-(language::name:s:NameProcessor)
     `-(language::name:r:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("string <- string")
      {
        std::string_view data = R"(
let s : string, s="foo";
)";

        std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, )" +
                             demangled_stdstring + ", " + demangled_stdstring + R"( >)
     +-(language::name:s:NameProcessor)
     `-(language::literal:"foo":ValueProcessor)
)";

        CHECK_AST(data, result);
      }
    }
  }

  SECTION("+=")
  {
    SECTION("N += N")
    {
      std::string_view data = R"(
let n : N, n=1; n+=n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, unsigned long, long>)
 |   +-(language::name:n:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 `-(language::pluseq_op:AffectationProcessor<language::pluseq_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R += N")
    {
      std::string_view data = R"(
let x : R, x=1; x+=2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 `-(language::pluseq_op:AffectationProcessor<language::pluseq_op, double, long>)
     +-(language::name:x:NameProcessor)
     `-(language::integer:2:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string += N")
    {
      std::string_view data = R"(
let s : string, s="foo"; s+=2;
)";

      std::string result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, )" +
                           demangled_stdstring + ", " + demangled_stdstring + R"( >)
 |   +-(language::name:s:NameProcessor)
 |   `-(language::literal:"foo":ValueProcessor)
 `-(language::pluseq_op:AffectationProcessor<language::pluseq_op, )" +
                           demangled_stdstring +
                           R"(, long>)
     +-(language::name:s:NameProcessor)
     `-(language::integer:2:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^1 += R^1")
    {
      std::string_view data = R"(
let x : R^1;
let y : R^1;
x += y;
)";

      std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::pluseq_op:AffectationProcessor<language::pluseq_op, TinyVector<1ul, double>, TinyVector<1ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^2 += R^2")
    {
      std::string_view data = R"(
let x : R^2;
let y : R^2;
x += y;
)";

      std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::pluseq_op:AffectationProcessor<language::pluseq_op, TinyVector<2ul, double>, TinyVector<2ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^3 += R^3")
    {
      std::string_view data = R"(
let x : R^3;
let y : R^3;
x += y;
)";

      std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::pluseq_op:AffectationProcessor<language::pluseq_op, TinyVector<3ul, double>, TinyVector<3ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("-=")
  {
    SECTION("Z -= Z")
    {
      std::string_view data = R"(
let z : Z, z=1; z-=2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, long, long>)
 |   +-(language::name:z:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 `-(language::minuseq_op:AffectationProcessor<language::minuseq_op, long, long>)
     +-(language::name:z:NameProcessor)
     `-(language::integer:2:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R -= R")
    {
      std::string_view data = R"(
let x : R, x=1; x-=2.3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 `-(language::minuseq_op:AffectationProcessor<language::minuseq_op, double, double>)
     +-(language::name:x:NameProcessor)
     `-(language::real:2.3:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^1 -= R^1")
    {
      std::string_view data = R"(
let x : R^1;
let y : R^1;
x -= y;
)";

      std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::minuseq_op:AffectationProcessor<language::minuseq_op, TinyVector<1ul, double>, TinyVector<1ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^2 -= R^2")
    {
      std::string_view data = R"(
let x : R^2;
let y : R^2;
x -= y;
)";

      std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::minuseq_op:AffectationProcessor<language::minuseq_op, TinyVector<2ul, double>, TinyVector<2ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^3 -= R^3")
    {
      std::string_view data = R"(
let x : R^3;
let y : R^3;
x -= y;
)";

      std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::minuseq_op:AffectationProcessor<language::minuseq_op, TinyVector<3ul, double>, TinyVector<3ul, double> >)
     +-(language::name:x:NameProcessor)
     `-(language::name:y:NameProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("*=")
  {
    SECTION("Z *= Z")
    {
      std::string_view data = R"(
let z : Z, z=1; z*=2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, long, long>)
 |   +-(language::name:z:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, long, long>)
     +-(language::name:z:NameProcessor)
     `-(language::integer:2:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R *= R")
    {
      std::string_view data = R"(
let x : R, x=1; x*=2.3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, double, double>)
     +-(language::name:x:NameProcessor)
     `-(language::real:2.3:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^1 *= R")
    {
      std::string_view data = R"(
let x : R^1; x*=2.3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, TinyVector<1ul, double>, double>)
     +-(language::name:x:NameProcessor)
     `-(language::real:2.3:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^2 *= R")
    {
      std::string_view data = R"(
let x : R^2; x*= 6.2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, TinyVector<2ul, double>, double>)
     +-(language::name:x:NameProcessor)
     `-(language::real:6.2:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^3 *= R")
    {
      std::string_view data = R"(
let x : R^3; x*= 3.1;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, TinyVector<3ul, double>, double>)
     +-(language::name:x:NameProcessor)
     `-(language::real:3.1:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R *= Z")
    {
      std::string_view data = R"(
let x : R, x=1; x*=2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, double, long>)
     +-(language::name:x:NameProcessor)
     `-(language::integer:2:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^1 *= Z")
    {
      std::string_view data = R"(
let x : R^1; x *= 3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, TinyVector<1ul, double>, long>)
     +-(language::name:x:NameProcessor)
     `-(language::integer:3:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^2 *= Z")
    {
      std::string_view data = R"(
let x : R^2; x *= 6;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, TinyVector<2ul, double>, long>)
     +-(language::name:x:NameProcessor)
     `-(language::integer:6:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^3 *= Z")
    {
      std::string_view data = R"(
let x : R^3; x *= 4;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, TinyVector<3ul, double>, long>)
     +-(language::name:x:NameProcessor)
     `-(language::integer:4:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R *= N")
    {
      std::string_view data = R"(
let n : N, n=2; let x : R, x=1; x *= n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, unsigned long, long>)
 |   +-(language::name:n:NameProcessor)
 |   `-(language::integer:2:ValueProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, double, unsigned long>)
     +-(language::name:x:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^1 *= N")
    {
      std::string_view data = R"(
let n : N;
let x : R^1; x *= n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, TinyVector<1ul, double>, unsigned long>)
     +-(language::name:x:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^2 *= N")
    {
      std::string_view data = R"(
let n : N;
let x : R^2; x *= n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, TinyVector<2ul, double>, unsigned long>)
     +-(language::name:x:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^3 *= N")
    {
      std::string_view data = R"(
let n : N;
let x : R^3; x *= n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, TinyVector<3ul, double>, unsigned long>)
     +-(language::name:x:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R *= B")
    {
      std::string_view data = R"(
let x : R, x=1; x *= true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, double, bool>)
     +-(language::name:x:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^1 *= B")
    {
      std::string_view data = R"(
let x : R^1; x *= true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, TinyVector<1ul, double>, bool>)
     +-(language::name:x:NameProcessor)
     `-(language::true_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^2 *= B")
    {
      std::string_view data = R"(
let x : R^2; x *= false;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, TinyVector<2ul, double>, bool>)
     +-(language::name:x:NameProcessor)
     `-(language::false_kw:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R^3 *= B")
    {
      std::string_view data = R"(
let b : B; let x : R^3; x *= b;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, TinyVector<3ul, double>, bool>)
     +-(language::name:x:NameProcessor)
     `-(language::name:b:NameProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("/=")
  {
    SECTION("Z /= Z")
    {
      std::string_view data = R"(
let z : Z, z=6; z/=2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, long, long>)
 |   +-(language::name:z:NameProcessor)
 |   `-(language::integer:6:ValueProcessor)
 `-(language::divideeq_op:AffectationProcessor<language::divideeq_op, long, long>)
     +-(language::name:z:NameProcessor)
     `-(language::integer:2:ValueProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R /= R")
    {
      std::string_view data = R"(
let x : R, x=1; x/=2.3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:ValueProcessor)
 `-(language::divideeq_op:AffectationProcessor<language::divideeq_op, double, double>)
     +-(language::name:x:NameProcessor)
     `-(language::real:2.3:ValueProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("Errors")
  {
    SECTION("Invalid affectation operator")
    {
      auto ast = std::make_unique<ASTNode>();
      REQUIRE_THROWS_WITH(ASTNodeAffectationExpressionBuilder{*ast},
                          "unexpected error: undefined affectation operator");
    }

    SECTION("Invalid lhs")
    {
      auto ast = std::make_unique<ASTNode>();
      ast->set_type<language::eq_op>();
      ast->children.emplace_back(std::make_unique<ASTNode>());
      ast->children.emplace_back(std::make_unique<ASTNode>());
      REQUIRE_THROWS_WITH(ASTNodeAffectationExpressionBuilder{*ast},
                          "unexpected error: undefined value type for affectation");
    }

    SECTION("Invalid rhs")
    {
      auto ast = std::make_unique<ASTNode>();
      ast->set_type<language::eq_op>();
      ast->m_data_type = ASTNodeDataType::int_t;

      ast->children.emplace_back(std::make_unique<ASTNode>());
      ast->children.emplace_back(std::make_unique<ASTNode>());
      REQUIRE_THROWS_WITH(ASTNodeAffectationExpressionBuilder{*ast},
                          "unexpected error: invalid implicit conversion: undefined -> Z");
    }

    SECTION("Invalid string rhs")
    {
      auto ast = std::make_unique<ASTNode>();
      ast->set_type<language::eq_op>();
      ast->m_data_type = ASTNodeDataType::string_t;

      ast->children.emplace_back(std::make_unique<ASTNode>());
      ast->children.emplace_back(std::make_unique<ASTNode>());
      REQUIRE_THROWS_WITH(ASTNodeAffectationExpressionBuilder{*ast},
                          "unexpected error: invalid implicit conversion: undefined -> string");
    }

    SECTION("Invalid string affectation operator")
    {
      SECTION("string -= string")
      {
        std::string_view data = R"(
let s : string, s="foo"; s-="bar";
)";

        std::string error_message = "invalid operator for string affectation";

        CHECK_AST_THROWS_WITH(data, error_message);
      }

      SECTION("string *= Z")
      {
        std::string_view data = R"(
let s : string, s="foo"; s*=2;
)";

        std::string error_message = "invalid operator for string affectation";

        CHECK_AST_THROWS_WITH(data, error_message);
      }

      SECTION("string /= string")
      {
        std::string_view data = R"(
let s : string, s="foo"; s/="bar";
)";

        std::string error_message = "invalid operator for string affectation";

        CHECK_AST_THROWS_WITH(data, error_message);
      }
    }

    SECTION("Invalid R^n -> R^m affectation")
    {
      SECTION("R^3 <- R^1")
      {
        std::string_view data = R"(
let x : R^3; let y : R^1; x = y;
)";

        std::string error_message = "incompatible dimensions in affectation";

        CHECK_AST_THROWS_WITH(data, error_message);
      }

      SECTION("R^3 <- R^2")
      {
        std::string_view data = R"(
let x : R^3; let y : R^2; x = y;
)";

        std::string error_message = "incompatible dimensions in affectation";

        CHECK_AST_THROWS_WITH(data, error_message);
      }

      SECTION("R^2 <- R^1")
      {
        std::string_view data = R"(
let x : R^2; let y : R^1; x = y;
)";

        std::string error_message = "incompatible dimensions in affectation";

        CHECK_AST_THROWS_WITH(data, error_message);
      }

      SECTION("R^2 <- R^3")
      {
        std::string_view data = R"(
let x : R^2; let y : R^3; x = y;
)";

        std::string error_message = "incompatible dimensions in affectation";

        CHECK_AST_THROWS_WITH(data, error_message);
      }

      SECTION("R^1 <- R^2")
      {
        std::string_view data = R"(
let x : R^1; let y : R^2; x = y;
)";

        std::string error_message = "incompatible dimensions in affectation";

        CHECK_AST_THROWS_WITH(data, error_message);
      }

      SECTION("R^1 <- R^3")
      {
        std::string_view data = R"(
let x : R^1; let y : R^2; x = y;
)";

        std::string error_message = "incompatible dimensions in affectation";

        CHECK_AST_THROWS_WITH(data, error_message);
      }
    }

    SECTION("Invalid Z -> R^m affectation [non-zero]")
    {
      SECTION("R^3 <- Z")
      {
        std::string_view data = R"(
let x : R^3, x = 3;
)";

        std::string error_message = "invalid implicit conversion: Z -> R^3";

        CHECK_AST_THROWS_WITH(data, error_message);
      }

      SECTION("R^2 <- Z")
      {
        std::string_view data = R"(
let x : R^2, x = 2;
)";

        std::string error_message = "invalid implicit conversion: Z -> R^2";

        CHECK_AST_THROWS_WITH(data, error_message);
      }
    }

    SECTION("Invalid R^d -> R^d affectation operator")
    {
      SECTION("R^3 <- R^3")
      {
        std::string_view data = R"(
let x : R^3; let y : R^3; x /= y;
)";

        std::string error_message = "invalid affectation operator for R^3";

        CHECK_AST_THROWS_WITH(data, error_message);
      }

      SECTION("R^2 <- R^2")
      {
        std::string_view data = R"(
let x : R^2; let y : R^2; x /= y;
)";

        std::string error_message = "invalid affectation operator for R^2";

        CHECK_AST_THROWS_WITH(data, error_message);
      }

      SECTION("R^1 <- R^1")
      {
        std::string_view data = R"(
let x : R^1; let y : R^1; x /= y;
)";

        std::string error_message = "invalid affectation operator for R^1";

        CHECK_AST_THROWS_WITH(data, error_message);
      }
    }

    SECTION("Invalid R^d -> R^d *= operand")
    {
      SECTION("R^3 <- R^3")
      {
        std::string_view data = R"(
let x : R^3; let y : R^3; x *= y;
)";

        std::string error_message = "expecting scalar operand type";

        CHECK_AST_THROWS_WITH(data, error_message);
      }

      SECTION("R^2 <- R^2")
      {
        std::string_view data = R"(
let x : R^2; let y : R^2; x *= y;
)";

        std::string error_message = "expecting scalar operand type";

        CHECK_AST_THROWS_WITH(data, error_message);
      }

      SECTION("R^1 <- R^1")
      {
        std::string_view data = R"(
let x : R^1; let y : R^1; x *= y;
)";

        std::string error_message = "expecting scalar operand type";

        CHECK_AST_THROWS_WITH(data, error_message);
      }
    }

    SECTION("incorrect declarative/definition number of symbols")
    {
      std::string_view data = R"(
let (x,y,z):R*R*R, (x,y) = (2,3);
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      ASTSymbolTableBuilder{*ast};
      REQUIRE_THROWS_WITH(ASTSymbolInitializationChecker{*ast},
                          std::string{"invalid number of definition identifiers, expecting 3 found 2"});
    }

    SECTION("incorrect identifier/expression number of symbols")
    {
      std::string_view data = R"(
let (x,y,z):R*R*R, (x,y,z) = (2,3);
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      ASTSymbolTableBuilder{*ast};
      REQUIRE_THROWS_WITH(ASTSymbolInitializationChecker{*ast},
                          std::string{"invalid number of definition expressions, expecting 3 found 2"});
    }

    SECTION("incorrect identifier/expression number of symbols")
    {
      std::string_view data = R"(
let y:R;
let x:R, (x,y) = (2,3);
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      ASTSymbolTableBuilder{*ast};
      REQUIRE_THROWS_WITH(ASTSymbolInitializationChecker{*ast},
                          std::string{"unexpected variable list, expecting one identifier"});
    }

    SECTION("incorrect definition variable identifier")
    {
      std::string_view data = R"(
let y:R;
let x:R, y = 3;
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      ASTSymbolTableBuilder{*ast};
      REQUIRE_THROWS_WITH(ASTSymbolInitializationChecker{*ast}, std::string{"invalid identifier, expecting 'x'"});
    }

    SECTION("invalid definition variable identifier order")
    {
      std::string_view data = R"(
let (x,y):R, (y,x) = (3,2);
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      ASTSymbolTableBuilder{*ast};
      REQUIRE_THROWS_WITH(ASTSymbolInitializationChecker{*ast}, std::string{"invalid identifier, expecting 'x'"});
    }
  }
}
