#ifndef BOUNDARY_CONDITION_HANDLER_HPP
#define BOUNDARY_CONDITION_HANDLER_HPP

#include <memory>
#include <vector>

#include <Array.hpp>

#include <MeshNodeBoundary.hpp>
#include <RefItemList.hpp>

class BoundaryCondition
{
 public:
  enum Type
  {
    velocity,
    normal_velocity,
    pressure,
    symmetry
  };

  const Type m_type;

 protected:
  BoundaryCondition(const Type& type) : m_type(type)
  {
    ;
  }

 public:
  const Type&
  type() const
  {
    return m_type;
  }

  virtual ~BoundaryCondition() = default;
};

class PressureBoundaryCondition : public BoundaryCondition
{
 private:
  const double m_value;

  const size_t m_number_of_faces;
  Array<const unsigned int> m_face_list;

 public:
  double
  value() const
  {
    return m_value;
  }

  const size_t&
  numberOfFaces() const
  {
    return m_number_of_faces;
  }

  const Array<const unsigned int>&
  faceList() const
  {
    return m_face_list;
  }

  PressureBoundaryCondition(const double& value, const std::vector<unsigned int>& faces)
    : BoundaryCondition(BoundaryCondition::pressure), m_value(value), m_number_of_faces(faces.size())
  {
    Array<unsigned int> face_list(faces.size());
    parallel_for(
      m_number_of_faces, PUGS_LAMBDA(const int& f) { face_list[f] = faces[f]; });
    m_face_list = face_list;
  }

  ~PressureBoundaryCondition() = default;
};

template <size_t dimension>
class SymmetryBoundaryCondition : public BoundaryCondition
{
 public:
  using Rd = TinyVector<dimension, double>;

 private:
  const MeshFlatNodeBoundary<dimension> m_mesh_flat_node_boundary;

 public:
  const Rd&
  outgoingNormal() const
  {
    return m_mesh_flat_node_boundary.outgoingNormal();
  }

  size_t
  numberOfNodes() const
  {
    return m_mesh_flat_node_boundary.nodeList().size();
  }

  const Array<const NodeId>&
  nodeList() const
  {
    return m_mesh_flat_node_boundary.nodeList();
  }

  SymmetryBoundaryCondition(const MeshFlatNodeBoundary<dimension>& mesh_flat_node_boundary)
    : BoundaryCondition(BoundaryCondition::symmetry), m_mesh_flat_node_boundary(mesh_flat_node_boundary)
  {
    ;
  }

  ~SymmetryBoundaryCondition() = default;
};

class BoundaryConditionHandler
{
 private:
  std::shared_ptr<BoundaryCondition> m_boundary_condition;

 public:
  const BoundaryCondition&
  boundaryCondition() const
  {
    return *m_boundary_condition;
  }

  PUGS_INLINE
  BoundaryConditionHandler& operator=(BoundaryConditionHandler&) = default;

  PUGS_INLINE
  BoundaryConditionHandler(const BoundaryConditionHandler&) = default;

  PUGS_INLINE
  BoundaryConditionHandler(BoundaryConditionHandler&&) = default;

  PUGS_INLINE
  BoundaryConditionHandler(std::shared_ptr<BoundaryCondition> boundary_condition)
    : m_boundary_condition(boundary_condition)
  {}

  ~BoundaryConditionHandler() = default;
};

#endif   // BOUNDARY_CONDITION_HANDLER_HPP
