#ifndef CONNECTIVITY_UTILS_HPP
#define CONNECTIVITY_UTILS_HPP

#include <map>

class ConnectivityUtils
{
public:
  void computeNodeCellConnectivity(const Kokkos::View<const unsigned int**> cell_nodes,
				   const Kokkos::View<const unsigned short*> cell_nb_nodes,
				   const size_t& number_of_cells,
				   size_t& max_nb_node_per_cell,
				   size_t& number_of_nodes,
				   Kokkos::View<unsigned short*>& node_nb_cells,
				   Kokkos::View<unsigned int**>& node_cells,
				   Kokkos::View<unsigned short**>& node_cell_local_node)
  {
    // Computes inefficiently node->cells connectivity [Version 0]
    std::multimap<unsigned int, unsigned int> node_cells_map;
    using namespace Kokkos::Experimental;
    Kokkos::parallel_reduce(number_of_cells, KOKKOS_LAMBDA(const int& j, size_t& nb_max) {
	const size_t n = cell_nb_nodes[j];
	if (n > nb_max) nb_max = n;
      }, Max<size_t>(max_nb_node_per_cell));

    for (unsigned int j=0; j<number_of_cells; ++j) {
      for (unsigned int r=0; r<cell_nb_nodes[j]; ++r) {
	node_cells_map.insert(std::make_pair(cell_nodes(j,r),j));
      }
    }

    std::vector<unsigned int> node_ids;
    node_ids.reserve(node_cells_map.size());
    for (const auto& node_cell: node_cells_map) {
      node_ids.push_back(node_cell.first);
    }
    auto last_unique = std::unique(node_ids.begin(), node_ids.end());
    node_ids.resize(std::distance(node_ids.begin(), last_unique));

    number_of_nodes = node_ids.size();

    if ((node_ids[0] != 0) or (node_ids[node_ids.size()-1] != node_ids.size()-1)) {
      std::cerr << "sparse node numerotation NIY\n";
      for (size_t i=0; i<node_ids.size(); ++i) {
	std::cout << "node_ids[" << i << "] = " << node_ids[i] << '\n';
      }
      std::exit(0);
    }

    std::vector<std::vector<unsigned int>> node_cells_vector(node_ids.size());
    for (const auto& node_cell: node_cells_map) {
      node_cells_vector[node_cell.first].push_back(node_cell.second);
    }

    node_nb_cells = Kokkos::View<unsigned short*>("node_nb_cells", node_ids.size());
    size_t max_node_cells = 0;
    for (size_t i=0; i<node_cells_vector.size(); ++i) {
      const auto& cells_vector = node_cells_vector[i];
      const size_t nb_cells = cells_vector.size();
      node_nb_cells[i] = nb_cells;
      if (nb_cells > max_node_cells) {
	max_node_cells = nb_cells;
      }
    }

    node_cells = Kokkos::View<unsigned int**>("node_cells", node_ids.size(), max_node_cells);
    for (size_t i=0; i<node_cells_vector.size(); ++i) {
      const auto& cells_vector = node_cells_vector[i];
      for (size_t j=0; j<cells_vector.size(); ++j) {
	node_cells(i,j) = cells_vector[j];
      }
    }

    node_cell_local_node = Kokkos::View<unsigned short**>("node_cell_local_node",
							  node_ids.size(), max_node_cells);
    Kokkos::parallel_for(number_of_nodes, KOKKOS_LAMBDA(const unsigned int& r){
	for (unsigned short J=0; J<node_nb_cells[r]; ++J) {
	  const unsigned int j = node_cells(r,J);
	  for (unsigned int R=0; R<cell_nb_nodes[j]; ++R) {
	    if (cell_nodes(j,R) == r) {
	      node_cell_local_node(r,J)=R;
	      break;
	    }
	  }
	}
      });
  }
};

#endif // CONNECTIVITY_UTILS_HPP
