#include <mesh/DualMeshManager.hpp>

#include <mesh/Connectivity.hpp>
#include <mesh/DiamondDualMeshBuilder.hpp>
#include <mesh/Dual1DMeshBuilder.hpp>
#include <mesh/MedianDualMeshBuilder.hpp>
#include <mesh/Mesh.hpp>
#include <utils/Exceptions.hpp>
#include <utils/PugsAssert.hpp>

#include <sstream>

DualMeshManager* DualMeshManager::m_instance{nullptr};

void
DualMeshManager::create()
{
  Assert(m_instance == nullptr, "DualMeshManager is already created");
  m_instance = new DualMeshManager;
}

void
DualMeshManager::destroy()
{
  Assert(m_instance != nullptr, "DualMeshManager was not created!");

  if (m_instance->m_mesh_to_dual_mesh_map.size() > 0) {
    // LCOV_EXCL_START
    std::stringstream error;
    error << ": some meshes are still registered\n";
    for (const auto& [key, parent_mesh] : m_instance->m_mesh_to_dual_mesh_map) {
      error << " - mesh " << rang::fgB::magenta << key.second << rang::style::reset << ": " << name(key.first)
            << " dual mesh of " << rang::fgB::yellow << parent_mesh.get() << rang::style::reset << '\n';
    }
    throw UnexpectedError(error.str());
    // LCOV_EXCL_STOP
  }
  delete m_instance;
  m_instance = nullptr;
}

void
DualMeshManager::deleteMesh(const IMesh* p_mesh)
{
  bool has_removed = false;
  do {
    has_removed = false;
    for (const auto& [key, dual_mesh] : m_mesh_to_dual_mesh_map) {
      const auto& [type, p_parent_mesh] = key;
      if (p_mesh == p_parent_mesh) {
        m_mesh_to_dual_mesh_map.erase(key);
        has_removed = true;
        break;
      }
    }
  } while (has_removed);
}

std::shared_ptr<const Mesh<Connectivity<1>>>
DualMeshManager::getDual1DMesh(const Mesh<Connectivity<1>>& mesh)
{
  const IMesh* p_mesh = &mesh;

  auto key = std::make_pair(DualMeshType::Dual1D, p_mesh);
  if (auto i_mesh_data = m_mesh_to_dual_mesh_map.find(key); i_mesh_data != m_mesh_to_dual_mesh_map.end()) {
    return std::dynamic_pointer_cast<const Mesh<Connectivity<1>>>(i_mesh_data->second);
  } else {
    Dual1DMeshBuilder builder{mesh};

    m_mesh_to_dual_mesh_map[key] = builder.mesh();
    return std::dynamic_pointer_cast<const Mesh<Connectivity<1>>>(builder.mesh());
  }
}

template <>
std::shared_ptr<const Mesh<Connectivity<1>>>
DualMeshManager::getMedianDualMesh(const Mesh<Connectivity<1>>& mesh)
{
  return this->getDual1DMesh(mesh);
}

template <size_t Dimension>
std::shared_ptr<const Mesh<Connectivity<Dimension>>>
DualMeshManager::getMedianDualMesh(const Mesh<Connectivity<Dimension>>& mesh)
{
  static_assert(Dimension > 1);
  const IMesh* p_mesh = &mesh;

  auto key = std::make_pair(DualMeshType::Median, p_mesh);
  if (auto i_mesh_data = m_mesh_to_dual_mesh_map.find(key); i_mesh_data != m_mesh_to_dual_mesh_map.end()) {
    return std::dynamic_pointer_cast<const Mesh<Connectivity<Dimension>>>(i_mesh_data->second);
  } else {
    MedianDualMeshBuilder builder{mesh};

    m_mesh_to_dual_mesh_map[key] = builder.mesh();
    return std::dynamic_pointer_cast<const Mesh<Connectivity<Dimension>>>(builder.mesh());
  }
}

template std::shared_ptr<const Mesh<Connectivity<2>>> DualMeshManager::getMedianDualMesh(const Mesh<Connectivity<2>>&);
template std::shared_ptr<const Mesh<Connectivity<3>>> DualMeshManager::getMedianDualMesh(const Mesh<Connectivity<3>>&);

template <>
std::shared_ptr<const Mesh<Connectivity<1>>>
DualMeshManager::getDiamondDualMesh(const Mesh<Connectivity<1>>& mesh)
{
  return this->getDual1DMesh(mesh);
}

template <size_t Dimension>
std::shared_ptr<const Mesh<Connectivity<Dimension>>>
DualMeshManager::getDiamondDualMesh(const Mesh<Connectivity<Dimension>>& mesh)
{
  static_assert(Dimension > 1);

  const IMesh* p_mesh = &mesh;

  auto key = std::make_pair(DualMeshType::Diamond, p_mesh);
  if (auto i_mesh_data = m_mesh_to_dual_mesh_map.find(key); i_mesh_data != m_mesh_to_dual_mesh_map.end()) {
    return std::dynamic_pointer_cast<const Mesh<Connectivity<Dimension>>>(i_mesh_data->second);
  } else {
    DiamondDualMeshBuilder builder{mesh};

    m_mesh_to_dual_mesh_map[key] = builder.mesh();
    return std::dynamic_pointer_cast<const Mesh<Connectivity<Dimension>>>(builder.mesh());
  }
}

template std::shared_ptr<const Mesh<Connectivity<2>>> DualMeshManager::getDiamondDualMesh(const Mesh<Connectivity<2>>&);
template std::shared_ptr<const Mesh<Connectivity<3>>> DualMeshManager::getDiamondDualMesh(const Mesh<Connectivity<3>>&);
