#include <scheme/DiscreteFunctionUtils.hpp>

#include <mesh/Connectivity.hpp>
#include <mesh/Mesh.hpp>
#include <mesh/MeshVariant.hpp>
#include <scheme/DiscreteFunctionP0.hpp>
#include <scheme/DiscreteFunctionVariant.hpp>
#include <utils/Stringify.hpp>

std::shared_ptr<const MeshVariant>
getCommonMesh(const std::vector<std::shared_ptr<const DiscreteFunctionVariant>>& discrete_function_variant_list)
{
  std::optional<size_t> mesh_id;
  std::shared_ptr<const MeshVariant> mesh_v;
  bool is_same_mesh = true;
  for (const auto& discrete_function_variant : discrete_function_variant_list) {
    std::visit(
      [&](auto&& discrete_function) {
        if (not mesh_id.has_value()) {
          mesh_v  = std::make_shared<MeshVariant>(discrete_function.mesh());
          mesh_id = discrete_function.mesh()->id();
        } else {
          if (mesh_id != discrete_function.mesh()->id()) {
            is_same_mesh = false;
            mesh_v.reset();
          }
        }
      },
      discrete_function_variant->discreteFunction());
  }

  return mesh_v;
}

bool
hasSameMesh(const std::vector<std::shared_ptr<const DiscreteFunctionVariant>>& discrete_function_variant_list)
{
  std::optional<size_t> mesh_id;

  bool same_mesh = true;
  for (const auto& discrete_function_variant : discrete_function_variant_list) {
    std::visit(
      [&](auto&& discrete_function) {
        if (not mesh_id.has_value()) {
          mesh_id = discrete_function.mesh()->id();
        } else {
          if (mesh_id != discrete_function.mesh()->id()) {
            same_mesh = false;
          }
        }
      },
      discrete_function_variant->discreteFunction());
  }

  return same_mesh;
}

bool
hasSameMesh(const std::vector<std::shared_ptr<const DiscreteFunctionVariant>>& discrete_function_variant_list,
            const std::shared_ptr<const MeshVariant>& mesh_v)
{
  const size_t mesh_id = mesh_v->id();

  bool same_mesh = true;
  for (const auto& discrete_function_variant : discrete_function_variant_list) {
    std::visit(
      [&](auto&& discrete_function) {
        if (mesh_id != discrete_function.mesh()->id()) {
          same_mesh = false;
        }
      },
      discrete_function_variant->discreteFunction());
  }

  return same_mesh;
}

template <typename MeshType, typename DiscreteFunctionT>
std::shared_ptr<const DiscreteFunctionVariant>
shallowCopy(const std::shared_ptr<const MeshType>& mesh, const DiscreteFunctionT& f)
{
  const size_t function_connectivity_id = f.mesh()->shared_connectivity()->id();

  if (mesh->shared_connectivity()->id() != function_connectivity_id) {
    throw NormalError("cannot shallow copy when connectivity changes");
  }

  if constexpr (std::is_same_v<MeshType, typename DiscreteFunctionT::MeshType>) {
    if constexpr (is_discrete_function_P0_v<DiscreteFunctionT>) {
      return std::make_shared<DiscreteFunctionVariant>(DiscreteFunctionT(mesh, f.cellValues()));
    } else if constexpr (is_discrete_function_P0_vector_v<DiscreteFunctionT>) {
      return std::make_shared<DiscreteFunctionVariant>(DiscreteFunctionT(mesh, f.cellArrays()));
    } else {
      throw UnexpectedError("invalid discrete function type");
    }
  } else {
    throw UnexpectedError("invalid mesh types");
  }
}

std::shared_ptr<const DiscreteFunctionVariant>
shallowCopy(const std::shared_ptr<const MeshVariant>& mesh_v,
            const std::shared_ptr<const DiscreteFunctionVariant>& discrete_function_variant)
{
  return std::visit(
    [&](auto&& f) {
      const size_t mesh_id        = std::visit([](auto&& mesh) { return mesh->id(); }, mesh_v->meshPointer());
      const size_t mesh_dimension = std::visit([](auto&& mesh) { return mesh->dimension(); }, mesh_v->meshPointer());
      if (mesh_id == f.mesh()->id()) {
        return discrete_function_variant;
      } else if (mesh_dimension != f.mesh()->dimension()) {
        throw NormalError("incompatible mesh dimensions");
      }

      return std::visit([&](auto&& mesh) { return shallowCopy(mesh, f); }, mesh_v->meshPointer());
    },
    discrete_function_variant->discreteFunction());
}
