#pragma once
#include <utils/pugs_config.hpp>

#ifdef PUGS_HAS_COSTO

#include <language/utils/InterpolateItemValue.hpp>
#include <mesh/Connectivity.hpp>
#include <mesh/ItemValueVariant.hpp>
#include <mesh/Mesh.hpp>
#include <mesh/MeshData.hpp>
#include <mesh/MeshDataManager.hpp>
#include <mesh/MeshFaceBoundary.hpp>
#include <mesh/MeshFlatFaceBoundary.hpp>
#include <mesh/MeshNodeBoundary.hpp>
#include <mesh/MeshNodeInterface.hpp>
#include <scheme/DirichletBoundaryConditionDescriptor.hpp>
#include <scheme/DiscreteFunctionUtils.hpp>
#include <scheme/NeumannBoundaryConditionDescriptor.hpp>
#include <scheme/SymmetryBoundaryConditionDescriptor.hpp>
#include <string>
#include <type_traits>
#include <utils/FPEManager.hpp>
#include <utils/Messenger.hpp>
#include <utils/PugsAssert.hpp>
#include <utils/PugsMacros.hpp>
#include <utils/pugs_config.hpp>
#include <vector>

#ifdef PUGS_HAS_MPI
#include <mpi.h>
#endif   // PUGS_HAS_MPI

template <size_t Dimension>
class Serializer
{
 private:
  using MeshType     = Mesh<Connectivity<Dimension>>;
  using Rd           = TinyVector<Dimension>;
  using MeshDataType = MeshData<Dimension>;

 public:
  enum locator : int
  {
    at_node,
    at_face,
    at_cell_center,
  };
  Serializer()  = default;
  ~Serializer() = default;
  void mesh(std::shared_ptr<const IMesh> i_mesh, const int64_t dest, const int64_t tag);
  void
  BC(const std::shared_ptr<const IBoundaryDescriptor>& boundary,
     std::shared_ptr<const IMesh> i_mesh,
     const int location,
     const int64_t dest,
     const int64_t tag)
  {
    this->BC(*boundary, i_mesh, location, dest, tag);
  }

  void BC(const IBoundaryDescriptor& boundary,
          std::shared_ptr<const IMesh> i_mesh,
          const int location,
          const int64_t dest,
          const int64_t tag);

  void BC_field(std::shared_ptr<const IMesh> i_mesh,
                const std::shared_ptr<const IBoundaryDescriptor>& boundary,
                const std::shared_ptr<const ItemValueVariant>& field,
                const int64_t dest,
                const int64_t tag);

  void interfaces(std::shared_ptr<const IMesh> i_mesh,
                  const std::vector<std::shared_ptr<const IInterfaceDescriptor>>& i_interface_list,
                  const int64_t dest,
                  const int64_t tag);

  void field(const std::shared_ptr<const DiscreteFunctionVariant>& field, const int64_t dest, const int64_t tag);

  std::shared_ptr<const IMesh> change_mesh_position(std::shared_ptr<const IMesh> i_mesh,
                                                    const int64_t src,
                                                    const int64_t tag);
};

// MESH node position
template <size_t Dimension>
void
Serializer<Dimension>::mesh(std::shared_ptr<const IMesh> i_mesh, const int64_t dest, const int64_t tag)
{
  auto costo = parallel::Messenger::getInstance().myCoupling;
  std::vector<int> shape;
  std::vector<double> pts;

  const std::shared_ptr mesh    = std::dynamic_pointer_cast<const MeshType>(i_mesh);
  const NodeValue<const Rd>& xr = mesh->xr();
  Array<TinyVector<Dimension>> positions(mesh->numberOfNodes());
  parallel_for(
    mesh->numberOfNodes(), PUGS_LAMBDA(NodeId r) {
      for (unsigned short i = 0; i < Dimension; ++i) {
        positions[r][i] = xr[r][i];
      }
    });

  /*TODO: serializer directement position pour eviter une copie*/
  pts.resize(mesh->numberOfNodes() * Dimension);
  for (unsigned short r = 0; r < mesh->numberOfNodes(); ++r) {
    for (unsigned short j = 0; j < Dimension; ++j) {
      pts[Dimension * r + j] = positions[r][j];
    }
  }

  shape.resize(3);
  shape[0] = mesh->numberOfNodes();
  shape[1] = Dimension;
  shape[2] = 0;
  costo->sendData(shape, pts, dest, tag);
  /* const int src  = 1; */
  /* std::vector<int> recv; */
  /* parallel::Messenger::getInstance().myCoupling->recvData(recv, src, tag + 1); */
}

// BC nodes position
template <size_t Dimension>
void
Serializer<Dimension>::BC(const IBoundaryDescriptor& boundary,
                          std::shared_ptr<const IMesh> i_mesh,
                          const int location,
                          const int64_t dest,
                          const int64_t tag)
{
  auto costo = parallel::Messenger::getInstance().myCoupling;
  std::vector<int> shape;
  std::vector<double> pts;
  std::cout << "\033[01;31m"
            << "L" << location << " ff " << at_face << "\033[00;00m" << std::endl;

  if (location == at_face) {
    std::cout << "\033[01;31m"
              << "ICI AT FACE"
              << "\033[00;00m" << std::endl;
    const std::shared_ptr mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);

    MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(*mesh);
    /* const ItemValue<const Rd>& xl = mesh_data.xl(); */
    const auto xl = mesh_data.xl();

    Array<TinyVector<Dimension>> fpositions(mesh->numberOfFaces());
    parallel_for(
      mesh->numberOfFaces(), PUGS_LAMBDA(FaceId l) {
        for (unsigned short i = 0; i < Dimension; ++i) {
          fpositions[l][i] = xl[l][i];
        }
        /* for (unsigned short i = Dimension; i < 3; ++i) { */
        /*   positions[r][i] = 0; */
        /* } */
      });

    /*TODO: serializer directement fposition pour eviter une copie*/
    MeshFaceBoundary<Dimension> mesh_face_boundary = getMeshFaceBoundary(*mesh, boundary);
    /* mesh_face_boundary.faceList() */
    const auto face_list = mesh_face_boundary.faceList();
    pts.resize(face_list.size() * Dimension);

    for (size_t i_face = 0; i_face < face_list.size(); ++i_face) {
      const FaceId face_id = face_list[i_face];
      for (unsigned short j = 0; j < Dimension; ++j) {
        pts[Dimension * i_face + j] = fpositions[face_id][j];
      }
      /* std::cout << "\033[01;31m" << pts[2 * r] << "; " << pts[2 * r + 1] << "\033[00;00m" << std::endl; */
    }
    shape.resize(3);
    shape[0] = face_list.size();
    shape[1] = Dimension;
    shape[2] = 0;
    costo->sendData(shape, pts, dest, tag);
  } else if (location == at_node) {
    std::cout << "\033[01;31m"
              << "ICI AT node"
              << "\033[00;00m" << std::endl;
    const std::shared_ptr mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);

    /* const ItemValue<const Rd>& xl = mesh_data.xl(); */
    const auto xr = mesh->xr();

    Array<TinyVector<Dimension>> positions(mesh->numberOfNodes());
    parallel_for(
      mesh->numberOfNodes(), PUGS_LAMBDA(NodeId r) {
        for (unsigned short i = 0; i < Dimension; ++i) {
          positions[r][i] = xr[r][i];
        }
        /* for (unsigned short i = Dimension; i < 3; ++i) { */
        /*   positions[r][i] = 0; */
        /* } */
      });

    /*TODO: serializer directement position pour eviter une copie*/
    MeshNodeBoundary<Dimension> mesh_node_boundary = getMeshNodeBoundary(*mesh, boundary);
    /* mesh_face_boundary.faceList() */
    const auto node_list = mesh_node_boundary.nodeList();
    pts.resize(node_list.size() * Dimension);

    for (size_t i_node = 0; i_node < node_list.size(); ++i_node) {
      const NodeId node_id = node_list[i_node];
      for (unsigned short j = 0; j < Dimension; ++j) {
        pts[Dimension * i_node + j] = positions[node_id][j];
      }
      /* std::cout << "\033[01;31m" << pts[2 * r] << "; " << pts[2 * r + 1] << "\033[00;00m" << std::endl; */
    }
    shape.resize(3);
    shape[0] = node_list.size();
    shape[1] = Dimension;
    shape[2] = 0;
    costo->sendData(shape, pts, dest, tag);
  } else {
    throw UnexpectedError(" location must be 0 or 1");
  }

  /* const int src  = 1; */
  /* std::vector<int> recv; */
  /* parallel::Messenger::getInstance().myCoupling->recvData(recv, src, tag + 1); */
}

template <size_t Dimension>
void
Serializer<Dimension>::interfaces(std::shared_ptr<const IMesh> i_mesh,
                                  const std::vector<std::shared_ptr<const IInterfaceDescriptor>>& i_interface_list,
                                  const int64_t dest,
                                  const int64_t tag)
{
  auto costo = parallel::Messenger::getInstance().myCoupling;
  /* const int64_t dest = costo->myGlobalSize() - 1; */
  std::vector<int> shape;
  std::vector<double> data;

  const std::shared_ptr p_mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);

  for (const auto& i_interface : i_interface_list) {
    auto node_interface = getMeshNodeInterface(*p_mesh, *i_interface);
    auto node_list      = node_interface.nodeList();

    data.resize(node_list.size() * Dimension);

    const auto xr = p_mesh->xr();

    Array<TinyVector<Dimension>> positions(p_mesh->numberOfNodes());
    parallel_for(
      p_mesh->numberOfNodes(), PUGS_LAMBDA(NodeId r) {
        for (unsigned short i = 0; i < Dimension; ++i) {
          positions[r][i] = xr[r][i];
        }
        /* for (unsigned short i = Dimension; i < 3; ++i) { */
        /*   positions[r][i] = 0; */
        /* } */
      });

    /*TODO: serializer directement positions pour eviter une copie */
    for (size_t i_node = 0; i_node < node_list.size(); ++i_node) {
      const NodeId node_id = node_list[i_node];
      for (unsigned short i_dim = 0; i_dim < Dimension; ++i_dim) {
        data[i_node * Dimension + i_dim] = positions[node_id][i_dim];
      }
    }

    shape.resize(3);
    shape[0] = node_list.size();
    shape[1] = Dimension;
    shape[2] = 0;
    costo->sendData(shape, data, dest, tag);
  }
}

// Field at node
template <size_t Dimension>
void
Serializer<Dimension>::BC_field(std::shared_ptr<const IMesh> i_mesh,
                                const std::shared_ptr<const IBoundaryDescriptor>& boundary,
                                const std::shared_ptr<const ItemValueVariant>& field,
                                const int64_t dest,
                                const int64_t tag)
{
  auto costo = parallel::Messenger::getInstance().myCoupling;
  /* const int64_t dest = costo->myGlobalSize() - 1; */
  std::vector<int> shape;
  std::vector<double> data;

  /* std::cout << "\033[01;31m" */
  /*           << "ival" << ival.type() << "\033[00;00m" << std::endl; */
  /* std::cout << "\033[01;31m" << ival << "\033[00;00m" << std::endl; */
  /* const auto test = field->mesh(); */
  /* const std::shared_ptr i_mesh           = getCommonMesh({field, field}); */

  const std::shared_ptr mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);

  if (not i_mesh) {
    throw NormalError("primal discrete functions are not defined on the same mesh");
  }
  const NodeValue<const TinyVector<Dimension>> nField = field->get<NodeValue<const TinyVector<Dimension>>>();
  /* assert(MeshType(i_mesh) == field->mesh()); */

  MeshNodeBoundary<Dimension> mesh_node_boundary = getMeshNodeBoundary(*mesh, *boundary);

  const auto node_list = mesh_node_boundary.nodeList();
  data.resize(node_list.size() * Dimension);

  for (size_t i_node = 0; i_node < node_list.size(); ++i_node) {
    const NodeId node_id = node_list[i_node];
    for (unsigned short i_dim = 0; i_dim < Dimension; ++i_dim) {
      data[i_node * Dimension + i_dim] = nField[node_id][i_dim];
    }
  }

  shape.resize(3);
  shape[0] = node_list.size();
  shape[1] = Dimension;
  shape[2] = 0;
  costo->sendData(shape, data, dest, tag);
}

// Field (faces)
template <size_t Dimension>
void
Serializer<Dimension>::field(const std::shared_ptr<const DiscreteFunctionVariant>& field,
                             const int64_t dest,
                             const int64_t tag)
{
  const std::shared_ptr i_mesh = getCommonMesh({field});
  if (not i_mesh) {
    throw NormalError("primal discrete functions are not defined on the same mesh");
  }
  /* assert(MeshType(i_mesh) == field->mesh()); */
  const std::shared_ptr mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);
  /* const IBoundaryDescriptor& bd = *boundary; */
  /* const Mesh<Connectivity<2>> mt = <const MeshType>(i_mesh); */

  /* const auto& n_list = MeshNodeBoundary<2>(mt, bd); */

  const NodeValue<const Rd>& xr = mesh->xr();
  Array<TinyVector<Dimension>> positions(mesh->numberOfNodes());
  parallel_for(
    mesh->numberOfNodes(), PUGS_LAMBDA(NodeId r) {
      for (unsigned short i = 0; i < Dimension; ++i) {
        positions[r][i] = xr[r][i];
      }
      /* for (unsigned short i = Dimension; i < 3; ++i) { */
      /*   positions[r][i] = 0; */
      /* } */
    });

  std::vector<double> pts;
  pts.resize(mesh->numberOfNodes() * Dimension);
  for (unsigned short r = 0; r < mesh->numberOfNodes(); ++r) {
    for (unsigned short j = 0; j < Dimension; ++j) {
      pts[Dimension * r + j] = positions[r][j];
    }
    /* std::cout << "\033[01;31m" << pts[2 * r] << "; " << pts[2 * r + 1] << "\033[00;00m" << std::endl; */
  }

  auto costo = parallel::Messenger::getInstance().myCoupling;
  /* const int64_t dest = costo->myGlobalSize() - 1; */
  std::vector<int> shape;
  shape.resize(3);
  shape[0] = mesh->numberOfNodes();
  shape[1] = Dimension;
  shape[2] = 0;
  costo->sendData(shape, pts, dest, tag);

  std::vector<int> recv;
  const int src = costo->myGlobalSize() - 1;

  costo->recvData(recv, src, tag + 1);
}

template <size_t Dimension>
std::shared_ptr<const IMesh>
Serializer<Dimension>::change_mesh_position(std::shared_ptr<const IMesh> i_mesh, const int64_t src, const int64_t tag)
{
  std::vector<double> recv_xr;
  parallel::Messenger::getInstance().myCoupling->recvData(recv_xr, src, tag);

  const std::shared_ptr given_mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);

  NodeValue<Rd> new_xr{given_mesh->connectivity()};
  parallel_for(
    given_mesh->numberOfNodes(), PUGS_LAMBDA(const NodeId node_id) {
      for (unsigned short j = 0; j < Dimension; ++j) {
        /* new_xr[node_id][j] = xr[node_id][j]; */
        new_xr[node_id][j] = recv_xr[Dimension * node_id + j];
      }
    });

  return std::make_shared<MeshType>(given_mesh->shared_connectivity(), new_xr);
}

using Serializer3D = Serializer<3>;
using Serializer2D = Serializer<2>;
using Serializer1D = Serializer<1>;

#endif   // PUGS_HAS_COSTO
