#pragma once
#include <algebra/TinyMatrix.hpp>
#include <algebra/TinyVector.hpp>
#include <mesh/IBoundaryDescriptor.hpp>
#include <mesh/ItemValueUtils.hpp>
#include <mesh/ItemValueVariant.hpp>
#include <mesh/Mesh.hpp>
#include <mesh/MeshNodeBoundary.hpp>
#include <mesh/SubItemValuePerItemVariant.hpp>
#include <scheme/CouplingBoundaryConditionDescriptor.hpp>
#include <scheme/IBoundaryConditionDescriptor.hpp>
#include <utils/Serializer.hpp>
#include <utils/pugs_config.hpp>
#include <vector>
#ifdef PUGS_HAS_COSTO

class IMesh;

template <size_t Dimension>
class CouplingData
{
 private:
  using MeshType = Mesh<Connectivity<Dimension>>;
  using Rd       = TinyVector<Dimension>;
  class CouplingBoundaryCondition;

  using BoundaryCondition1     = std::variant<CouplingBoundaryCondition>;
  using BoundaryConditionList1 = std::vector<CouplingBoundaryCondition>;

 public:
  NodeValue<Rd> CR_ur;
  NodeValuePerCell<Rd> CR_Fjr;

  ~CouplingData() = default;

  static auto&
  getInstanceOrBuildIt()
  {
    static CouplingData<Dimension> cdata;
    return cdata;
  }
  std::tuple<const std::shared_ptr<const ItemValueVariant>,
             const std::shared_ptr<const SubItemValuePerItemVariant>,
             const std::shared_ptr<const ItemValueVariant>,
             const std::shared_ptr<const SubItemValuePerItemVariant>>
  get_flux_and_velocity(const std::shared_ptr<const IMesh>& i_mesh,
                        const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list,
                        const std::shared_ptr<const ItemValueVariant>& ur,
                        const std::shared_ptr<const SubItemValuePerItemVariant>& Fjr,
                        const std::shared_ptr<const ItemValueVariant>& ur_saved,
                        const std::shared_ptr<const SubItemValuePerItemVariant>& Fjr_saved);

  std::shared_ptr<const IMesh> update_mesh(const std::shared_ptr<const IMesh>& i_mesh,
                                           const std::shared_ptr<const IBoundaryDescriptor>& boundary,
                                           const int64_t src,
                                           const int64_t tag);

  BoundaryConditionList1
  _getBCList(const MeshType& mesh,
             const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list) const
  {
    BoundaryConditionList1 coupling_bc_list;
    for (const auto& bc_descriptor : bc_descriptor_list) {
      switch (bc_descriptor->type()) {
      case IBoundaryConditionDescriptor::Type::coupling: {
        const CouplingBoundaryConditionDescriptor& coupling_bc_descriptor =
          dynamic_cast<const CouplingBoundaryConditionDescriptor&>(*bc_descriptor);
        coupling_bc_list.emplace_back(
          CouplingBoundaryCondition(getMeshNodeBoundary(mesh, bc_descriptor->boundaryDescriptor())));
        break;
      }
      default: {
        break;
      }
      }
    }
    return coupling_bc_list;
  }

 private:
  CouplingData();
};

template <size_t Dimension>
class CouplingData<Dimension>::CouplingBoundaryCondition
{
 private:
  const MeshNodeBoundary<Dimension> m_mesh_node_boundary;

 public:
  const Array<const NodeId>&
  nodeList() const
  {
    return m_mesh_node_boundary.nodeList();
  }

  CouplingBoundaryCondition(const MeshNodeBoundary<Dimension>& mesh_node_boundary)
    : m_mesh_node_boundary{mesh_node_boundary}
  {
    ;
  }

  ~CouplingBoundaryCondition() = default;
};

template <size_t Dimension>
std::shared_ptr<const IMesh>
CouplingData<Dimension>::update_mesh(const std::shared_ptr<const IMesh>& i_mesh,
                                     const std::shared_ptr<const IBoundaryDescriptor>& boundary,
                                     const int64_t src,
                                     const int64_t tag)
{
  auto costo = parallel::Messenger::getInstance().myCoupling;
  auto rank  = costo->myGlobalRank();

  /* const int src = costo->myGlobalSize() - 1; */
  std::vector<double> recv;

  costo->recvData(recv, src, tag);

  /* const std::shared_ptr given_mesh               = std::dynamic_pointer_cast<const MeshType>(*i_mesh); */
  const MeshType& mesh                           = dynamic_cast<const MeshType&>(*i_mesh);
  MeshNodeBoundary<Dimension> mesh_node_boundary = getMeshNodeBoundary(mesh, *boundary);
  const auto node_list                           = mesh_node_boundary.nodeList();

  NodeValue<Rd> new_xr = copy(mesh.xr());
  const size_t& n      = node_list.size();

  const auto& node_to_cell_matrix = mesh.connectivity().nodeToCellMatrix();

  for (size_t i = 0; i < n; i++) {
    const NodeId node_id     = node_list[i];
    const auto& node_to_cell = node_to_cell_matrix[node_id];
    /* std::cout << "\033[01;32m" */
    /*           << "rank_1: " << node_id << " " << node_to_cell.size() << "\033[00;00m" << std::endl; */
    for (unsigned short j = 0; j < Dimension; ++j) {
      new_xr[node_id][j] = recv[Dimension * i + j];
    }
  }
  return std::make_shared<MeshType>(mesh.shared_connectivity(), new_xr);
}

template <size_t Dimension>
std::tuple<const std::shared_ptr<const ItemValueVariant>,
           const std::shared_ptr<const SubItemValuePerItemVariant>,
           const std::shared_ptr<const ItemValueVariant>,
           const std::shared_ptr<const SubItemValuePerItemVariant>>
CouplingData<Dimension>::get_flux_and_velocity(
  const std::shared_ptr<const IMesh>& i_mesh,
  const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list,
  const std::shared_ptr<const ItemValueVariant>& ur,
  const std::shared_ptr<const SubItemValuePerItemVariant>& Fjr,
  const std::shared_ptr<const ItemValueVariant>& ur_saved,
  const std::shared_ptr<const SubItemValuePerItemVariant>& Fjr_saved)
{
  auto costo = parallel::Messenger::getInstance().myCoupling;
  auto rank  = costo->myGlobalRank();

  const MeshType& mesh = dynamic_cast<const MeshType&>(*i_mesh);

  auto coupling_bcs = this->_getBCList(mesh, bc_descriptor_list);
  auto ur_ok        = copy(ur->get<NodeValue<const Rd>>());
  auto Fjr_ok       = copy(Fjr->get<NodeValuePerCell<const Rd>>());

  for (const auto& boundary_condition : coupling_bcs) {
    const auto& node_list = boundary_condition.nodeList();
    if (costo->m == 1) {
      std::cout << "\033[01;33m"
                << "m=1"
                << "\033[00;00m" << std::endl;
      return std::make_tuple(std::make_shared<const ItemValueVariant>(ur_ok),
                             std::make_shared<const SubItemValuePerItemVariant>(Fjr_ok),
                             std::make_shared<const ItemValueVariant>(ur_ok),
                             std::make_shared<const SubItemValuePerItemVariant>(Fjr_ok));
    } else {
      costo->ctm += 1;
      if (costo->ctm == 1) {
        auto ur_saved_ok  = copy(ur->get<NodeValue<const Rd>>());
        auto Fjr_saved_ok = copy(Fjr->get<NodeValuePerCell<const Rd>>());
        std::cout << "\033[01;34m"
                  << "(" << rank << ")"
                  << "(" << costo->ctm << "|" << costo->m << ")"
                  << "(SAVE QTT)\n"
                  << "\033[00;00m" << std::endl;

        if (costo->ctm == costo->m) {
          costo->ctm = 0;
        }
        return std::make_tuple(std::make_shared<const ItemValueVariant>(ur_ok),
                               std::make_shared<const SubItemValuePerItemVariant>(Fjr_ok),
                               std::make_shared<const ItemValueVariant>(ur_saved_ok),
                               std::make_shared<const SubItemValuePerItemVariant>(Fjr_saved_ok));
      } else {
        std::cout << "\033[01;34m"
                  << "(" << rank << ")"
                  << "(" << costo->ctm << "|" << costo->m << ")"
                  << "(APPLIED QTT FLUX AND UR)\n"
                  << "\033[00;00m" << std::endl;

        auto ur_saved_ok  = copy(ur_saved->get<NodeValue<const Rd>>());
        auto Fjr_saved_ok = copy(Fjr_saved->get<NodeValuePerCell<const Rd>>());

        const size_t& n = node_list.size();

        const auto& node_to_cell_matrix               = mesh.connectivity().nodeToCellMatrix();
        const auto& node_local_numbers_in_their_cells = mesh.connectivity().nodeLocalNumbersInTheirCells();

        for (size_t i = 0; i < n; i++) {
          const NodeId node_id                       = node_list[i];
          const auto& node_to_cell                   = node_to_cell_matrix[node_id];
          const auto& node_local_number_in_its_cells = node_local_numbers_in_their_cells.itemArray(node_id);
          /* std::cout << "\033[01;32m" */
          /*           << "rank_1: " << node_id << " " << node_to_cell.size() << "\033[00;00m" << std::endl; */

          for (size_t j = 0; j < node_to_cell.size(); j++) {
            const CellId J       = node_to_cell[j];
            const unsigned int R = node_local_number_in_its_cells[j];
            Fjr_ok(J, R)         = Fjr_saved_ok(J, R);
          }
          ur_ok[node_id] = ur_saved_ok[node_id];
        }

        if (costo->ctm == costo->m) {
          /* Serializer<Dimension>().BC(std::make_shared<const IBoundaryDescriptor>( */
          /*                              bc_descriptor_list[0]->boundaryDescriptor()), */
          /*                            i_mesh, 0, 2, 800); */
          /* auto ptr = std::make_shared<const IBoundaryDescriptor>(bc_descriptor_list[0]->boundaryDescriptor()); */
          /* Serializer<Dimension>().BC(ptr, i_mesh, 0, 2, 800); */

          auto costo = parallel::Messenger::getInstance().myCoupling;
          /* std::vector<int> shape; */
          /* std::vector<double> pts; */
          /* const std::shared_ptr mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh); */

          /* /\* const ItemValue<const Rd>& xl = mesh_data.xl(); *\/ */
          /* const auto xr = mesh->xr(); */

          /* Array<TinyVector<Dimension>> positions(mesh->numberOfNodes()); */
          /* parallel_for( */
          /*   mesh->numberOfNodes(), PUGS_LAMBDA(NodeId r) { */
          /*     for (unsigned short i = 0; i < Dimension; ++i) { */
          /*       positions[r][i] = xr[r][i]; */
          /*     } */
          /*     /\* for (unsigned short i = Dimension; i < 3; ++i) { *\/ */
          /*     /\*   positions[r][i] = 0; *\/ */
          /*     /\* } *\/ */
          /*   }); */

          /* /\*TODO: serializer directement position pour eviter une copie*\/ */   //
          /* MeshNodeBoundary<Dimension> mesh_node_boundary = */
          /*   getMeshNodeBoundary(*mesh, bc_descriptor_list[0]->boundaryDescriptor()); */
          /* const auto node_list = mesh_node_boundary.nodeList(); */
          /* pts.resize(node_list.size() * Dimension); */

          /* for (size_t i_node = 0; i_node < node_list.size(); ++i_node) { */
          /*   const NodeId node_id = node_list[i_node]; */
          /*   for (unsigned short j = 0; j < Dimension; ++j) { */
          /*     pts[Dimension * i_node + j] = positions[node_id][j]; */
          /*   } */
          /*   /\* std::cout << "\033[01;31m" << pts[2 * r] << "; " << pts[2 * r + 1] << "\033[00;00m" << std::endl;
           * *\/
           */
          /* } */
          /* shape.resize(3); */
          /* shape[0] = node_list.size(); */
          /* shape[1] = Dimension; */
          /* shape[2] = 0; */
          /* costo->sendData(shape, pts, 2, 800); */
          /* auto ptr = std::make_shared<const IBoundaryDescriptor>(bc_descriptor_list[0]->boundaryDescriptor()); */

          Serializer<Dimension>().BC(bc_descriptor_list[0]->boundaryDescriptor(), i_mesh, 0, 2, 800);
          costo->ctm = 0;
        }
        return std::make_tuple(std::make_shared<const ItemValueVariant>(ur_ok),
                               std::make_shared<const SubItemValuePerItemVariant>(Fjr_ok),
                               std::make_shared<const ItemValueVariant>(ur_saved_ok),
                               std::make_shared<const SubItemValuePerItemVariant>(Fjr_saved_ok));
      }
    }
  }
}

template <size_t Dimension>
CouplingData<Dimension>::CouplingData()
{
  std::cout << "\033[01;31m"
            << "COUPLING DATA CREATION" << Dimension << "\033[00;00m" << std::endl;
}

using CouplingData3D = CouplingData<3>;
using CouplingData2D = CouplingData<2>;
using CouplingData1D = CouplingData<1>;

#endif   // PUGS_HAS_COSTO
