#ifndef CHECKPOINT_UTILS_HPP
#define CHECKPOINT_UTILS_HPP

#include <utils/HighFivePugsUtils.hpp>

#include <language/utils/SymbolTable.hpp>
#include <mesh/CellType.hpp>
#include <mesh/ItemArray.hpp>
#include <mesh/ItemValue.hpp>
#include <utils/Messenger.hpp>

template <typename DataType>
PUGS_INLINE void
write(HighFive::Group& group, const std::string& name, const Array<DataType>& array)
{
  auto get_address = [](auto& x) { return (x.size() > 0) ? &(x[0]) : nullptr; };

  Array<size_t> size_per_rank = parallel::allGather(array.size());
  size_t global_size          = sum(size_per_rank);

  size_t current_offset = 0;
  for (size_t i = 0; i < parallel::rank(); ++i) {
    current_offset += size_per_rank[i];
  }
  std::vector<size_t> offset{current_offset, 0ul};
  std::vector<size_t> count{array.size()};

  using data_type = std::remove_const_t<DataType>;
  HighFive::DataSetCreateProps properties;
  properties.add(HighFive::Chunking(std::vector<hsize_t>{std::min(4ul * 1024ul * 1024ul, global_size)}));
  properties.add(HighFive::Shuffle());
  properties.add(HighFive::Deflate(3));

  auto xfer_props = HighFive::DataTransferProps{};
  xfer_props.add(HighFive::UseCollectiveIO{});

  HighFive::DataSet dataset;
  if constexpr (std::is_same_v<CellType, data_type>) {
    using base_type = std::underlying_type_t<CellType>;
    dataset = group.createDataSet<base_type>(name, HighFive::DataSpace{std::vector<size_t>{global_size}}, properties);
    dataset.select(offset, count)
      .template write_raw<base_type>(reinterpret_cast<const base_type*>(get_address(array)), xfer_props);
  } else if constexpr ((std::is_same_v<CellId, data_type>) or (std::is_same_v<FaceId, data_type>) or
                       (std::is_same_v<EdgeId, data_type>) or (std::is_same_v<NodeId, data_type>)) {
    using base_type = typename data_type::base_type;

    dataset = group.createDataSet<base_type>(name, HighFive::DataSpace{std::vector<size_t>{global_size}}, properties);
    dataset.select(offset, count)
      .template write_raw<base_type>(reinterpret_cast<const base_type*>(get_address(array)), xfer_props);
  } else {
    dataset = group.createDataSet<data_type>(name, HighFive::DataSpace{std::vector<size_t>{global_size}}, properties);
    dataset.select(offset, count).template write_raw<data_type>(get_address(array), xfer_props);
  }

  std::vector<size_t> size_vector;
  for (size_t i = 0; i < size_per_rank.size(); ++i) {
    size_vector.push_back(size_per_rank[i]);
  }

  dataset.createAttribute("size_per_rank", size_vector);
}

template <typename DataType>
PUGS_INLINE void
write(HighFive::Group& group, const std::string& name, const Table<DataType>& table)
{
  const size_t number_of_columns = parallel::allReduceMax(table.numberOfColumns());
  if ((table.numberOfColumns() != number_of_columns) and (table.numberOfRows() > 0)) {
    throw UnexpectedError("table must have same number of columns in parallel");
  }

  auto get_address = [](auto& t) { return (t.numberOfRows() * t.numberOfColumns() > 0) ? &(t(0, 0)) : nullptr; };

  Array<size_t> number_of_rows_per_rank = parallel::allGather(table.numberOfRows());
  size_t global_size                    = sum(number_of_rows_per_rank) * number_of_columns;

  size_t current_offset = 0;
  for (size_t i = 0; i < parallel::rank(); ++i) {
    current_offset += number_of_rows_per_rank[i] * table.numberOfColumns();
  }
  std::vector<size_t> offset{current_offset, 0ul};
  std::vector<size_t> count{table.numberOfRows() * table.numberOfColumns()};

  using data_type = std::remove_const_t<DataType>;
  HighFive::DataSetCreateProps properties;
  properties.add(HighFive::Chunking(std::vector<hsize_t>{std::min(4ul * 1024ul * 1024ul, global_size)}));
  properties.add(HighFive::Shuffle());
  properties.add(HighFive::Deflate(3));

  auto xfer_props = HighFive::DataTransferProps{};
  xfer_props.add(HighFive::UseCollectiveIO{});

  HighFive::DataSet dataset =
    group.createDataSet<data_type>(name, HighFive::DataSpace{std::vector<size_t>{global_size}}, properties);
  dataset.select(offset, count).template write_raw<data_type>(get_address(table), xfer_props);

  std::vector<size_t> number_of_rows_per_rank_vector;
  for (size_t i = 0; i < number_of_rows_per_rank.size(); ++i) {
    number_of_rows_per_rank_vector.push_back(number_of_rows_per_rank[i]);
  }

  dataset.createAttribute("number_of_rows_per_rank", number_of_rows_per_rank_vector);
  dataset.createAttribute("number_of_columns", number_of_columns);
}

template <typename DataType, ItemType item_type, typename ConnectivityPtr>
void write(HighFive::Group& group,
           const std::string& name,
           const ItemValue<DataType, item_type, ConnectivityPtr>& item_value);

template <typename DataType, ItemType item_type, typename ConnectivityPtr>
void write(HighFive::Group& group,
           const std::string& name,
           const ItemArray<DataType, item_type, ConnectivityPtr>& item_array);

void writeDiscreteFunctionVariant(const std::string& symbol_name,
                                  const EmbeddedData& embedded_data,
                                  HighFive::File& file,
                                  HighFive::Group& checkpoint_group,
                                  HighFive::Group& symbol_table_group);

void writeIBoundaryConditionDescriptor(const std::string& symbol_name,
                                       const EmbeddedData& embedded_data,
                                       HighFive::File& file,
                                       HighFive::Group& checkpoint_group,
                                       HighFive::Group& symbol_table_group);

void writeIBoundaryDescriptor(const std::string& symbol_name,
                              const EmbeddedData& embedded_data,
                              HighFive::File& file,
                              HighFive::Group& checkpoint_group,
                              HighFive::Group& symbol_table_group);

void writeIDiscreteFunctionDescriptor(const std::string& symbol_name,
                                      const EmbeddedData& embedded_data,
                                      HighFive::File& file,
                                      HighFive::Group& checkpoint_group,
                                      HighFive::Group& symbol_table_group);

void writeIInterfaceDescriptor(const std::string& symbol_name,
                               const EmbeddedData& embedded_data,
                               HighFive::File& file,
                               HighFive::Group& checkpoint_group,
                               HighFive::Group& symbol_table_group);

void writeINamedDiscreteData(const std::string& symbol_name,
                             const EmbeddedData& embedded_data,
                             HighFive::File& file,
                             HighFive::Group& checkpoint_group,
                             HighFive::Group& symbol_table_group);

void writeIQuadratureDescriptor(const std::string& symbol_name,
                                const EmbeddedData& embedded_data,
                                HighFive::File& file,
                                HighFive::Group& checkpoint_group,
                                HighFive::Group& symbol_table_group);

void writeItemArrayVariant(const std::string& symbol_name,
                           const EmbeddedData& embedded_data,
                           HighFive::File& file,
                           HighFive::Group& checkpoint_group,
                           HighFive::Group& symbol_table_group);

void writeItemType(const std::string& symbol_name,
                   const EmbeddedData& embedded_data,
                   HighFive::File& file,
                   HighFive::Group& checkpoint_group,
                   HighFive::Group& symbol_table_group);

void writeItemValueVariant(const std::string& symbol_name,
                           const EmbeddedData& embedded_data,
                           HighFive::File& file,
                           HighFive::Group& checkpoint_group,
                           HighFive::Group& symbol_table_group);

void writeIZoneDescriptor(const std::string& symbol_name,
                          const EmbeddedData& embedded_data,
                          HighFive::File& file,
                          HighFive::Group& checkpoint_group,
                          HighFive::Group& symbol_table_group);

void writeMesh(const std::string& symbol_name,
               const EmbeddedData& embedded_data,
               HighFive::File& file,
               HighFive::Group& checkpoint_group,
               HighFive::Group& symbol_table_group);

void writeOStream(const std::string& symbol_name,
                  const EmbeddedData& embedded_data,
                  HighFive::File& file,
                  HighFive::Group& checkpoint_group,
                  HighFive::Group& symbol_table_group);

#endif   // CHECKPOINT_UTILS_HPP
