#include <utils/checkpointing/CheckpointUtils.hpp>

#include <analysis/IQuadratureDescriptor.hpp>
#include <language/modules/MeshModuleTypes.hpp>
#include <language/modules/SchemeModuleTypes.hpp>
#include <language/modules/WriterModuleTypes.hpp>
#include <language/utils/ASTNodeDataTypeTraits.hpp>
#include <language/utils/DataHandler.hpp>
#include <language/utils/OFStream.hpp>
#include <language/utils/OStream.hpp>
#include <mesh/ItemArrayVariant.hpp>
#include <mesh/ItemType.hpp>
#include <mesh/ItemValueVariant.hpp>
#include <mesh/Mesh.hpp>
#include <mesh/MeshVariant.hpp>
#include <mesh/NamedBoundaryDescriptor.hpp>
#include <mesh/NamedInterfaceDescriptor.hpp>
#include <mesh/NamedZoneDescriptor.hpp>
#include <mesh/NumberedBoundaryDescriptor.hpp>
#include <mesh/NumberedInterfaceDescriptor.hpp>
#include <mesh/NumberedZoneDescriptor.hpp>
#include <output/INamedDiscreteData.hpp>
#include <output/NamedDiscreteFunction.hpp>
#include <output/NamedItemArrayVariant.hpp>
#include <output/NamedItemValueVariant.hpp>
#include <scheme/DiscreteFunctionP0.hpp>
#include <scheme/DiscreteFunctionP0Vector.hpp>
#include <scheme/DiscreteFunctionVariant.hpp>
#include <scheme/IBoundaryConditionDescriptor.hpp>
#include <utils/checkpointing/DiscreteFunctionTypeHFType.hpp>
#include <utils/checkpointing/IBoundaryConditionDescriptorHFType.hpp>
#include <utils/checkpointing/IBoundaryDescriptorHFType.hpp>
#include <utils/checkpointing/IInterfaceDescriptorHFType.hpp>
#include <utils/checkpointing/INamedDiscreteDataHF.hpp>
#include <utils/checkpointing/IZoneDescriptorHFType.hpp>
#include <utils/checkpointing/ItemTypeHFType.hpp>
#include <utils/checkpointing/OStreamTypeHFType.hpp>
#include <utils/checkpointing/QuadratureTypeHFType.hpp>
#include <utils/checkpointing/RefItemListHFType.hpp>

#include <scheme/AxisBoundaryConditionDescriptor.hpp>
#include <scheme/DirichletBoundaryConditionDescriptor.hpp>
#include <scheme/ExternalBoundaryConditionDescriptor.hpp>
#include <scheme/FixedBoundaryConditionDescriptor.hpp>
#include <scheme/FourierBoundaryConditionDescriptor.hpp>
#include <scheme/FreeBoundaryConditionDescriptor.hpp>
#include <scheme/InflowBoundaryConditionDescriptor.hpp>
#include <scheme/NeumannBoundaryConditionDescriptor.hpp>
#include <scheme/OutflowBoundaryConditionDescriptor.hpp>
#include <scheme/SymmetryBoundaryConditionDescriptor.hpp>

void writeMesh(std::shared_ptr<const MeshVariant> mesh_v, HighFive::File& file, HighFive::Group& checkpoint_group);

template <typename DataType, ItemType item_type, typename ConnectivityPtr>
void
write(HighFive::Group& group,
      const std::string& name,
      const ItemValue<DataType, item_type, ConnectivityPtr>& item_value)
{
  write(group, name, item_value.arrayView());
}

template <typename DataType, ItemType item_type, typename ConnectivityPtr>
void
write(HighFive::Group& group,
      const std::string& name,
      const ItemArray<DataType, item_type, ConnectivityPtr>& item_array)
{
  write(group, name, item_array.tableView());
}

template <ItemType item_type, size_t Dimension>
void
writeRefItemList(const Connectivity<Dimension>& connectivity, HighFive::Group& connectivity_group)
{
  for (size_t i_item_list = 0; i_item_list < connectivity.template numberOfRefItemList<item_type>(); ++i_item_list) {
    auto ref_item_list = connectivity.template refItemList<item_type>(i_item_list);

    std::ostringstream ref_item_list_group_name;
    ref_item_list_group_name << "item_ref_list/" << itemName(item_type) << '/' << ref_item_list.refId().tagName();
    HighFive::Group ref_item_list_group = connectivity_group.createGroup(ref_item_list_group_name.str());
    ref_item_list_group.createAttribute("tag_name", ref_item_list.refId().tagName());
    ref_item_list_group.createAttribute("tag_number", ref_item_list.refId().tagNumber());
    ref_item_list_group.createAttribute("type", ref_item_list.type());

    write(ref_item_list_group, "list", ref_item_list.list());
  }
}

template <size_t Dimension>
void
writeConnectivity(const Connectivity<Dimension>& connectivity, HighFive::File& file, HighFive::Group& checkpoint_group)
{
  std::string connectivity_group_name = "connectivity/" + std::to_string(connectivity.id());
  if (not checkpoint_group.exist(connectivity_group_name)) {
    bool linked = false;
    for (auto group_name : file.listObjectNames()) {
      if (file.exist(group_name + "/" + connectivity_group_name)) {
        checkpoint_group.createHardLink(connectivity_group_name,
                                        file.getGroup(group_name + "/" + connectivity_group_name));
        linked = true;
        break;
      }
    }

    if (not linked) {
      HighFive::Group connectivity_group = checkpoint_group.createGroup(connectivity_group_name);

      connectivity_group.createAttribute("dimension", connectivity.dimension());
      connectivity_group.createAttribute("id", connectivity.id());
      connectivity_group.createAttribute("type", std::string{"unstructured"});

      write(connectivity_group, "cell_to_node_matrix_values",
            connectivity.getMatrix(ItemType::cell, ItemType::node).values());
      write(connectivity_group, "cell_to_node_matrix_rowsMap",
            connectivity.getMatrix(ItemType::cell, ItemType::node).rowsMap());

      if constexpr (Dimension > 1) {
        write(connectivity_group, "cell_to_face_matrix_values",
              connectivity.getMatrix(ItemType::cell, ItemType::face).values());
        write(connectivity_group, "cell_to_face_matrix_rowsMap",
              connectivity.getMatrix(ItemType::cell, ItemType::face).rowsMap());

        write(connectivity_group, "face_to_node_matrix_values",
              connectivity.getMatrix(ItemType::face, ItemType::node).values());
        write(connectivity_group, "face_to_node_matrix_rowsMap",
              connectivity.getMatrix(ItemType::face, ItemType::node).rowsMap());

        write(connectivity_group, "node_to_face_matrix_values",
              connectivity.getMatrix(ItemType::node, ItemType::face).values());
        write(connectivity_group, "node_to_face_matrix_rowsMap",
              connectivity.getMatrix(ItemType::node, ItemType::face).rowsMap());

        write(connectivity_group, "cell_face_is_reversed", connectivity.cellFaceIsReversed().arrayView());
      }

      if constexpr (Dimension > 2) {
        write(connectivity_group, "cell_to_edge_matrix_values",
              connectivity.getMatrix(ItemType::cell, ItemType::edge).values());
        write(connectivity_group, "cell_to_edge_matrix_rowsMap",
              connectivity.getMatrix(ItemType::cell, ItemType::edge).rowsMap());

        write(connectivity_group, "face_to_edge_matrix_values",
              connectivity.getMatrix(ItemType::face, ItemType::edge).values());
        write(connectivity_group, "face_to_edge_matrix_rowsMap",
              connectivity.getMatrix(ItemType::face, ItemType::edge).rowsMap());

        write(connectivity_group, "edge_to_node_matrix_values",
              connectivity.getMatrix(ItemType::edge, ItemType::node).values());
        write(connectivity_group, "edge_to_node_matrix_rowsMap",
              connectivity.getMatrix(ItemType::edge, ItemType::node).rowsMap());

        write(connectivity_group, "node_to_edge_matrix_values",
              connectivity.getMatrix(ItemType::node, ItemType::edge).values());
        write(connectivity_group, "node_to_edge_matrix_rowsMap",
              connectivity.getMatrix(ItemType::node, ItemType::edge).rowsMap());

        write(connectivity_group, "face_edge_is_reversed", connectivity.faceEdgeIsReversed().arrayView());
      }

      write(connectivity_group, "cell_type", connectivity.cellType());

      write(connectivity_group, "cell_numbers", connectivity.cellNumber());
      write(connectivity_group, "node_numbers", connectivity.nodeNumber());

      write(connectivity_group, "cell_owner", connectivity.cellOwner());
      write(connectivity_group, "node_owner", connectivity.nodeOwner());

      if constexpr (Dimension > 1) {
        write(connectivity_group, "face_numbers", connectivity.faceNumber());

        write(connectivity_group, "face_owner", connectivity.faceOwner());
      }
      if constexpr (Dimension > 2) {
        write(connectivity_group, "edge_numbers", connectivity.edgeNumber());

        write(connectivity_group, "edge_owner", connectivity.edgeOwner());
      }

      writeRefItemList<ItemType::cell>(connectivity, connectivity_group);
      writeRefItemList<ItemType::face>(connectivity, connectivity_group);
      writeRefItemList<ItemType::edge>(connectivity, connectivity_group);
      writeRefItemList<ItemType::node>(connectivity, connectivity_group);
    }
  }
}

void
writeConnectivity(const IConnectivity& connectivity, HighFive::File& file, HighFive::Group& checkpoint_group)
{
  switch (connectivity.dimension()) {
  case 1: {
    writeConnectivity(dynamic_cast<const Connectivity<1>&>(connectivity), file, checkpoint_group);
    break;
  }
  case 2: {
    writeConnectivity(dynamic_cast<const Connectivity<2>&>(connectivity), file, checkpoint_group);
    break;
  }
  case 3: {
    writeConnectivity(dynamic_cast<const Connectivity<3>&>(connectivity), file, checkpoint_group);
    break;
  }
  default: {
    throw UnexpectedError("invalid connectivity dimension");
  }
  }
}

void
writeDiscreteFunctionVariant(HighFive::Group& variable_group,
                             std::shared_ptr<const DiscreteFunctionVariant> discrete_function_v,
                             HighFive::File& file,
                             HighFive::Group& checkpoint_group)
{
  variable_group.createAttribute("type", dataTypeName(ast_node_data_type_from<decltype(discrete_function_v)>));

  std::visit(
    [&](auto&& discrete_function) {
      auto mesh_v  = discrete_function.meshVariant();
      using DFType = std::decay_t<decltype(discrete_function)>;
      variable_group.createAttribute("Vh_type", discrete_function.descriptor().type());

      variable_group.createAttribute("mesh_id", mesh_v->id());
      writeMesh(mesh_v, file, checkpoint_group);
      if constexpr (is_discrete_function_P0_v<DFType>) {
        using data_type = std::decay_t<typename DFType::data_type>;
        variable_group.createAttribute("data_type", dataTypeName(ast_node_data_type_from<data_type>));
        write(variable_group, "values", discrete_function.cellValues());
      } else if constexpr (is_discrete_function_P0_vector_v<DFType>) {
        using data_type = std::decay_t<typename DFType::data_type>;
        variable_group.createAttribute("data_type", dataTypeName(ast_node_data_type_from<data_type>));
        write(variable_group, "values", discrete_function.cellArrays());
      }
    },
    discrete_function_v->discreteFunction());
}

void
writeDiscreteFunctionVariant(const std::string& symbol_name,
                             const EmbeddedData& embedded_data,
                             HighFive::File& file,
                             HighFive::Group& checkpoint_group,
                             HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const DiscreteFunctionVariant> discrete_function_p =
    dynamic_cast<const DataHandler<const DiscreteFunctionVariant>&>(embedded_data.get()).data_ptr();

  writeDiscreteFunctionVariant(variable_group, discrete_function_p, file, checkpoint_group);
}

void
writeIBoundaryDescriptor(HighFive::Group& variable_group, const IBoundaryDescriptor& iboundary_descriptor)
{
  variable_group.createAttribute("iboundary_descriptor_type", iboundary_descriptor.type());

  switch (iboundary_descriptor.type()) {
  case IBoundaryDescriptor::Type::named: {
    const NamedBoundaryDescriptor& named_boundary_descriptor =
      dynamic_cast<const NamedBoundaryDescriptor&>(iboundary_descriptor);
    variable_group.createAttribute("name", named_boundary_descriptor.name());
    break;
  }
  case IBoundaryDescriptor::Type::numbered: {
    const NumberedBoundaryDescriptor& numbered_boundary_descriptor =
      dynamic_cast<const NumberedBoundaryDescriptor&>(iboundary_descriptor);
    variable_group.createAttribute("number", numbered_boundary_descriptor.number());
    break;
  }
  }
}

void
writeIBoundaryConditionDescriptor(const std::string& symbol_name,
                                  const EmbeddedData& embedded_data,
                                  HighFive::File&,
                                  HighFive::Group&,
                                  HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const IBoundaryConditionDescriptor> iboundary_condition_descriptor_p =
    dynamic_cast<const DataHandler<const IBoundaryConditionDescriptor>&>(embedded_data.get()).data_ptr();

  const IBoundaryConditionDescriptor& iboundary_condition_descriptor = *iboundary_condition_descriptor_p;

  variable_group.createAttribute("type",
                                 dataTypeName(ast_node_data_type_from<decltype(iboundary_condition_descriptor_p)>));
  variable_group.createAttribute("iboundary_condition_descriptor_type", iboundary_condition_descriptor.type());

  HighFive::Group boundary_group = variable_group.createGroup("boundary");

  switch (iboundary_condition_descriptor.type()) {
  case IBoundaryConditionDescriptor::Type::axis: {
    const AxisBoundaryConditionDescriptor& axis_bc_descriptor =
      dynamic_cast<const AxisBoundaryConditionDescriptor&>(iboundary_condition_descriptor);
    writeIBoundaryDescriptor(boundary_group, axis_bc_descriptor.boundaryDescriptor());
    break;
  }
  case IBoundaryConditionDescriptor::Type::dirichlet: {
    const DirichletBoundaryConditionDescriptor& dirichlet_bc_descriptor =
      dynamic_cast<const DirichletBoundaryConditionDescriptor&>(iboundary_condition_descriptor);
    writeIBoundaryDescriptor(boundary_group, dirichlet_bc_descriptor.boundaryDescriptor());
    variable_group.createAttribute("name", dirichlet_bc_descriptor.name());
    variable_group.createAttribute("rhs_function_id", dirichlet_bc_descriptor.rhsSymbolId().id());
    break;
  }
  case IBoundaryConditionDescriptor::Type::external: {
    throw NotImplementedError("checkpoint/resume with sockets");

    break;
  }
  case IBoundaryConditionDescriptor::Type::fourier: {
    const FourierBoundaryConditionDescriptor& fourier_bc_descriptor =
      dynamic_cast<const FourierBoundaryConditionDescriptor&>(iboundary_condition_descriptor);
    writeIBoundaryDescriptor(boundary_group, fourier_bc_descriptor.boundaryDescriptor());
    variable_group.createAttribute("name", fourier_bc_descriptor.name());
    variable_group.createAttribute("rhs_function_id", fourier_bc_descriptor.rhsSymbolId().id());
    variable_group.createAttribute("mass_function_id", fourier_bc_descriptor.massSymbolId().id());
    break;
  }
  case IBoundaryConditionDescriptor::Type::fixed: {
    const FixedBoundaryConditionDescriptor& fixed_bc_descriptor =
      dynamic_cast<const FixedBoundaryConditionDescriptor&>(iboundary_condition_descriptor);
    writeIBoundaryDescriptor(boundary_group, fixed_bc_descriptor.boundaryDescriptor());
    break;
  }
  case IBoundaryConditionDescriptor::Type::free: {
    const FreeBoundaryConditionDescriptor& free_bc_descriptor =
      dynamic_cast<const FreeBoundaryConditionDescriptor&>(iboundary_condition_descriptor);
    writeIBoundaryDescriptor(boundary_group, free_bc_descriptor.boundaryDescriptor());
    break;
  }
  case IBoundaryConditionDescriptor::Type::inflow: {
    const InflowBoundaryConditionDescriptor& inflow_bc_descriptor =
      dynamic_cast<const InflowBoundaryConditionDescriptor&>(iboundary_condition_descriptor);
    writeIBoundaryDescriptor(boundary_group, inflow_bc_descriptor.boundaryDescriptor());
    variable_group.createAttribute("function_id", inflow_bc_descriptor.functionSymbolId().id());
    break;
  }
  case IBoundaryConditionDescriptor::Type::neumann: {
    const NeumannBoundaryConditionDescriptor& neumann_bc_descriptor =
      dynamic_cast<const NeumannBoundaryConditionDescriptor&>(iboundary_condition_descriptor);
    writeIBoundaryDescriptor(boundary_group, neumann_bc_descriptor.boundaryDescriptor());
    variable_group.createAttribute("name", neumann_bc_descriptor.name());
    variable_group.createAttribute("rhs_function_id", neumann_bc_descriptor.rhsSymbolId().id());
    break;
  }
  case IBoundaryConditionDescriptor::Type::outflow: {
    const OutflowBoundaryConditionDescriptor& outflow_bc_descriptor =
      dynamic_cast<const OutflowBoundaryConditionDescriptor&>(iboundary_condition_descriptor);
    writeIBoundaryDescriptor(boundary_group, outflow_bc_descriptor.boundaryDescriptor());
    break;
  }
  case IBoundaryConditionDescriptor::Type::symmetry: {
    const SymmetryBoundaryConditionDescriptor& symmetric_bc_descriptor =
      dynamic_cast<const SymmetryBoundaryConditionDescriptor&>(iboundary_condition_descriptor);
    writeIBoundaryDescriptor(boundary_group, symmetric_bc_descriptor.boundaryDescriptor());
    break;
  }
  }
}

void
writeIBoundaryDescriptor(const std::string& symbol_name,
                         const EmbeddedData& embedded_data,
                         HighFive::File&,
                         HighFive::Group&,
                         HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const IBoundaryDescriptor> iboundary_descriptor_p =
    dynamic_cast<const DataHandler<const IBoundaryDescriptor>&>(embedded_data.get()).data_ptr();

  const IBoundaryDescriptor& iboundary_descriptor = *iboundary_descriptor_p;

  variable_group.createAttribute("type", dataTypeName(ast_node_data_type_from<decltype(iboundary_descriptor_p)>));

  writeIBoundaryDescriptor(variable_group, iboundary_descriptor);
}

void
writeIDiscreteFunctionDescriptor(const std::string& symbol_name,
                                 const EmbeddedData& embedded_data,
                                 HighFive::File&,
                                 HighFive::Group&,
                                 HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const IDiscreteFunctionDescriptor> idiscrete_function_desriptor_p =
    dynamic_cast<const DataHandler<const IDiscreteFunctionDescriptor>&>(embedded_data.get()).data_ptr();

  const IDiscreteFunctionDescriptor& idiscrete_function_descriptor = *idiscrete_function_desriptor_p;

  variable_group.createAttribute("type",
                                 dataTypeName(ast_node_data_type_from<decltype(idiscrete_function_desriptor_p)>));
  variable_group.createAttribute("discrete_function_type", idiscrete_function_descriptor.type());
}

void
writeIInterfaceDescriptor(const std::string& symbol_name,
                          const EmbeddedData& embedded_data,
                          HighFive::File&,
                          HighFive::Group&,
                          HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const IInterfaceDescriptor> iinterface_descriptor_p =
    dynamic_cast<const DataHandler<const IInterfaceDescriptor>&>(embedded_data.get()).data_ptr();

  const IInterfaceDescriptor& iinterface_descriptor = *iinterface_descriptor_p;

  variable_group.createAttribute("type", dataTypeName(ast_node_data_type_from<decltype(iinterface_descriptor_p)>));
  variable_group.createAttribute("iinterface_descriptor_type", iinterface_descriptor.type());

  switch (iinterface_descriptor.type()) {
  case IInterfaceDescriptor::Type::named: {
    const NamedInterfaceDescriptor& named_interface_descriptor =
      dynamic_cast<const NamedInterfaceDescriptor&>(iinterface_descriptor);
    variable_group.createAttribute("name", named_interface_descriptor.name());
    break;
  }
  case IInterfaceDescriptor::Type::numbered: {
    const NumberedInterfaceDescriptor& numbered_boundary_descriptor =
      dynamic_cast<const NumberedInterfaceDescriptor&>(iinterface_descriptor);
    variable_group.createAttribute("number", numbered_boundary_descriptor.number());
    break;
  }
  }
}

void
writeINamedDiscreteData(const std::string& symbol_name,
                        const EmbeddedData& embedded_data,
                        HighFive::File& file,
                        HighFive::Group& checkpoint_group,
                        HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const INamedDiscreteData> inamed_discrete_data_p =
    dynamic_cast<const DataHandler<const INamedDiscreteData>&>(embedded_data.get()).data_ptr();

  const INamedDiscreteData& inamed_discrete_data = *inamed_discrete_data_p;

  variable_group.createAttribute("type", dataTypeName(ast_node_data_type_from<decltype(inamed_discrete_data_p)>));
  variable_group.createAttribute("named_discrete_data_type", inamed_discrete_data.type());
  variable_group.createAttribute("named_discrete_data_name", inamed_discrete_data.name());

  switch (inamed_discrete_data.type()) {
  case INamedDiscreteData::Type::discrete_function: {
    const NamedDiscreteFunction& named_discrete_function =
      dynamic_cast<const NamedDiscreteFunction&>(inamed_discrete_data);

    HighFive::Group discrete_function_group = variable_group.createGroup("discrete_function_variant");
    writeDiscreteFunctionVariant(discrete_function_group, named_discrete_function.discreteFunctionVariant(), file,
                                 checkpoint_group);
    break;
  }
  case INamedDiscreteData::Type::item_array: {
    const NamedItemArrayVariant& named_item_array_v = dynamic_cast<const NamedItemArrayVariant&>(inamed_discrete_data);

    throw NotImplementedError("INamedDiscreteData::Type::item_array");
#warning NOT IMPLEMENTED
    break;
  }
  case INamedDiscreteData::Type::item_value: {
    const NamedItemValueVariant& named_item_value_v = dynamic_cast<const NamedItemValueVariant&>(inamed_discrete_data);

    throw NotImplementedError("INamedDiscreteData::Type::item_value");
#warning NOT IMPLEMENTED
    break;
  }
  }
}

void
writeIQuadratureDescriptor(const std::string& symbol_name,
                           const EmbeddedData& embedded_data,
                           HighFive::File&,
                           HighFive::Group&,
                           HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const IQuadratureDescriptor> iquadrature_descriptor_p =
    dynamic_cast<const DataHandler<const IQuadratureDescriptor>&>(embedded_data.get()).data_ptr();

  const IQuadratureDescriptor& iquadrature_descriptor = *iquadrature_descriptor_p;

  variable_group.createAttribute("type", dataTypeName(ast_node_data_type_from<decltype(iquadrature_descriptor_p)>));
  variable_group.createAttribute("quadrature_type", iquadrature_descriptor.type());
  variable_group.createAttribute("quadrature_degree", iquadrature_descriptor.degree());
}

void
writeItemArrayVariant(HighFive::Group& variable_group,
                      std::shared_ptr<const ItemArrayVariant> item_array_variant_v,
                      HighFive::File& file,
                      HighFive::Group& checkpoint_group)
{
  variable_group.createAttribute("type", dataTypeName(ast_node_data_type_from<decltype(item_array_variant_v)>));

  std::visit(
    [&](auto&& item_array) {
      using ItemArrayT = std::decay_t<decltype(item_array)>;

      variable_group.createAttribute("item_type", ItemArrayT::item_t);
      using data_type = std::decay_t<typename ItemArrayT::data_type>;
      variable_group.createAttribute("data_type", dataTypeName(ast_node_data_type_from<data_type>));

      const IConnectivity& connectivity = *item_array.connectivity_ptr();
      variable_group.createAttribute("connectivity_id", connectivity.id());
      writeConnectivity(connectivity, file, checkpoint_group);

      write(variable_group, "arrays", item_array);
    },
    item_array_variant_v->itemArray());
}

void
writeItemArrayVariant(const std::string& symbol_name,
                      const EmbeddedData& embedded_data,
                      HighFive::File& file,
                      HighFive::Group& checkpoint_group,
                      HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const ItemArrayVariant> item_array_variant_p =
    dynamic_cast<const DataHandler<const ItemArrayVariant>&>(embedded_data.get()).data_ptr();

  writeItemArrayVariant(variable_group, item_array_variant_p, file, checkpoint_group);
}

void
writeItemType(const std::string& symbol_name,
              const EmbeddedData& embedded_data,
              HighFive::File&,
              HighFive::Group&,
              HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const ItemType> item_type_p =
    dynamic_cast<const DataHandler<const ItemType>&>(embedded_data.get()).data_ptr();

  const ItemType& item_type = *item_type_p;

  variable_group.createAttribute("type", dataTypeName(ast_node_data_type_from<decltype(item_type_p)>));
  variable_group.createAttribute("item_type", item_type);
}

void
writeItemValueVariant(HighFive::Group& variable_group,
                      std::shared_ptr<const ItemValueVariant> item_value_variant_v,
                      HighFive::File& file,
                      HighFive::Group& checkpoint_group)
{
  variable_group.createAttribute("type", dataTypeName(ast_node_data_type_from<decltype(item_value_variant_v)>));

  std::visit(
    [&](auto&& item_value) {
      using ItemValueT = std::decay_t<decltype(item_value)>;

      variable_group.createAttribute("item_type", ItemValueT::item_t);
      using data_type = std::decay_t<typename ItemValueT::data_type>;
      variable_group.createAttribute("data_type", dataTypeName(ast_node_data_type_from<data_type>));

      const IConnectivity& connectivity = *item_value.connectivity_ptr();
      variable_group.createAttribute("connectivity_id", connectivity.id());
      writeConnectivity(connectivity, file, checkpoint_group);

      write(variable_group, "values", item_value);
    },
    item_value_variant_v->itemValue());
}

void
writeItemValueVariant(const std::string& symbol_name,
                      const EmbeddedData& embedded_data,
                      HighFive::File& file,
                      HighFive::Group& checkpoint_group,
                      HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const ItemValueVariant> item_value_variant_p =
    dynamic_cast<const DataHandler<const ItemValueVariant>&>(embedded_data.get()).data_ptr();

  writeItemValueVariant(variable_group, item_value_variant_p, file, checkpoint_group);
}

void
writeIZoneDescriptor(const std::string& symbol_name,
                     const EmbeddedData& embedded_data,
                     HighFive::File&,
                     HighFive::Group&,
                     HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const IZoneDescriptor> izone_descriptor_p =
    dynamic_cast<const DataHandler<const IZoneDescriptor>&>(embedded_data.get()).data_ptr();

  const IZoneDescriptor& izone_descriptor = *izone_descriptor_p;

  variable_group.createAttribute("type", dataTypeName(ast_node_data_type_from<decltype(izone_descriptor_p)>));
  variable_group.createAttribute("izone_descriptor_type", izone_descriptor.type());

  switch (izone_descriptor.type()) {
  case IZoneDescriptor::Type::named: {
    const NamedZoneDescriptor& named_zone_descriptor = dynamic_cast<const NamedZoneDescriptor&>(izone_descriptor);
    variable_group.createAttribute("name", named_zone_descriptor.name());
    break;
  }
  case IZoneDescriptor::Type::numbered: {
    const NumberedZoneDescriptor& numbered_boundary_descriptor =
      dynamic_cast<const NumberedZoneDescriptor&>(izone_descriptor);
    variable_group.createAttribute("number", numbered_boundary_descriptor.number());
    break;
  }
  }
}

void
writeMesh(std::shared_ptr<const MeshVariant> mesh_v, HighFive::File& file, HighFive::Group& checkpoint_group)
{
  std::string mesh_group_name = "mesh/" + std::to_string(mesh_v->id());
  if (not checkpoint_group.exist(mesh_group_name)) {
    bool linked = false;
    for (auto group_name : file.listObjectNames()) {
      if (file.exist(group_name + "/" + mesh_group_name)) {
        checkpoint_group.createHardLink(mesh_group_name, file.getGroup(group_name + "/" + mesh_group_name));
        linked = true;
        break;
      }
    }

    if (not linked) {
      HighFive::Group mesh_group = checkpoint_group.createGroup(mesh_group_name);
      mesh_group.createAttribute("connectivity", mesh_v->connectivity().id());
      std::visit(
        [&](auto&& mesh) {
          using MeshType = mesh_type_t<decltype(mesh)>;
          if constexpr (is_polygonal_mesh_v<MeshType>) {
            mesh_group.createAttribute("id", mesh->id());
            mesh_group.createAttribute("type", std::string{"polygonal"});
            mesh_group.createAttribute("dimension", mesh->dimension());
            write(mesh_group, "xr", mesh->xr());
          } else {
            throw UnexpectedError("unexpected mesh type");
          }
        },
        mesh_v->variant());
    }
  }

  std::visit(
    [&](auto&& mesh) {
      using MeshType = mesh_type_t<decltype(mesh)>;
      if constexpr (is_polygonal_mesh_v<MeshType>) {
        writeConnectivity(mesh->connectivity(), file, checkpoint_group);
      }
    },
    mesh_v->variant());
}

void
writeMesh(const std::string& symbol_name,
          const EmbeddedData& embedded_data,
          HighFive::File& file,
          HighFive::Group& checkpoint_group,
          HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const MeshVariant> mesh_v =
    dynamic_cast<const DataHandler<const MeshVariant>&>(embedded_data.get()).data_ptr();

  variable_group.createAttribute("type", dataTypeName(ast_node_data_type_from<decltype(mesh_v)>));
  variable_group.createAttribute("id", mesh_v->id());

  writeMesh(mesh_v, file, checkpoint_group);
}

void
writeOStream(const std::string& symbol_name,
             const EmbeddedData& embedded_data,
             HighFive::File&,
             HighFive::Group&,
             HighFive::Group& symbol_table_group)
{
  HighFive::Group variable_group = symbol_table_group.createGroup("embedded/" + symbol_name);

  std::shared_ptr<const OStream> ostream_p =
    dynamic_cast<const DataHandler<const OStream>&>(embedded_data.get()).data_ptr();

  const OStream& ostream = *ostream_p;

  variable_group.createAttribute("type", dataTypeName(ast_node_data_type_from<decltype(ostream_p)>));
  variable_group.createAttribute("ostream_type", ostream.type());

  switch (ostream.type()) {
  case OStream::Type::std_ofstream: {
    const OFStream& ofstream = dynamic_cast<const OFStream&>(ostream);
    variable_group.createAttribute("filename", ofstream.filename());
    break;
  }
  case OStream::Type::std_ostream: {
    throw NotImplementedError("std::ostream checkpoint");
  }
  }
}
