#ifndef AFFINE_TRANSFORMATION_HPP
#define AFFINE_TRANSFORMATION_HPP

#include <algebra/TinyMatrix.hpp>
#include <algebra/TinyVector.hpp>

#include <array>

template <size_t Dimension>
class AffineTransformation
{
 private:
  constexpr static size_t NumberOfPoints = Dimension + 1;

  TinyMatrix<Dimension> m_jacobian;
  TinyVector<Dimension> m_shift;
  double m_jacobian_determinant;

 public:
  PUGS_INLINE
  TinyVector<Dimension>
  operator()(const TinyVector<Dimension>& x) const
  {
    return m_jacobian * x + m_shift;
  }

  double
  jacobianDeterminant() const
  {
    return m_jacobian_determinant;
  }

  PUGS_INLINE
  AffineTransformation(const std::array<TinyVector<Dimension>, NumberOfPoints>& image_nodes)
  {
    static_assert(Dimension >= 1 and Dimension <= 3, "invalid dimension");
    if constexpr (Dimension == 1) {
      const TinyVector<Dimension>& a = image_nodes[0];
      const TinyVector<Dimension>& b = image_nodes[1];

      m_jacobian(0, 0) = 0.5 * (b[0] - a[0]);

      m_shift[0] = 0.5 * (a[0] + b[0]);
    } else if constexpr (Dimension == 2) {
      const TinyVector<Dimension>& a = image_nodes[0];
      const TinyVector<Dimension>& b = image_nodes[1];
      const TinyVector<Dimension>& c = image_nodes[2];

      for (size_t i = 0; i < Dimension; ++i) {
        m_jacobian(i, 0) = 0.5 * (b[i] - a[i]);
        m_jacobian(i, 1) = 0.5 * (c[i] - a[i]);

        m_shift[i] = 0.5 * (b[i] + c[i]);
      }
    } else {
      static_assert(Dimension == 3);

      const TinyVector<Dimension>& a = image_nodes[0];
      const TinyVector<Dimension>& b = image_nodes[1];
      const TinyVector<Dimension>& c = image_nodes[2];
      const TinyVector<Dimension>& d = image_nodes[3];

      for (size_t i = 0; i < Dimension; ++i) {
        m_jacobian(i, 0) = 0.5 * (b[i] - a[i]);
        m_jacobian(i, 1) = 0.5 * (c[i] - a[i]);
        m_jacobian(i, 2) = 0.5 * (d[i] - a[i]);

        m_shift[i] = 0.5 * (b[i] + c[i] + d[i] - a[i]);
      }
    }

    m_jacobian_determinant = det(m_jacobian);
  }

  ~AffineTransformation() = default;
};

#endif   // AFFINE_TRANSFORMATION_HPP
