#include <scheme/Order2LocalDtHyperelasticSolver.hpp>

#include <language/utils/InterpolateItemValue.hpp>
#include <mesh/ItemValueUtils.hpp>
#include <mesh/ItemValueVariant.hpp>
#include <mesh/MeshFaceBoundary.hpp>
#include <mesh/MeshFlatNodeBoundary.hpp>
#include <mesh/MeshNodeBoundary.hpp>
#include <mesh/MeshTraits.hpp>
#include <mesh/MeshVariant.hpp>
#include <mesh/StencilArray.hpp>
#include <mesh/StencilManager.hpp>
#include <mesh/SubItemValuePerItemVariant.hpp>
#include <scheme/CouplingBoundaryConditionDescriptor.hpp>
#include <scheme/DirichletBoundaryConditionDescriptor.hpp>
#include <scheme/DiscreteFunctionDPk.hpp>
#include <scheme/DiscreteFunctionDPkVariant.hpp>
#include <scheme/DiscreteFunctionP0.hpp>
#include <scheme/DiscreteFunctionUtils.hpp>
#include <scheme/DiscreteFunctionVariant.hpp>
#include <scheme/ExternalBoundaryConditionDescriptor.hpp>
#include <scheme/FixedBoundaryConditionDescriptor.hpp>
#include <scheme/HyperelasticSolver.hpp>
#include <scheme/IBoundaryConditionDescriptor.hpp>
#include <scheme/IDiscreteFunctionDescriptor.hpp>
#include <scheme/PolynomialReconstruction.hpp>
#include <scheme/PolynomialReconstructionDescriptor.hpp>
#include <scheme/SymmetryBoundaryConditionDescriptor.hpp>
#include <utils/Socket.hpp>

#include <variant>
#include <vector>

template <MeshConcept MeshType>
class Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver final
  : public Order2LocalDtHyperelasticSolverHandler::IOrder2LocalDtHyperelasticSolver
{
 private:
  static constexpr size_t Dimension = MeshType::Dimension;

  using Rdxd = TinyMatrix<Dimension>;
  using Rd   = TinyVector<Dimension>;

  using MeshDataType = MeshData<MeshType>;

  using DiscreteScalarFunction = DiscreteFunctionP0<const double>;
  using DiscreteVectorFunction = DiscreteFunctionP0<const Rd>;
  using DiscreteTensorFunction = DiscreteFunctionP0<const Rdxd>;

  class FixedBoundaryCondition;
  class PressureBoundaryCondition;
  class NormalStressBoundaryCondition;
  class SymmetryBoundaryCondition;
  class VelocityBoundaryCondition;
  class CouplingBoundaryCondition;

  using BoundaryCondition = std::variant<FixedBoundaryCondition,
                                         PressureBoundaryCondition,
                                         NormalStressBoundaryCondition,
                                         SymmetryBoundaryCondition,
                                         VelocityBoundaryCondition,
                                         CouplingBoundaryCondition>;

  using BoundaryConditionList = std::vector<BoundaryCondition>;

  NodeValuePerCell<const Rdxd>
  _computeGlaceAjr(const MeshType& mesh, const DiscreteScalarFunction& rhoaL, const DiscreteScalarFunction& rhoaT) const
  {
    MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(mesh);

    const NodeValuePerCell<const Rd> Cjr = mesh_data.Cjr();
    const NodeValuePerCell<const Rd> njr = mesh_data.njr();

    NodeValuePerCell<Rdxd> Ajr{mesh.connectivity()};
    const Rdxd I = identity;
    parallel_for(
      mesh.numberOfCells(), PUGS_LAMBDA(CellId j) {
        const size_t& nb_nodes = Ajr.numberOfSubValues(j);
        const double& rhoaL_j  = rhoaL[j];
        const double& rhoaT_j  = rhoaT[j];
        for (size_t r = 0; r < nb_nodes; ++r) {
          const Rdxd& M = tensorProduct(Cjr(j, r), njr(j, r));
          Ajr(j, r)     = rhoaL_j * M + rhoaT_j * (l2Norm(Cjr(j, r)) * I - M);
        }
      });

    return Ajr;
  }

  NodeValuePerCell<const Rdxd>
  _computeEucclhydAjr(const MeshType& mesh,
                      const DiscreteScalarFunction& rhoaL,
                      const DiscreteScalarFunction& rhoaT) const
  {
    MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(mesh);

    const NodeValuePerFace<const Rd> Nlr = mesh_data.Nlr();
    const NodeValuePerFace<const Rd> nlr = mesh_data.nlr();

    const auto& face_to_node_matrix = mesh.connectivity().faceToNodeMatrix();
    const auto& cell_to_node_matrix = mesh.connectivity().cellToNodeMatrix();
    const auto& cell_to_face_matrix = mesh.connectivity().cellToFaceMatrix();

    NodeValuePerCell<Rdxd> Ajr{mesh.connectivity()};

    parallel_for(
      Ajr.numberOfValues(), PUGS_LAMBDA(size_t jr) { Ajr[jr] = zero; });
    const Rdxd I = identity;
    parallel_for(
      mesh.numberOfCells(), PUGS_LAMBDA(CellId j) {
        const auto& cell_nodes = cell_to_node_matrix[j];

        const auto& cell_faces = cell_to_face_matrix[j];

        const double& rho_aL = rhoaL[j];
        const double& rho_aT = rhoaT[j];

        for (size_t L = 0; L < cell_faces.size(); ++L) {
          const FaceId& l        = cell_faces[L];
          const auto& face_nodes = face_to_node_matrix[l];

          auto local_node_number_in_cell = [&](NodeId node_number) {
            for (size_t i_node = 0; i_node < cell_nodes.size(); ++i_node) {
              if (node_number == cell_nodes[i_node]) {
                return i_node;
              }
            }
            return std::numeric_limits<size_t>::max();
          };

          for (size_t rl = 0; rl < face_nodes.size(); ++rl) {
            const size_t R = local_node_number_in_cell(face_nodes[rl]);
            const Rdxd& M  = tensorProduct(Nlr(l, rl), nlr(l, rl));
            Ajr(j, R) += rho_aL * M + rho_aT * (l2Norm(Nlr(l, rl)) * I - M);
          }
        }
      });

    return Ajr;
  }

  NodeValuePerCell<const Rdxd>
  _computeAjr(const SolverType& solver_type,
              const MeshType& mesh,
              const DiscreteScalarFunction& rhoaL,
              const DiscreteScalarFunction& rhoaT) const
  {
    if constexpr (Dimension == 1) {
      return _computeGlaceAjr(mesh, rhoaL, rhoaT);
    } else {
      switch (solver_type) {
      case SolverType::Glace: {
        return _computeGlaceAjr(mesh, rhoaL, rhoaT);
      }
      case SolverType::Eucclhyd: {
        return _computeEucclhydAjr(mesh, rhoaL, rhoaT);
      }
      default: {
        throw UnexpectedError("invalid solver type");
      }
      }
    }
  }

  NodeValue<Rdxd>
  _computeAr(const MeshType& mesh, const NodeValuePerCell<const Rdxd>& Ajr) const
  {
    const auto& node_to_cell_matrix               = mesh.connectivity().nodeToCellMatrix();
    const auto& node_local_numbers_in_their_cells = mesh.connectivity().nodeLocalNumbersInTheirCells();

    NodeValue<Rdxd> Ar{mesh.connectivity()};

    parallel_for(
      mesh.numberOfNodes(), PUGS_LAMBDA(NodeId r) {
        Rdxd sum                                   = zero;
        const auto& node_to_cell                   = node_to_cell_matrix[r];
        const auto& node_local_number_in_its_cells = node_local_numbers_in_their_cells.itemArray(r);

        for (size_t j = 0; j < node_to_cell.size(); ++j) {
          const CellId J       = node_to_cell[j];
          const unsigned int R = node_local_number_in_its_cells[j];
          sum += Ajr(J, R);
        }
        Ar[r] = sum;
      });

    return Ar;
  }

  NodeValue<Rd>
  _computeBr(const Mesh<Dimension>& mesh,
             const NodeValuePerCell<const Rdxd>& Ajr,
             DiscreteFunctionDPk<Dimension, const Rd> DPk_u,
             NodeValuePerCell<const Rdxd> DPk_sigma) const
  {
    MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(mesh);

    const NodeValuePerCell<const Rd>& Cjr = mesh_data.Cjr();
    const auto& xr = mesh.xr();

    const auto& node_to_cell_matrix               = mesh.connectivity().nodeToCellMatrix();
    const auto& node_local_numbers_in_their_cells = mesh.connectivity().nodeLocalNumbersInTheirCells();

    NodeValue<Rd> b{mesh.connectivity()};

    parallel_for(
      mesh.numberOfNodes(), PUGS_LAMBDA(NodeId r) {
        const auto& node_to_cell                   = node_to_cell_matrix[r];
        const auto& node_local_number_in_its_cells = node_local_numbers_in_their_cells.itemArray(r);

        Rd br = zero;
        for (size_t j = 0; j < node_to_cell.size(); ++j) {
          const CellId J       = node_to_cell[j];
          const unsigned int R = node_local_number_in_its_cells[j];
          br += Ajr(J, R) * DPk_u[J](xr[r]) - DPk_sigma(J, R) * Cjr(J, R);
        }

        b[r] = br;
      });

    return b;
  }

  BoundaryConditionList
  _getBCList(const MeshType& mesh,
             const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list) const
  {
    BoundaryConditionList bc_list;

    for (const auto& bc_descriptor : bc_descriptor_list) {
      bool is_valid_boundary_condition = true;

      switch (bc_descriptor->type()) {
      case IBoundaryConditionDescriptor::Type::symmetry: {
        bc_list.emplace_back(
          SymmetryBoundaryCondition(getMeshFlatNodeBoundary(mesh, bc_descriptor->boundaryDescriptor())));
        break;
      }
      case IBoundaryConditionDescriptor::Type::fixed: {
        bc_list.emplace_back(FixedBoundaryCondition(getMeshNodeBoundary(mesh, bc_descriptor->boundaryDescriptor())));
        break;
      }
      case IBoundaryConditionDescriptor::Type::dirichlet: {
        const DirichletBoundaryConditionDescriptor& dirichlet_bc_descriptor =
          dynamic_cast<const DirichletBoundaryConditionDescriptor&>(*bc_descriptor);
        if (dirichlet_bc_descriptor.name() == "velocity") {
          MeshNodeBoundary mesh_node_boundary = getMeshNodeBoundary(mesh, dirichlet_bc_descriptor.boundaryDescriptor());

          Array<const Rd> value_list =
            InterpolateItemValue<Rd(Rd)>::template interpolate<ItemType::node>(dirichlet_bc_descriptor.rhsSymbolId(),
                                                                               mesh.xr(),
                                                                               mesh_node_boundary.nodeList());

          bc_list.emplace_back(VelocityBoundaryCondition{mesh_node_boundary, value_list});
        } else if (dirichlet_bc_descriptor.name() == "pressure") {
          const FunctionSymbolId pressure_id = dirichlet_bc_descriptor.rhsSymbolId();

          if constexpr (Dimension == 1) {
            MeshNodeBoundary mesh_node_boundary = getMeshNodeBoundary(mesh, bc_descriptor->boundaryDescriptor());

            Array<const double> node_values =
              InterpolateItemValue<double(Rd)>::template interpolate<ItemType::node>(pressure_id, mesh.xr(),
                                                                                     mesh_node_boundary.nodeList());

            bc_list.emplace_back(PressureBoundaryCondition{mesh_node_boundary, node_values});
          } else {
            MeshFaceBoundary mesh_face_boundary = getMeshFaceBoundary(mesh, bc_descriptor->boundaryDescriptor());

            MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(mesh);
            Array<const double> face_values =
              InterpolateItemValue<double(Rd)>::template interpolate<ItemType::face>(pressure_id, mesh_data.xl(),
                                                                                     mesh_face_boundary.faceList());
            bc_list.emplace_back(PressureBoundaryCondition{mesh_face_boundary, face_values});
          }

        } else if (dirichlet_bc_descriptor.name() == "normal-stress") {
          const FunctionSymbolId normal_stress_id = dirichlet_bc_descriptor.rhsSymbolId();

          if constexpr (Dimension == 1) {
            MeshNodeBoundary mesh_node_boundary = getMeshNodeBoundary(mesh, bc_descriptor->boundaryDescriptor());

            Array<const Rdxd> node_values =
              InterpolateItemValue<Rdxd(Rd)>::template interpolate<ItemType::node>(normal_stress_id, mesh.xr(),
                                                                                   mesh_node_boundary.nodeList());

            bc_list.emplace_back(NormalStressBoundaryCondition{mesh_node_boundary, node_values});
          } else {
            MeshFaceBoundary mesh_face_boundary = getMeshFaceBoundary(mesh, bc_descriptor->boundaryDescriptor());

            MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(mesh);
            Array<const Rdxd> face_values =
              InterpolateItemValue<Rdxd(Rd)>::template interpolate<ItemType::face>(normal_stress_id, mesh_data.xl(),
                                                                                   mesh_face_boundary.faceList());
            bc_list.emplace_back(NormalStressBoundaryCondition{mesh_face_boundary, face_values});
          }

        } else {
          is_valid_boundary_condition = false;
        }
        break;
      }
      case IBoundaryConditionDescriptor::Type::coupling: {
        const CouplingBoundaryConditionDescriptor& coupling_bc_descriptor =
          dynamic_cast<const CouplingBoundaryConditionDescriptor&>(*bc_descriptor);
        bc_list.emplace_back(CouplingBoundaryCondition(getMeshNodeBoundary(mesh, bc_descriptor->boundaryDescriptor()),
                                                       coupling_bc_descriptor.label()));
        break;
      }
      default: {
        is_valid_boundary_condition = false;
      }
      }
      if (not is_valid_boundary_condition) {
        std::ostringstream error_msg;
        error_msg << *bc_descriptor << " is an invalid boundary condition for hyperelastic solver";
        throw NormalError(error_msg.str());
      }
    }

    return bc_list;
  }

  void _applyPressureBC(const BoundaryConditionList& bc_list, const MeshType& mesh, NodeValue<Rd>& br) const;
  void _applyNormalStressBC(const BoundaryConditionList& bc_list, const MeshType& mesh, NodeValue<Rd>& br) const;
  void _applySymmetryBC(const BoundaryConditionList& bc_list, NodeValue<Rdxd>& Ar, NodeValue<Rd>& br) const;
  void _applyVelocityBC(const BoundaryConditionList& bc_list, NodeValue<Rdxd>& Ar, NodeValue<Rd>& br) const;
  void _applyCouplingBC(NodeValue<Rdxd>& Ar1,
                        NodeValue<Rdxd>& Ar2,
                        NodeValue<Rd>& br1,
                        NodeValue<Rd>& br2,
                        const std::vector<NodeId>& map1,
                        const std::vector<NodeId>& map2) const;
  void _applyCouplingBC(const MeshType& mesh,
                        NodeValue<Rd>& ur,
                        NodeValue<Rd>& CR_ur,
                        NodeValuePerCell<Rd>& Fjr,
                        NodeValuePerCell<Rd>& CR_Fjr,
                        const std::vector<NodeId>& map2) const;
  void _applyCouplingBC2(NodeValue<Rdxd>& Ar1,
                         NodeValue<Rdxd>& Ar2,
                         NodeValue<Rd>& ur1,
                         NodeValue<Rd>& ur2,
                         const std::vector<NodeId>& map1,
                         const std::vector<NodeId>& map2) const;
  void
  _applyBoundaryConditions(const BoundaryConditionList& bc_list,
                           const MeshType& mesh,
                           NodeValue<Rdxd>& Ar,
                           NodeValue<Rd>& br) const
  {
    this->_applyPressureBC(bc_list, mesh, br);
    this->_applyNormalStressBC(bc_list, mesh, br);
    this->_applySymmetryBC(bc_list, Ar, br);
    this->_applyVelocityBC(bc_list, Ar, br);
  }

  NodeValue<Rd>
  _computeUr(const MeshType& mesh, const NodeValue<Rdxd>& Ar, const NodeValue<Rd>& br) const
  {
    NodeValue<Rd> u{mesh.connectivity()};
    parallel_for(
      mesh.numberOfNodes(), PUGS_LAMBDA(NodeId r) { u[r] = inverse(Ar[r]) * br[r]; });

    return u;
  }

  NodeValuePerCell<Rd>
  _computeFjr(const MeshType& mesh,
              const NodeValuePerCell<const Rdxd>& Ajr,
              const NodeValue<const Rd>& ur,
              DiscreteFunctionDPk<Dimension,const Rd> DPk_uh,
              NodeValuePerCell<const Rdxd> DPk_sigma) const
  {
    MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(mesh);

    const NodeValuePerCell<const Rd> Cjr = mesh_data.Cjr();
    const NodeValue<const Rd>& xr = mesh.xr();

    const auto& node_to_cell_matrix               = mesh.connectivity().nodeToCellMatrix();
    const auto& node_local_numbers_in_their_cells = mesh.connectivity().nodeLocalNumbersInTheirCells();

    NodeValuePerCell<Rd> F{mesh.connectivity()};
    parallel_for(
      mesh.numberOfNodes(), PUGS_LAMBDA(NodeId r) {
        const auto& node_to_cell                   = node_to_cell_matrix[r];
        const auto& node_local_number_in_its_cells = node_local_numbers_in_their_cells.itemArray(r);

        for(size_t j = 0; j < node_to_cell.size(); ++j){
          const CellId J       = node_to_cell[j];
          const unsigned int R = node_local_number_in_its_cells[j];
          F(J,R) = -Ajr(J,R) * (DPk_uh[J](xr[r]) - ur[r]) + DPk_sigma(J,R) * Cjr(J,R);
        }
      });

    return F;
  }

  std::tuple<std::vector<NodeId>, std::vector<NodeId>>
  _computeMapping(std::shared_ptr<const MeshVariant>& i_mesh1,
                  std::shared_ptr<const MeshVariant>& i_mesh2,
                  const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list1,
                  const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list2) const
  {
    const MeshType& mesh1 = *i_mesh1->get<MeshType>();
    const MeshType& mesh2 = *i_mesh2->get<MeshType>();

    const BoundaryConditionList bc_list1 = this->_getBCList(mesh1, bc_descriptor_list1);
    const BoundaryConditionList bc_list2 = this->_getBCList(mesh2, bc_descriptor_list2);

    std::vector<NodeId> map1;
    std::vector<NodeId> map2;

    NodeValue<Rd> xr1 = copy(mesh1.xr());
    NodeValue<Rd> xr2 = copy(mesh2.xr());
    for (const auto& boundary_condition1 : bc_list1) {
      std::visit(
        [&](auto&& bc1) {
          using T1 = std::decay_t<decltype(bc1)>;
          if constexpr (std::is_same_v<CouplingBoundaryCondition, T1>) {
            const auto& node_list1 = bc1.nodeList();
            for (const auto& boundary_condition2 : bc_list2) {
              std::visit(
                [&](auto&& bc2) {
                  using T2 = std::decay_t<decltype(bc2)>;
                  if constexpr (std::is_same_v<CouplingBoundaryCondition, T2>) {
                    if (bc1.label() == bc2.label()) {
                      const auto& node_list2 = bc2.nodeList();
                      for (size_t i = 0; i < node_list1.size(); i++) {
                        NodeId node_id1 = node_list1[i];
                        NodeId node_id2;
                        for (size_t j = 0; j < node_list2.size(); j++) {
                          node_id2   = node_list2[j];
                          double err = 0;
                          err        = dot((xr1[node_id1] - xr2[node_id2]), (xr1[node_id1] - xr2[node_id2]));
                          if (sqrt(err) < 1e-14) {
                            map1.push_back(node_id1);
                            map2.push_back(node_id2);
                          }
                        }
                      }
                    }
                  }
                },
                boundary_condition2);
            }
          }
        },
        boundary_condition1);
    }
    return std::make_tuple(map1, map2);
  }

     void
  _vector_limiter(const MeshType& mesh,
                  const DiscreteFunctionP0<const Rd>& f,
                  DiscreteFunctionDPk<Dimension, Rd>& DPk_fh) const
  {
    MeshData<MeshType>& mesh_data = MeshDataManager::instance().getMeshData(mesh);
    StencilManager::BoundaryDescriptorList symmetry_boundary_descriptor_list;
    StencilDescriptor stencil_descriptor{1, StencilDescriptor::ConnectionType::by_nodes};
    auto stencil = StencilManager::instance().getCellToCellStencilArray(mesh.connectivity(), stencil_descriptor, symmetry_boundary_descriptor_list);

    const auto xj = mesh_data.xj();

    parallel_for(mesh.numberOfCells(), PUGS_LAMBDA(const CellId cell_id){
      const Rd fj = f[cell_id];

      Rd f_min = fj;
      Rd f_max = fj;

      const auto cell_stencil = stencil[cell_id];
      for(size_t i_cell = 0; i_cell < cell_stencil.size(); ++i_cell){
        for(size_t dim = 0; dim < Dimension; ++dim){
          f_min[dim] = std::min(f_min[dim], f[cell_stencil[i_cell]][dim]);
          f_max[dim] = std::max(f_max[dim], f[cell_stencil[i_cell]][dim]);
        }
      }

      Rd f_bar_min = fj;
      Rd f_bar_max = fj;

      for(size_t i_cell = 0; i_cell < cell_stencil.size(); ++i_cell){
        const CellId cell_k_id = cell_stencil[i_cell];
        const Rd f_xk    = DPk_fh[cell_id](xj[cell_k_id]);

        for(size_t dim = 0; dim < Dimension; ++dim){
          f_bar_min[dim] = std::min(f_bar_min[dim], f_xk[dim]);
          f_bar_max[dim] = std::max(f_bar_max[dim], f_xk[dim]);
        }
      }

      const double eps = 1E-14;
      Rd coef1;
      for(size_t dim = 0; dim < Dimension; ++dim){
        coef1[dim] = 1;
        if (std::abs(f_bar_max[dim] - fj[dim]) > eps) {
          coef1[dim] = (f_max[dim] - fj[dim]) / ((f_bar_max[dim] - fj[dim]));
        }
      }

      Rd coef2;
      for(size_t dim = 0; dim < Dimension; ++dim){
        coef2[dim] = 1;
        if (std::abs(f_bar_min[dim] - fj[dim]) > eps) {
          coef2[dim] = (f_min[dim] - fj[dim]) / ((f_bar_min[dim] - fj[dim]));
        }
      }

      double min_coef1 = coef1[0];
      double min_coef2 = coef2[0];
      for(size_t dim = 1; dim < Dimension; ++dim){
        min_coef1 = std::min(min_coef1,coef1[dim]);
        min_coef2 = std::min(min_coef2,coef2[dim]);
      }

      const double lambda = std::max(0., std::min(1., std::min(min_coef1, min_coef2)));

      auto coefficients = DPk_fh.coefficients(cell_id);

      coefficients[0] = (1 - lambda) * f[cell_id] + lambda * coefficients[0];
      for (size_t i = 1; i < coefficients.size(); ++i) {
        coefficients[i] *= lambda;
      }
    });
  }

  CellValue<double> 
  _scalar_limiter_coefficient(const MeshType& mesh,
                              const DiscreteFunctionP0<const double>& f,
                              const DiscreteFunctionDPk<Dimension, const double>& DPk_fh) const
  {
    MeshData<MeshType>& mesh_data = MeshDataManager::instance().getMeshData(mesh);
    StencilManager::BoundaryDescriptorList symmetry_boundary_descriptor_list;
    StencilDescriptor stencil_descriptor{1, StencilDescriptor::ConnectionType::by_nodes};
    auto stencil = StencilManager::instance().getCellToCellStencilArray(mesh.connectivity(), stencil_descriptor, symmetry_boundary_descriptor_list);

    const auto xj = mesh_data.xj();

    CellValue<double> lambda{mesh.connectivity()};

    parallel_for(mesh.numberOfCells(), PUGS_LAMBDA(const CellId cell_id){
      const double fj = f[cell_id];

      double f_min = fj;
      double f_max = fj;

      const auto cell_stencil = stencil[cell_id];
      for(size_t i_cell = 0; i_cell < cell_stencil.size(); ++i_cell){
        f_min = std::min(f_min, f[cell_stencil[i_cell]]);
        f_max = std::max(f_max, f[cell_stencil[i_cell]]);
      }

      double f_bar_min = fj;
      double f_bar_max = fj;

      for(size_t i_cell = 0; i_cell < cell_stencil.size(); ++i_cell){
        const CellId cell_k_id = cell_stencil[i_cell];
        const double f_xk    = DPk_fh[cell_id](xj[cell_k_id]);

        f_bar_min = std::min(f_bar_min, f_xk);
        f_bar_max = std::max(f_bar_max, f_xk);
      }

      const double eps = 1E-14;
      double coef1     = 1;
      if (std::abs(f_bar_max - fj) > eps) {
        coef1 = (f_max - fj) / ((f_bar_max - fj));
      }

      double coef2 = 1.;
      if (std::abs(f_bar_min - fj) > eps) {
        coef2 = (f_min - fj) / ((f_bar_min - fj));
      }

      lambda[cell_id] = std::max(0., std::min(1., std::min(coef1, coef2)));
    });

    return lambda;
  }

  NodeValuePerCell<Rdxd>
  _compute_tensor_reconstruction(const std::shared_ptr<const MeshVariant>& v_mesh,
                                 DiscreteTensorFunction sigma,
                                 PolynomialReconstructionDescriptor reconstruction_descriptor) const
    {
        const MeshType& mesh                = *v_mesh->get<MeshType>();

        CellValue<double> limiter_j{mesh.connectivity()};
        parallel_for(
          mesh.numberOfCells(), PUGS_LAMBDA(CellId j){
            limiter_j[j] = 1;
          }
        );

        const CellValue<const Rdxd>& sigma_j = copy(sigma.cellValues());
        std::vector<DiscreteFunctionDPk<Dimension, double>> sigma_coef;
        std::vector<size_t> row;    
        std::vector<size_t> col;
        for(size_t i = 0; i < Dimension; ++i){
            for(size_t k = i; k < Dimension; ++k){

                CellValue<double> coef{mesh.connectivity()};
                for(CellId j = 0; j <mesh.numberOfCells(); ++j){
                    coef[j] = sigma_j[j](i,k);
                }

                const auto& coef_scalar_function = DiscreteScalarFunction(v_mesh, coef);
                auto coef_v = std::make_shared<DiscreteFunctionVariant>(coef_scalar_function);
                auto reconstruction = PolynomialReconstruction{reconstruction_descriptor}.build(coef_v);
                auto DPk_coef = reconstruction[0]->get<DiscreteFunctionDPk<Dimension, const double>>();

                const CellValue<const double>& limiter_ik = _scalar_limiter_coefficient(mesh,coef_scalar_function,DPk_coef);
                parallel_for(
                  mesh.numberOfCells(), PUGS_LAMBDA(CellId j) {
                    limiter_j[j] = std::min(limiter_j[j],limiter_ik[j]);
                });

                sigma_coef.push_back(copy(DPk_coef));
                row.push_back(i);
                col.push_back(k);
            }
        }

        NodeValuePerCell<Rdxd> sigma_lim{mesh.connectivity()} ;
        const auto& cell_to_node_matrix = mesh.connectivity().cellToNodeMatrix();

        parallel_for(
          mesh.numberOfCells(), PUGS_LAMBDA(CellId j){
            const auto& cell_nodes = cell_to_node_matrix[j];
            for(size_t i = 0; i < Dimension; ++i){
              for(size_t k = 0; k < Dimension; ++k){
                for(size_t R = 0; R < cell_nodes.size(); ++R){
                  sigma_lim(j,R)(i,k) = 0;
                }
              }
            }
          }
        );

        const NodeValue<const Rd>& xr = copy(mesh.xr());
        parallel_for(
            mesh.numberOfCells(), PUGS_LAMBDA(CellId j) {
                const auto& cell_nodes = cell_to_node_matrix[j];

                for(size_t R = 0; R < cell_nodes.size(); ++R){
                    const NodeId r = cell_nodes[R];

                    for(size_t l = 0; l < sigma_coef.size(); ++l){
                        const size_t& i = row[l];
                        const size_t& k = col[l];

                        auto coefficients = sigma_coef[l].coefficients(j);

                        coefficients[0] = (1-limiter_j[j])*sigma_j[j](i,k) + limiter_j[j] * coefficients[0];
                        for (size_t indice = 1; indice < coefficients.size(); ++indice) {
                          coefficients[indice] *= limiter_j[j];
                        }

                        sigma_lim(j,R)(i,k) = sigma_coef[l][j](xr[r]);
                        if(i != k){
                          sigma_lim(j,R)(i,k) *= 2;
                        }
                    }

                    sigma_lim(j,R) += transpose(sigma_lim(j,R));
                    sigma_lim(j,R) *= 0.5;
                }

            }
        );

        return sigma_lim;
    }

  std::tuple<const std::shared_ptr<const DiscreteFunctionVariant>, const std::shared_ptr<const DiscreteFunctionVariant>, const std::shared_ptr<const DiscreteFunctionVariant>>
  _apply_NeoHook(const DiscreteScalarFunction rho, 
                  const DiscreteVectorFunction u, 
                  const DiscreteScalarFunction E, 
                  const DiscreteTensorFunction CG, 
                  const size_t& chi_solid,
                  const double& lambda,
                  const double& mu,
                  const double& gamma,
                  const double& p_inf) const
  {
    const TinyMatrix<Dimension> I = identity;

    const DiscreteScalarFunction& eps         = E - 0.5 * dot(u,u);
    const DiscreteScalarFunction& p_d         = (1-chi_solid)*(gamma-1) * rho * eps - p_inf*gamma;
    const DiscreteTensorFunction& sigma_d     = chi_solid * (mu / sqrt(det(CG)) * (CG - I) + lambda / sqrt(det(CG)) * log(sqrt(det(CG))) * I) - p_d * I;
    const DiscreteScalarFunction& aL_d        = sqrt(chi_solid * (lambda + 2 * mu) / rho + gamma * (p_d + p_inf) / rho);
    const DiscreteScalarFunction& aT_d        = sqrt(chi_solid * mu / rho);

    return {std::make_shared<DiscreteFunctionVariant>(sigma_d),
            std::make_shared<DiscreteFunctionVariant>(aL_d),
            std::make_shared<DiscreteFunctionVariant>(aT_d)};
  }

  std::tuple<const std::shared_ptr<const DiscreteFunctionVariant>, const std::shared_ptr<const DiscreteFunctionVariant>, const std::shared_ptr<const DiscreteFunctionVariant>>
  _apply_NeoHook2(const DiscreteScalarFunction rho, 
                  const DiscreteVectorFunction u, 
                  const DiscreteScalarFunction E, 
                  const DiscreteTensorFunction CG, 
                  const size_t& chi_solid,
                  const double& mu,
                  const double& gamma,
                  const double& p_inf) const
  {
    const TinyMatrix<Dimension> I = identity;

    const DiscreteScalarFunction& eps         = E - 0.5 * dot(u,u);
    const DiscreteScalarFunction& p_d         = (gamma-1) * rho * eps - p_inf*gamma;
    const DiscreteTensorFunction& sigma_d     = chi_solid * 2*mu/sqrt(det(CG))*(CG-1./3. * trace(CG)*I) - p_d * I;
    const DiscreteScalarFunction& aL_d        = sqrt(chi_solid * (2 * mu) / rho + ((1-chi_solid)*gamma * p_d + chi_solid * (gamma*(p_d+p_inf))) / rho);
    const DiscreteScalarFunction& aT_d        = sqrt(chi_solid * mu / rho);

    return {std::make_shared<DiscreteFunctionVariant>(sigma_d),
            std::make_shared<DiscreteFunctionVariant>(aL_d),
            std::make_shared<DiscreteFunctionVariant>(aT_d)};
  }

  std::tuple<const std::shared_ptr<const DiscreteFunctionVariant>, const std::shared_ptr<const DiscreteFunctionVariant>, const std::shared_ptr<const DiscreteFunctionVariant>>
  _apply_state_law(const size_t law,
                   const std::shared_ptr<const DiscreteFunctionVariant>& rho_v,
                   const std::shared_ptr<const DiscreteFunctionVariant>& u_v,
                   const std::shared_ptr<const DiscreteFunctionVariant>& E_v,
                   const std::shared_ptr<const DiscreteFunctionVariant>& CG_v,
                   const size_t& chi_solid,
                   const double& lambda,
                   const double& mu,
                   const double& gamma,
                   const double& p_inf) const 
  {
    const DiscreteScalarFunction& rho_d       = rho_v->get<DiscreteScalarFunction>();
    const DiscreteVectorFunction& u_d         = u_v->get<DiscreteVectorFunction>();
    const DiscreteScalarFunction& E_d         = E_v->get<DiscreteScalarFunction>();
    const DiscreteTensorFunction& CG_d        = CG_v->get<DiscreteTensorFunction>();

    if(law == 1){
      return _apply_NeoHook(rho_d,u_d,E_d,CG_d,chi_solid,lambda,mu,gamma,p_inf);
    } else {
      return _apply_NeoHook2(rho_d,u_d,E_d,CG_d,chi_solid,lambda,gamma,p_inf);
    }
  }

 public:
  std::tuple<const std::shared_ptr<const ItemValueVariant>, const std::shared_ptr<const SubItemValuePerItemVariant>>
  compute_fluxes(const SolverType& solver_type,
                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& aL_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& aT_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& u_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& sigma_v,
                 const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list) const
  {
    std::shared_ptr i_mesh = getCommonMesh({rho_v, aL_v, aT_v, u_v, sigma_v});
    if (not i_mesh) {
      throw NormalError("discrete functions are not defined on the same mesh");
    }

    if (not checkDiscretizationType({rho_v, aL_v, u_v, sigma_v}, DiscreteFunctionType::P0)) {
      throw NormalError("hyperelastic solver expects P0 functions");
    }

    const MeshType& mesh                = *i_mesh->get<MeshType>();
    const DiscreteScalarFunction& rho   = rho_v->get<DiscreteScalarFunction>();
    const DiscreteVectorFunction& u     = u_v->get<DiscreteVectorFunction>();
    const DiscreteScalarFunction& aL    = aL_v->get<DiscreteScalarFunction>();
    const DiscreteScalarFunction& aT    = aT_v->get<DiscreteScalarFunction>();
    const DiscreteTensorFunction& sigma = sigma_v->get<DiscreteTensorFunction>();

    std::vector<std::shared_ptr<const IBoundaryDescriptor>> symmetry_boundary_descriptor_list;

    for(auto&& bc_descriptor : bc_descriptor_list){
      if(bc_descriptor->type() == IBoundaryConditionDescriptor::Type::symmetry){
        symmetry_boundary_descriptor_list.push_back(bc_descriptor->boundaryDescriptor_shared());
      }
    }

    PolynomialReconstructionDescriptor reconstruction_descriptor(IntegrationMethodType::cell_center, 1,
                                                                symmetry_boundary_descriptor_list);

    NodeValuePerCell<Rdxd> sigma_lim = _compute_tensor_reconstruction(i_mesh,sigma,reconstruction_descriptor);
    auto reconstruction = PolynomialReconstruction{reconstruction_descriptor}.build(u_v);
    auto DPk_uh = reconstruction[0]->get<DiscreteFunctionDPk<Dimension, const Rd>>();

    DiscreteFunctionDPk<Dimension, Rd> u_lim  = copy(DPk_uh);
    this->_vector_limiter(mesh,u,u_lim);

    NodeValuePerCell<const Rdxd> Ajr = this->_computeAjr(solver_type, mesh, rho * aL, rho * aT);

    NodeValue<Rdxd> Ar = this->_computeAr(mesh, Ajr);
    NodeValue<Rd> br   = this->_computeBr(mesh, Ajr, u_lim, sigma_lim);

    const BoundaryConditionList bc_list = this->_getBCList(mesh, bc_descriptor_list);
    this->_applyBoundaryConditions(bc_list, mesh, Ar, br);

    synchronize(Ar);
    synchronize(br);

    NodeValue<const Rd> ur         = this->_computeUr(mesh, Ar, br);
    NodeValuePerCell<const Rd> Fjr = this->_computeFjr(mesh, Ajr, ur, u_lim, sigma_lim);

    return std::make_tuple(std::make_shared<const ItemValueVariant>(ur),
                           std::make_shared<const SubItemValuePerItemVariant>(Fjr));
  }

  std::tuple<const std::shared_ptr<const ItemValueVariant>,
             const std::shared_ptr<const SubItemValuePerItemVariant>,
             const std::shared_ptr<const ItemValueVariant>,
             const std::shared_ptr<const SubItemValuePerItemVariant>,
             NodeValue<Rd>,
             NodeValuePerCell<Rd>>
  compute_fluxes(const SolverType& solver_type,
                 const std::shared_ptr<const DiscreteFunctionVariant>& rho1_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& aL1_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& aT1_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& u1_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& sigma1_v,
                 const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list1,
                 const std::shared_ptr<const DiscreteFunctionVariant>& rho2_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& aL2_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& aT2_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& u2_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& sigma2_v,
                 const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list2,
                 const std::vector<NodeId>& map1,
                 const std::vector<NodeId>& map2) const
  {
    std::shared_ptr i_mesh1 = getCommonMesh({rho1_v, aL1_v, aT1_v, u1_v, sigma1_v});
    std::shared_ptr i_mesh2 = getCommonMesh({rho2_v, aL2_v, aT2_v, u2_v, sigma2_v});
    if (not i_mesh1) {
      throw NormalError("discrete functions are not defined on the same mesh");
    }

    if (not checkDiscretizationType({rho1_v, aL1_v, u1_v, sigma1_v}, DiscreteFunctionType::P0)) {
      throw NormalError("hyperelastic solver expects P0 functions");
    }
    if (not i_mesh2) {
      throw NormalError("discrete functions are not defined on the same mesh");
    }

    if (not checkDiscretizationType({rho2_v, aL2_v, u2_v, sigma2_v}, DiscreteFunctionType::P0)) {
      throw NormalError("acoustic solver expects P0 functions");
    }

    const MeshType& mesh1                = *i_mesh1->get<MeshType>();
    const DiscreteScalarFunction& rho1   = rho1_v->get<DiscreteScalarFunction>();
    const DiscreteVectorFunction& u1     = u1_v->get<DiscreteVectorFunction>();
    const DiscreteScalarFunction& aL1    = aL1_v->get<DiscreteScalarFunction>();
    const DiscreteScalarFunction& aT1    = aT1_v->get<DiscreteScalarFunction>();
    const DiscreteTensorFunction& sigma1 = sigma1_v->get<DiscreteTensorFunction>();

    const MeshType& mesh2                = *i_mesh2->get<MeshType>();
    const DiscreteScalarFunction& rho2   = rho2_v->get<DiscreteScalarFunction>();
    const DiscreteVectorFunction& u2     = u2_v->get<DiscreteVectorFunction>();
    const DiscreteScalarFunction& aL2    = aL2_v->get<DiscreteScalarFunction>();
    const DiscreteScalarFunction& aT2    = aT2_v->get<DiscreteScalarFunction>();
    const DiscreteTensorFunction& sigma2 = sigma2_v->get<DiscreteTensorFunction>();

    std::vector<std::shared_ptr<const IBoundaryDescriptor>> symmetry_boundary_descriptor_list1;
    std::vector<std::shared_ptr<const IBoundaryDescriptor>> symmetry_boundary_descriptor_list2;

    for(auto&& bc_descriptor : bc_descriptor_list1){
      if(bc_descriptor->type() == IBoundaryConditionDescriptor::Type::symmetry){
        symmetry_boundary_descriptor_list1.push_back(bc_descriptor->boundaryDescriptor_shared());
      }
    }
    for(auto&& bc_descriptor : bc_descriptor_list2){
      if(bc_descriptor->type() == IBoundaryConditionDescriptor::Type::symmetry){
        symmetry_boundary_descriptor_list2.push_back(bc_descriptor->boundaryDescriptor_shared());
      }
    }

    PolynomialReconstructionDescriptor reconstruction_descriptor1(IntegrationMethodType::cell_center, 1,
                                                                 symmetry_boundary_descriptor_list1);
    PolynomialReconstructionDescriptor reconstruction_descriptor2(IntegrationMethodType::cell_center, 1,
                                                                 symmetry_boundary_descriptor_list2);

    NodeValuePerCell<Rdxd> sigma1_lim = _compute_tensor_reconstruction(i_mesh1,sigma1,reconstruction_descriptor1);
    NodeValuePerCell<Rdxd> sigma2_lim = _compute_tensor_reconstruction(i_mesh2,sigma2,reconstruction_descriptor2);

    auto reconstruction1 = PolynomialReconstruction{reconstruction_descriptor1}.build(u1_v);
    auto DPk_uh1 = reconstruction1[0]->get<DiscreteFunctionDPk<Dimension, const Rd>>();
    auto reconstruction2 = PolynomialReconstruction{reconstruction_descriptor2}.build(u2_v);
    auto DPk_uh2 = reconstruction2[0]->get<DiscreteFunctionDPk<Dimension, const Rd>>();

    DiscreteFunctionDPk<Dimension, Rd> u1_lim  = copy(DPk_uh1);
    this->_vector_limiter(mesh1,u1,u1_lim);
    DiscreteFunctionDPk<Dimension, Rd> u2_lim  = copy(DPk_uh2);
    this->_vector_limiter(mesh2,u2,u2_lim);

    NodeValuePerCell<const Rdxd> Ajr1 = this->_computeAjr(solver_type, mesh1, rho1 * aL1, rho1 * aT1);
    NodeValuePerCell<const Rdxd> Ajr2 = this->_computeAjr(solver_type, mesh2, rho2 * aL2, rho2 * aT2);

    NodeValue<Rdxd> Ar1 = this->_computeAr(mesh1, Ajr1);
    NodeValue<Rd> br1   = this->_computeBr(mesh1, Ajr1, u1_lim, sigma1_lim);
    NodeValue<Rdxd> Ar2 = this->_computeAr(mesh2, Ajr2);
    NodeValue<Rd> br2   = this->_computeBr(mesh2, Ajr2, u2_lim, sigma2_lim);
    this->_applyCouplingBC(Ar1, Ar2, br1, br2, map1, map2);

    const BoundaryConditionList bc_list1 = this->_getBCList(mesh1, bc_descriptor_list1);
    const BoundaryConditionList bc_list2 = this->_getBCList(mesh2, bc_descriptor_list2);
    this->_applyBoundaryConditions(bc_list1, mesh1, Ar1, br1);
    this->_applyBoundaryConditions(bc_list2, mesh2, Ar2, br2);

    synchronize(Ar1);
    synchronize(br1);
    synchronize(Ar2);
    synchronize(br2);

    NodeValue<Rd> ur1           = this->_computeUr(mesh1, Ar1, br1);
    NodeValuePerCell<Rd> Fjr1   = this->_computeFjr(mesh1, Ajr1, ur1, u1_lim, sigma1_lim);
    NodeValue<Rd> ur2           = this->_computeUr(mesh2, Ar2, br2);
    NodeValuePerCell<Rd> Fjr2   = this->_computeFjr(mesh2, Ajr2, ur2, u2_lim, sigma2_lim);
    NodeValue<Rd> CR_ur         = this->_computeUr(mesh2, Ar2, br2);
    NodeValuePerCell<Rd> CR_Fjr = this->_computeFjr(mesh2, Ajr2, ur2, u2_lim, sigma2_lim);

    return std::make_tuple(std::make_shared<const ItemValueVariant>(ur1),
                           std::make_shared<const SubItemValuePerItemVariant>(Fjr1),
                           std::make_shared<const ItemValueVariant>(ur2),
                           std::make_shared<const SubItemValuePerItemVariant>(Fjr2), CR_ur, CR_Fjr);
  }

  std::tuple<const std::shared_ptr<const ItemValueVariant>, const std::shared_ptr<const SubItemValuePerItemVariant>>
  compute_fluxes(const SolverType& solver_type,
                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& aL_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& aT_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& u_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& sigma_v,
                 const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list,
                 NodeValue<Rd> CR_ur,
                 NodeValuePerCell<Rd> CR_Fjr,
                 const std::vector<NodeId> map2) const
  {
    std::shared_ptr i_mesh = getCommonMesh({rho_v, aL_v, aT_v, u_v, sigma_v});
    if (not i_mesh) {
      throw NormalError("discrete functions are not defined on the same mesh ici1");
    }

    if (not checkDiscretizationType({rho_v, aL_v, u_v, sigma_v}, DiscreteFunctionType::P0)) {
      throw NormalError("hyperelastic solver expects P0 functions");
    }

    const MeshType& mesh                = *i_mesh->get<MeshType>();
    const DiscreteScalarFunction& rho   = rho_v->get<DiscreteScalarFunction>();
    const DiscreteVectorFunction& u     = u_v->get<DiscreteVectorFunction>();
    const DiscreteScalarFunction& aL    = aL_v->get<DiscreteScalarFunction>();
    const DiscreteScalarFunction& aT    = aT_v->get<DiscreteScalarFunction>();
    const DiscreteTensorFunction& sigma = sigma_v->get<DiscreteTensorFunction>();

    std::vector<std::shared_ptr<const IBoundaryDescriptor>> symmetry_boundary_descriptor_list;

    for(auto&& bc_descriptor : bc_descriptor_list){
      if(bc_descriptor->type() == IBoundaryConditionDescriptor::Type::symmetry){
        symmetry_boundary_descriptor_list.push_back(bc_descriptor->boundaryDescriptor_shared());
      }
    }

    PolynomialReconstructionDescriptor reconstruction_descriptor(IntegrationMethodType::cell_center, 1,
                                                                symmetry_boundary_descriptor_list);

    NodeValuePerCell<Rdxd> sigma_lim = _compute_tensor_reconstruction(i_mesh,sigma,reconstruction_descriptor);
    auto reconstruction = PolynomialReconstruction{reconstruction_descriptor}.build(u_v);
    auto DPk_uh = reconstruction[0]->get<DiscreteFunctionDPk<Dimension, const Rd>>();

    DiscreteFunctionDPk<Dimension, Rd> u_lim  = copy(DPk_uh);
    this->_vector_limiter(mesh,u,u_lim);

    NodeValuePerCell<const Rdxd> Ajr = this->_computeAjr(solver_type, mesh, rho * aL, rho * aT);

    NodeValue<Rdxd> Ar = this->_computeAr(mesh, Ajr);
    NodeValue<Rd> br   = this->_computeBr(mesh, Ajr, u_lim, sigma_lim);

    const BoundaryConditionList bc_list = this->_getBCList(mesh, bc_descriptor_list);
    this->_applyBoundaryConditions(bc_list, mesh, Ar, br);

    synchronize(Ar);
    synchronize(br);

    NodeValue<Rd> ur         = this->_computeUr(mesh, Ar, br);
    NodeValuePerCell<Rd> Fjr = this->_computeFjr(mesh, Ajr, ur, u_lim, sigma_lim);

    this->_applyCouplingBC(mesh, ur, CR_ur, Fjr, CR_Fjr, map2);

    return std::make_tuple(std::make_shared<const ItemValueVariant>(ur),
                           std::make_shared<const SubItemValuePerItemVariant>(Fjr));
  }

    std::tuple<const std::shared_ptr<const ItemValueVariant>,
             const std::shared_ptr<const SubItemValuePerItemVariant>,
             const std::shared_ptr<const ItemValueVariant>,
             const std::shared_ptr<const SubItemValuePerItemVariant>,
             NodeValue<Rd>,
             NodeValuePerCell<Rd>>
  compute_fluxes2(const SolverType& solver_type,
                 const std::shared_ptr<const DiscreteFunctionVariant>& rho1_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& aL1_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& aT1_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& u1_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& sigma1_v,
                 const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list1,
                 const std::shared_ptr<const DiscreteFunctionVariant>& rho2_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& aL2_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& aT2_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& u2_v,
                 const std::shared_ptr<const DiscreteFunctionVariant>& sigma2_v,
                 const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list2,
                 const std::vector<NodeId>& map1,
                 const std::vector<NodeId>& map2) const
  {
    std::shared_ptr i_mesh1 = getCommonMesh({rho1_v, aL1_v, aT1_v, u1_v, sigma1_v});
    std::shared_ptr i_mesh2 = getCommonMesh({rho2_v, aL2_v, aT2_v, u2_v, sigma2_v});
    if (not i_mesh1) {
      throw NormalError("discrete functions are not defined on the same mesh ici2");
    }

    if (not checkDiscretizationType({rho1_v, aL1_v, u1_v, sigma1_v}, DiscreteFunctionType::P0)) {
      throw NormalError("hyperelastic solver expects P0 functions");
    }
    if (not i_mesh2) {
      throw NormalError("discrete functions are not defined on the same mesh ici2 mesh2");
    }

    if (not checkDiscretizationType({rho2_v, aL2_v, u2_v, sigma2_v}, DiscreteFunctionType::P0)) {
      throw NormalError("acoustic solver expects P0 functions");
    }

    const MeshType& mesh1                = *i_mesh1->get<MeshType>();
    const DiscreteScalarFunction& rho1   = rho1_v->get<DiscreteScalarFunction>();
    const DiscreteVectorFunction& u1     = u1_v->get<DiscreteVectorFunction>();
    const DiscreteScalarFunction& aL1    = aL1_v->get<DiscreteScalarFunction>();
    const DiscreteScalarFunction& aT1    = aT1_v->get<DiscreteScalarFunction>();
    const DiscreteTensorFunction& sigma1 = sigma1_v->get<DiscreteTensorFunction>();

    const MeshType& mesh2                = *i_mesh2->get<MeshType>();
    const DiscreteScalarFunction& rho2   = rho2_v->get<DiscreteScalarFunction>();
    const DiscreteVectorFunction& u2     = u2_v->get<DiscreteVectorFunction>();
    const DiscreteScalarFunction& aL2    = aL2_v->get<DiscreteScalarFunction>();
    const DiscreteScalarFunction& aT2    = aT2_v->get<DiscreteScalarFunction>();
    const DiscreteTensorFunction& sigma2 = sigma2_v->get<DiscreteTensorFunction>();

    std::vector<std::shared_ptr<const IBoundaryDescriptor>> symmetry_boundary_descriptor_list1;
    std::vector<std::shared_ptr<const IBoundaryDescriptor>> symmetry_boundary_descriptor_list2;

    for(auto&& bc_descriptor : bc_descriptor_list1){
      if(bc_descriptor->type() == IBoundaryConditionDescriptor::Type::symmetry){
        symmetry_boundary_descriptor_list1.push_back(bc_descriptor->boundaryDescriptor_shared());
      }
    }
    for(auto&& bc_descriptor : bc_descriptor_list2){
      if(bc_descriptor->type() == IBoundaryConditionDescriptor::Type::symmetry){
        symmetry_boundary_descriptor_list2.push_back(bc_descriptor->boundaryDescriptor_shared());
      }
    }

    PolynomialReconstructionDescriptor reconstruction_descriptor1(IntegrationMethodType::cell_center, 1,
                                                                 symmetry_boundary_descriptor_list1);
    PolynomialReconstructionDescriptor reconstruction_descriptor2(IntegrationMethodType::cell_center, 1,
                                                                 symmetry_boundary_descriptor_list2);

    NodeValuePerCell<Rdxd> sigma1_lim = _compute_tensor_reconstruction(i_mesh1,sigma1,reconstruction_descriptor1);
    NodeValuePerCell<Rdxd> sigma2_lim = _compute_tensor_reconstruction(i_mesh2,sigma2,reconstruction_descriptor2);

    auto reconstruction1 = PolynomialReconstruction{reconstruction_descriptor1}.build(u1_v);
    auto DPk_uh1 = reconstruction1[0]->get<DiscreteFunctionDPk<Dimension, const Rd>>();
    auto reconstruction2 = PolynomialReconstruction{reconstruction_descriptor2}.build(u2_v);
    auto DPk_uh2 = reconstruction2[0]->get<DiscreteFunctionDPk<Dimension, const Rd>>();

    DiscreteFunctionDPk<Dimension, Rd> u1_lim  = copy(DPk_uh1);
    this->_vector_limiter(mesh1,u1,u1_lim);
    DiscreteFunctionDPk<Dimension, Rd> u2_lim  = copy(DPk_uh2);
    this->_vector_limiter(mesh2,u2,u2_lim);

    NodeValuePerCell<const Rdxd> Ajr1 = this->_computeAjr(solver_type, mesh1, rho1 * aL1, rho1 * aT1);
    NodeValuePerCell<const Rdxd> Ajr2 = this->_computeAjr(solver_type, mesh2, rho2 * aL2, rho2 * aT2);

    NodeValue<Rdxd> Ar1 = this->_computeAr(mesh1, Ajr1);
    NodeValue<Rd> br1   = this->_computeBr(mesh1, Ajr1, u1_lim, sigma1_lim);
    NodeValue<Rdxd> Ar2 = this->_computeAr(mesh2, Ajr2);
    NodeValue<Rd> br2   = this->_computeBr(mesh2, Ajr2, u2_lim, sigma2_lim);

    const BoundaryConditionList bc_list1 = this->_getBCList(mesh1, bc_descriptor_list1);
    const BoundaryConditionList bc_list2 = this->_getBCList(mesh2, bc_descriptor_list2);
    this->_applyBoundaryConditions(bc_list1, mesh1, Ar1, br1);
    this->_applyBoundaryConditions(bc_list2, mesh2, Ar2, br2);

    synchronize(Ar1);
    synchronize(br1);
    synchronize(Ar2);
    synchronize(br2);

    NodeValue<Rd> ur1           = this->_computeUr(mesh1, Ar1, br1);
    NodeValue<Rd> ur2           = this->_computeUr(mesh2, Ar2, br2);
    this->_applyCouplingBC2(Ar1,Ar2,ur1,ur2,map1,map2);
    NodeValuePerCell<Rd> Fjr1   = this->_computeFjr(mesh1, Ajr1, ur1, u1_lim, sigma1_lim);
    NodeValuePerCell<Rd> Fjr2   = this->_computeFjr(mesh2, Ajr2, ur2, u2_lim, sigma2_lim);

    return std::make_tuple(std::make_shared<const ItemValueVariant>(ur1),
                           std::make_shared<const SubItemValuePerItemVariant>(Fjr1),
                           std::make_shared<const ItemValueVariant>(ur2),
                           std::make_shared<const SubItemValuePerItemVariant>(Fjr2),
                           ur2,
                           Fjr2);
  }

  std::tuple<std::shared_ptr<const MeshVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>>
  apply_fluxes(const double& dt,
               const MeshType& mesh,
               const DiscreteScalarFunction& rho,
               const DiscreteVectorFunction& u,
               const DiscreteScalarFunction& E,
               const DiscreteTensorFunction& CG,
               const NodeValue<const Rd>& ur,
               const NodeValuePerCell<const Rd>& Fjr) const
  {
    const auto& cell_to_node_matrix = mesh.connectivity().cellToNodeMatrix();

    if ((mesh.shared_connectivity() != ur.connectivity_ptr()) or
        (mesh.shared_connectivity() != Fjr.connectivity_ptr())) {
      throw NormalError("fluxes are not defined on the same connectivity than the mesh");
    }

    NodeValue<Rd> new_xr = copy(mesh.xr());
    parallel_for(
      mesh.numberOfNodes(), PUGS_LAMBDA(NodeId r) { new_xr[r] += dt * ur[r]; });

    std::shared_ptr<const MeshType> new_mesh = std::make_shared<MeshType>(mesh.shared_connectivity(), new_xr);

    CellValue<const double> Vj           = MeshDataManager::instance().getMeshData(mesh).Vj();
    const NodeValuePerCell<const Rd> Cjr = MeshDataManager::instance().getMeshData(mesh).Cjr();

    CellValue<double> new_rho = copy(rho.cellValues());
    CellValue<Rd> new_u       = copy(u.cellValues());
    CellValue<double> new_E   = copy(E.cellValues());
    CellValue<Rdxd> new_CG    = copy(CG.cellValues());

    parallel_for(
      mesh.numberOfCells(), PUGS_LAMBDA(CellId j) {
        const auto& cell_nodes = cell_to_node_matrix[j];

        Rd momentum_fluxes   = zero;
        double energy_fluxes = 0;
        Rdxd gradv           = zero;
        for (size_t R = 0; R < cell_nodes.size(); ++R) {
          const NodeId r = cell_nodes[R];
          gradv += tensorProduct(ur[r], Cjr(j, R));
          momentum_fluxes += Fjr(j, R);
          energy_fluxes += dot(Fjr(j, R), ur[r]);
        }
        const Rdxd cauchy_green_fluxes = gradv * CG[j] + CG[j] * transpose(gradv);
        const double dt_over_Mj        = dt / (rho[j] * Vj[j]);
        const double dt_over_Vj        = dt / Vj[j];
        new_u[j] += dt_over_Mj * momentum_fluxes;
        new_E[j] += dt_over_Mj * energy_fluxes;
        new_CG[j] += dt_over_Vj * cauchy_green_fluxes;
        new_CG[j] += transpose(new_CG[j]);
        new_CG[j] *= 0.5;
      });

    CellValue<const double> new_Vj = MeshDataManager::instance().getMeshData(*new_mesh).Vj();

    parallel_for(
      mesh.numberOfCells(), PUGS_LAMBDA(CellId j) { new_rho[j] *= Vj[j] / new_Vj[j]; });

    return {std::make_shared<MeshVariant>(new_mesh),
            std::make_shared<DiscreteFunctionVariant>(DiscreteScalarFunction(new_mesh, new_rho)),
            std::make_shared<DiscreteFunctionVariant>(DiscreteVectorFunction(new_mesh, new_u)),
            std::make_shared<DiscreteFunctionVariant>(DiscreteScalarFunction(new_mesh, new_E)),
            std::make_shared<DiscreteFunctionVariant>(DiscreteTensorFunction(new_mesh, new_CG))};
  }

  std::tuple<std::shared_ptr<const MeshVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>>
  apply_fluxes(const double& dt,
               const std::shared_ptr<const DiscreteFunctionVariant>& rho,
               const std::shared_ptr<const DiscreteFunctionVariant>& u,
               const std::shared_ptr<const DiscreteFunctionVariant>& E,
               const std::shared_ptr<const DiscreteFunctionVariant>& CG,
               const std::shared_ptr<const ItemValueVariant>& ur,
               const std::shared_ptr<const SubItemValuePerItemVariant>& Fjr) const
  {
    std::shared_ptr i_mesh = getCommonMesh({rho, u, E});
    if (not i_mesh) {
      throw NormalError("discrete functions are not defined on the same mesh ici3");
    }

    if (not checkDiscretizationType({rho, u, E}, DiscreteFunctionType::P0)) {
      throw NormalError("hyperelastic solver expects P0 functions");
    }

    return this->apply_fluxes(dt,                                   //
                              *i_mesh->get<MeshType>(),             //
                              rho->get<DiscreteScalarFunction>(),   //
                              u->get<DiscreteVectorFunction>(),     //
                              E->get<DiscreteScalarFunction>(),     //
                              CG->get<DiscreteTensorFunction>(),    //
                              ur->get<NodeValue<const Rd>>(),       //
                              Fjr->get<NodeValuePerCell<const Rd>>());
  }

  std::tuple<std::shared_ptr<const MeshVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>>
  order2_apply_fluxes(const double& dt,
               const MeshType& mesh,
               const DiscreteScalarFunction& rho,
               const DiscreteVectorFunction& u,
               const DiscreteScalarFunction& E,
               const DiscreteTensorFunction& CG,
               const DiscreteScalarFunction& rho_app,
               const DiscreteTensorFunction& CG_app,
               const NodeValue<const Rd>& ur,
               const NodeValuePerCell<const Rd>& Fjr) const
  {
    const auto& cell_to_node_matrix = mesh.connectivity().cellToNodeMatrix();

    if ((mesh.shared_connectivity() != ur.connectivity_ptr()) or
        (mesh.shared_connectivity() != Fjr.connectivity_ptr())) {
      throw NormalError("fluxes are not defined on the same connectivity than the mesh");
    }

    NodeValue<Rd> new_xr = copy(mesh.xr());
    parallel_for(
      mesh.numberOfNodes(), PUGS_LAMBDA(NodeId r) { new_xr[r] += dt * ur[r]; });

    std::shared_ptr<const MeshType> new_mesh = std::make_shared<MeshType>(mesh.shared_connectivity(), new_xr);

    CellValue<const double> Vj           = MeshDataManager::instance().getMeshData(mesh).Vj();
    const NodeValuePerCell<const Rd> Cjr = MeshDataManager::instance().getMeshData(mesh).Cjr();

    CellValue<double> new_rho = copy(rho.cellValues());
    CellValue<Rd> new_u       = copy(u.cellValues());
    CellValue<double> new_E   = copy(E.cellValues());
    CellValue<Rdxd> new_CG    = copy(CG.cellValues());

    parallel_for(
      mesh.numberOfCells(), PUGS_LAMBDA(CellId j) {
        const auto& cell_nodes = cell_to_node_matrix[j];

        Rd momentum_fluxes   = zero;
        double energy_fluxes = 0;
        Rdxd gradv           = zero;
        for (size_t R = 0; R < cell_nodes.size(); ++R) {
          const NodeId r = cell_nodes[R];
          gradv += tensorProduct(ur[r], Cjr(j, R));
          momentum_fluxes += Fjr(j, R);
          energy_fluxes += dot(Fjr(j, R), ur[r]);
        }
        const Rdxd cauchy_green_fluxes = rho_app[j] * (gradv * CG_app[j] + CG_app[j] * transpose(gradv));
        const double dt_over_Mj        = dt / (rho[j] * Vj[j]);
        new_u[j] += dt_over_Mj * momentum_fluxes;
        new_E[j] += dt_over_Mj * energy_fluxes;
        new_CG[j] += dt_over_Mj * cauchy_green_fluxes;
        new_CG[j] += transpose(new_CG[j]);
        new_CG[j] *= 0.5;
      });

    CellValue<const double> new_Vj = MeshDataManager::instance().getMeshData(*new_mesh).Vj();

    parallel_for(
      mesh.numberOfCells(), PUGS_LAMBDA(CellId j) { new_rho[j] *= Vj[j] / new_Vj[j]; });

    return {std::make_shared<MeshVariant>(new_mesh),
            std::make_shared<DiscreteFunctionVariant>(DiscreteScalarFunction(new_mesh, new_rho)),
            std::make_shared<DiscreteFunctionVariant>(DiscreteVectorFunction(new_mesh, new_u)),
            std::make_shared<DiscreteFunctionVariant>(DiscreteScalarFunction(new_mesh, new_E)),
            std::make_shared<DiscreteFunctionVariant>(DiscreteTensorFunction(new_mesh, new_CG))};
  }

  std::tuple<std::shared_ptr<const MeshVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>>
  order2_apply_fluxes(const double& dt,
               const std::shared_ptr<const DiscreteFunctionVariant>& rho,
               const std::shared_ptr<const DiscreteFunctionVariant>& u,
               const std::shared_ptr<const DiscreteFunctionVariant>& E,
               const std::shared_ptr<const DiscreteFunctionVariant>& CG,
               const std::shared_ptr<const DiscreteFunctionVariant>& rho_app,
               const std::shared_ptr<const DiscreteFunctionVariant>& CG_app,
               const std::shared_ptr<const ItemValueVariant>& ur,
               const std::shared_ptr<const SubItemValuePerItemVariant>& Fjr) const
  {
    std::shared_ptr mesh_v = getCommonMesh({rho, u, E});
    if (not mesh_v) {
      throw NormalError("discrete functions are not defined on the same mesh ici4");
    }

    if (not checkDiscretizationType({rho, u, E}, DiscreteFunctionType::P0)) {
      throw NormalError("hyperelastic solver expects P0 functions");
    }

    return this->order2_apply_fluxes(dt,                                   //
                              *mesh_v->get<MeshType>(),             //
                              rho->get<DiscreteScalarFunction>(),   //
                              u->get<DiscreteVectorFunction>(),     //
                              E->get<DiscreteScalarFunction>(),     //
                              CG->get<DiscreteTensorFunction>(),    //
                              rho_app->get<DiscreteScalarFunction>(),   //
                              CG_app->get<DiscreteTensorFunction>(),    //
                              ur->get<NodeValue<const Rd>>(),       //
                              Fjr->get<NodeValuePerCell<const Rd>>());
  }

  std::tuple<std::shared_ptr<const MeshVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>>
  mesh_correction(const MeshType& mesh1,
                  const MeshType& mesh2,
                  const DiscreteScalarFunction& rho,
                  const DiscreteVectorFunction& u,
                  const DiscreteScalarFunction& E,
                  const DiscreteTensorFunction& CG,
                  std::vector<NodeId>& map1,
                  std::vector<NodeId>& map2) const
  {
    NodeValue<Rd> xr1     = copy(mesh1.xr());
    NodeValue<Rd> new_xr2 = copy(mesh2.xr());

    size_t n = map1.size();

    for (size_t i = 0; i < n; i++) {
      for (size_t j = 0; j < Dimension; j++) {
        new_xr2[map2[i]][j] = xr1[map1[i]][j];
      }
    }

    std::shared_ptr<const MeshType> new_mesh2 = std::make_shared<MeshType>(mesh2.shared_connectivity(), new_xr2);

    CellValue<double> new_rho = copy(rho.cellValues());
    CellValue<Rd> new_u       = copy(u.cellValues());
    CellValue<double> new_E   = copy(E.cellValues());
    CellValue<Rdxd> new_CG    = copy(CG.cellValues());

    return {std::make_shared<MeshVariant>(new_mesh2),
            std::make_shared<DiscreteFunctionVariant>(DiscreteScalarFunction(new_mesh2, new_rho)),
            std::make_shared<DiscreteFunctionVariant>(DiscreteVectorFunction(new_mesh2, new_u)),
            std::make_shared<DiscreteFunctionVariant>(DiscreteScalarFunction(new_mesh2, new_E)),
            std::make_shared<DiscreteFunctionVariant>(DiscreteTensorFunction(new_mesh2, new_CG))};
  }

  std::tuple<std::shared_ptr<const MeshVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>>
  mesh_correction(const std::shared_ptr<const MeshVariant>& i_mesh1,
                  const std::shared_ptr<const MeshVariant>& i_mesh2,
                  const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                  const std::shared_ptr<const DiscreteFunctionVariant>& u,
                  const std::shared_ptr<const DiscreteFunctionVariant>& E,
                  const std::shared_ptr<const DiscreteFunctionVariant>& CG,
                  std::vector<NodeId>& map1,
                  std::vector<NodeId>& map2) const
  {
    return this->mesh_correction(*i_mesh1->get<MeshType>(), *i_mesh2->get<MeshType>(),
                                 rho->get<DiscreteScalarFunction>(), u->get<DiscreteVectorFunction>(),
                                 E->get<DiscreteScalarFunction>(), CG->get<DiscreteTensorFunction>(), map1, map2);
  }

  std::tuple<std::shared_ptr<const MeshVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,             
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>>            
  _compute_classic_midpoint_method(const SolverType& solver_type,
                                   const size_t& law,
                                   const double& dt1,
                                   const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                   const std::shared_ptr<const DiscreteFunctionVariant>& u,
                                   const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                   const std::shared_ptr<const DiscreteFunctionVariant>& CG,
                                   const std::shared_ptr<const DiscreteFunctionVariant>& aL,
                                   const std::shared_ptr<const DiscreteFunctionVariant>& aT,
                                   const std::shared_ptr<const DiscreteFunctionVariant>& sigma,                               
                                   const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list,
                                   NodeValue<Rd> CR_ur,
                                   NodeValuePerCell<Rd> CR_Fjr,
                                   std::vector<NodeId>& map2,
                                   size_t chi_solid,
                                   const double& lambda,
                                   const double& mu,
                                   const double& gamma,
                                   const double& p_inf) const
  {
    
    std::shared_ptr m_app = getCommonMesh({rho, aL, aT, u, sigma});
    std::shared_ptr<const DiscreteFunctionVariant> rho_app = rho;
    std::shared_ptr<const DiscreteFunctionVariant> u_app   = u;
    std::shared_ptr<const DiscreteFunctionVariant> E_app   = E;
    std::shared_ptr<const DiscreteFunctionVariant> CG_app  = CG;

    std::shared_ptr new_m = getCommonMesh({rho, aL, aT, u, sigma});
    std::shared_ptr<const DiscreteFunctionVariant> new_rho = rho;
    std::shared_ptr<const DiscreteFunctionVariant> new_u   = u;
    std::shared_ptr<const DiscreteFunctionVariant> new_E   = E;
    std::shared_ptr<const DiscreteFunctionVariant> new_CG  = CG;
    std::shared_ptr<const DiscreteFunctionVariant> new_sigma = sigma;
    std::shared_ptr<const DiscreteFunctionVariant> new_aL   = aL;
    std::shared_ptr<const DiscreteFunctionVariant> new_aT   = aT;

    double sum_dt = 0;

    while(sum_dt < dt1){
      
      DiscreteScalarFunction new_aL_d = new_aL->get<DiscreteScalarFunction>();
      DiscreteScalarFunction new_aT_d = new_aT->get<DiscreteScalarFunction>();
      const std::shared_ptr<const DiscreteFunctionVariant>& new_c = std::make_shared<const DiscreteFunctionVariant>(new_aL_d + new_aT_d);

      double dt2 = 0.4 * hyperelastic_dt(new_c);
      if(sum_dt + dt2 > dt1){
        dt2 = dt1 - sum_dt;
      }

      auto [ur, Fjr] = compute_fluxes(solver_type,new_rho,new_aL,new_aT,new_u,new_sigma,bc_descriptor_list,CR_ur,CR_Fjr,map2);
      std::tie(m_app, rho_app, u_app, E_app, CG_app) = apply_fluxes(dt2/2,new_rho,new_u,new_E,new_CG,ur,Fjr);

      const auto [sigma_app, aL_app, aT_app] = _apply_state_law(law,rho_app,u_app,E_app,CG_app,chi_solid,lambda,mu,gamma,p_inf);

      auto [ur_app, Fjr_app] = compute_fluxes(solver_type,rho_app,aL_app,aT_app,u_app,sigma_app,bc_descriptor_list,CR_ur,CR_Fjr,map2);
      std::tie(new_m,new_rho,new_u,new_E,new_CG) = order2_apply_fluxes(dt2,new_rho,new_u,new_E,new_CG,rho_app,CG_app,ur_app,Fjr_app); 

      std::tie(new_sigma,new_aL,new_aT) = _apply_state_law(law,new_rho,new_u,new_E,new_CG,chi_solid,lambda,mu,gamma,p_inf);
        
      sum_dt += dt2;

    }

    return {new_m, new_rho, new_u, new_E, new_CG};

  }

  std::tuple<std::shared_ptr<const MeshVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const MeshVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>>
  apply(const SolverType& solver_type,
        const double& dt1,
        const size_t& q,
        const std::shared_ptr<const DiscreteFunctionVariant>& rho1,
        const std::shared_ptr<const DiscreteFunctionVariant>& rho2,
        const std::shared_ptr<const DiscreteFunctionVariant>& u1,
        const std::shared_ptr<const DiscreteFunctionVariant>& u2,
        const std::shared_ptr<const DiscreteFunctionVariant>& E1,
        const std::shared_ptr<const DiscreteFunctionVariant>& E2,
        const std::shared_ptr<const DiscreteFunctionVariant>& CG1,
        const std::shared_ptr<const DiscreteFunctionVariant>& CG2,
        const std::shared_ptr<const DiscreteFunctionVariant>& aL1,
        const std::shared_ptr<const DiscreteFunctionVariant>& aL2,
        const std::shared_ptr<const DiscreteFunctionVariant>& aT1,
        const std::shared_ptr<const DiscreteFunctionVariant>& aT2,
        const std::shared_ptr<const DiscreteFunctionVariant>& sigma1,
        const std::shared_ptr<const DiscreteFunctionVariant>& sigma2,
        const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list1,
        const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list2,
        const double& mu,
        const double& lambda) const
  {
    std::shared_ptr new_m2 = getCommonMesh({rho2, aL2, aT2, u2, sigma2});
    std::shared_ptr m1     = getCommonMesh({rho1, aL1, aT1, u1, sigma1});

    std::shared_ptr<const DiscreteFunctionVariant> new_rho2 = rho2;
    std::shared_ptr<const DiscreteFunctionVariant> new_u2   = u2;
    std::shared_ptr<const DiscreteFunctionVariant> new_E2   = E2;
    std::shared_ptr<const DiscreteFunctionVariant> new_CG2  = CG2;

    auto [map1, map2] = _computeMapping(m1, new_m2, bc_descriptor_list1, bc_descriptor_list2);

    auto [ur1, Fjr1, ur2, Fjr2, CR_ur, CR_Fjr] =
      compute_fluxes(solver_type, rho1, aL1, aT1, u1, sigma1, bc_descriptor_list1, rho2, aL2, aT2, u2, sigma2,
                     bc_descriptor_list2, map1, map2);
    const auto [new_m1, new_rho1, new_u1, new_E1, new_CG1] = apply_fluxes(dt1, rho1, u1, E1, CG1, ur1, Fjr1);

    double dt2                    = dt1 / q;
    const double gamma            = 1.4;
    const TinyMatrix<Dimension> I = identity;
    double sum_dt                 = 0;

    size_t fluid = 0;
    if ((mu == 0) and (lambda == 0)) {
      fluid = 1;
    }

    for (size_t i = 0; i < q; i++) {
      if (i == q - 1) {
        dt2 = dt1 - sum_dt;
      }

      const DiscreteScalarFunction& rho_d = new_rho2->get<DiscreteScalarFunction>();
      const DiscreteVectorFunction& u_d   = new_u2->get<DiscreteVectorFunction>();
      const DiscreteScalarFunction& E_d   = new_E2->get<DiscreteScalarFunction>();
      const DiscreteScalarFunction& eps   = E_d - 0.5 * dot(u_d, u_d);
      const DiscreteTensorFunction& CG_d  = new_CG2->get<DiscreteTensorFunction>();
      const DiscreteScalarFunction& p     = fluid * (gamma - 1) * rho_d * eps;
      const DiscreteTensorFunction& sigma_d =
        mu / sqrt(det(CG_d)) * (CG_d - I) + lambda / sqrt(det(CG_d)) * log(sqrt(det(CG_d))) * I - p * I;
      const DiscreteScalarFunction& aL_d = sqrt((lambda + 2 * mu) / rho_d + gamma * p / rho_d);

      const DiscreteScalarFunction& aT_d = sqrt(mu / rho_d);

      const std::shared_ptr<const DiscreteFunctionVariant>& new_sigma2 =
        std::make_shared<const DiscreteFunctionVariant>(sigma_d);
      const std::shared_ptr<const DiscreteFunctionVariant>& new_aL2 =
        std::make_shared<const DiscreteFunctionVariant>(aL_d);
      const std::shared_ptr<const DiscreteFunctionVariant>& new_aT2 =
        std::make_shared<const DiscreteFunctionVariant>(aT_d);

      auto [sub_ur2, sub_Fjr2] = compute_fluxes(solver_type, new_rho2, new_aL2, new_aT2, new_u2, new_sigma2,
                                        bc_descriptor_list2, CR_ur, CR_Fjr, map2);

      std::tie(new_m2, new_rho2, new_u2, new_E2, new_CG2) =
        apply_fluxes(dt2, new_rho2, new_u2, new_E2, new_CG2, sub_ur2, sub_Fjr2);

      sum_dt += dt2;
    }

    std::tie(new_m2, new_rho2, new_u2, new_E2, new_CG2) =
      mesh_correction(new_m1, new_m2, new_rho2, new_u2, new_E2, new_CG2, map1, map2);

    return {new_m1, new_rho1, new_u1, new_E1, new_CG1, new_m2, new_rho2, new_u2, new_E2, new_CG2};
  }

  std::tuple<std::shared_ptr<const MeshVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const MeshVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>,
             std::shared_ptr<const DiscreteFunctionVariant>>
  apply(const SolverType& solver_type,
        const size_t& law, 
        const double& dt1,
        const std::shared_ptr<const DiscreteFunctionVariant>& rho1,
        const std::shared_ptr<const DiscreteFunctionVariant>& rho2,
        const std::shared_ptr<const DiscreteFunctionVariant>& u1,
        const std::shared_ptr<const DiscreteFunctionVariant>& u2,
        const std::shared_ptr<const DiscreteFunctionVariant>& E1,
        const std::shared_ptr<const DiscreteFunctionVariant>& E2,
        const std::shared_ptr<const DiscreteFunctionVariant>& CG1,
        const std::shared_ptr<const DiscreteFunctionVariant>& CG2,
        const std::shared_ptr<const DiscreteFunctionVariant>& aL1,
        const std::shared_ptr<const DiscreteFunctionVariant>& aL2,
        const std::shared_ptr<const DiscreteFunctionVariant>& aT1,
        const std::shared_ptr<const DiscreteFunctionVariant>& aT2,
        const std::shared_ptr<const DiscreteFunctionVariant>& sigma1,
        const std::shared_ptr<const DiscreteFunctionVariant>& sigma2,
        const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list1,
        const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list2,
        const double& lambda_f,
        const double& mu_f,
        const double& lambda_s,
        const double& mu_s,
        const double& gamma_f,
        const double& p_inf_f,
        const double& gamma_s,
        const double& p_inf_s) const
  {
    std::shared_ptr m2 = getCommonMesh({rho2, aL2, aT2, u2, sigma2});
    std::shared_ptr m1     = getCommonMesh({rho1, aL1, aT1, u1, sigma1});

    std::shared_ptr m1_app = getCommonMesh({rho1, aL1, aT1, u1, sigma1});
    std::shared_ptr<const DiscreteFunctionVariant> rho1_app = rho1;
    std::shared_ptr<const DiscreteFunctionVariant> u1_app   = u1;
    std::shared_ptr<const DiscreteFunctionVariant> E1_app   = E1;
    std::shared_ptr<const DiscreteFunctionVariant> CG1_app  = CG1;

    std::shared_ptr m2_app = getCommonMesh({rho2, aL2, aT2, u2, sigma2});
    std::shared_ptr<const DiscreteFunctionVariant> rho2_app   = rho2;
    std::shared_ptr<const DiscreteFunctionVariant> u2_app     = u2;
    std::shared_ptr<const DiscreteFunctionVariant> E2_app     = E2;
    std::shared_ptr<const DiscreteFunctionVariant> CG2_app    = CG2;
    std::shared_ptr<const DiscreteFunctionVariant> aL2_app    = aL2;
    std::shared_ptr<const DiscreteFunctionVariant> aT2_app    = aT2;
    std::shared_ptr<const DiscreteFunctionVariant> sigma2_app = sigma2;


    auto [map1, map2] = _computeMapping(m1,m2,bc_descriptor_list1,bc_descriptor_list2);

    auto [ur1, Fjr1, ur2, Fjr2, CR_ur, CR_Fjr] = 
      compute_fluxes2(solver_type, rho1, aL1, aT1, u1, sigma1, bc_descriptor_list1, rho2, aL2, aT2, u2, sigma2,
                     bc_descriptor_list2, map1, map2);                                                              //Fluxes t^n
    std::tie(m1_app,rho1_app,u1_app,E1_app,CG1_app) = apply_fluxes(dt1/2, rho1, u1, E1, CG1, ur1, Fjr1);            //Fluid t^n+(dt*0.5)

    DiscreteScalarFunction aL_d = aL2->get<DiscreteScalarFunction>();
    DiscreteScalarFunction aT_d = aT2->get<DiscreteScalarFunction>();
    const std::shared_ptr<const DiscreteFunctionVariant>& c = std::make_shared<const DiscreteFunctionVariant>(aL_d + aT_d);

    double dt2 = 0.4 * hyperelastic_dt(c);
    if(dt2 > dt1/2){
	    dt2 = dt1/2;
    }

    std::tie(m2_app,rho2_app,u2_app,E2_app,CG2_app) = apply_fluxes(dt2, rho2, u2, E2, CG2, ur2, Fjr2);              //Solid t^n + dt2

    double sum_dt = dt2;

    size_t chi_solid = 0;
    if((mu_s!=0) and (lambda_s!=0)){
      chi_solid = 1;
    }
    const size_t chi_fluid = 1-chi_solid;

    while(sum_dt < dt1/2){  //Partie a revoir avec loi de comportement diff
      std::tie(sigma2_app, aL2_app, aT2_app) = _apply_state_law(law,rho2_app,u2_app,E2_app,CG2_app,chi_solid,lambda_s,mu_s,gamma_s,p_inf_s);

      DiscreteScalarFunction aL2_d_app = aL2_app->get<DiscreteScalarFunction>();
      DiscreteScalarFunction aT2_d_app = aT2_app->get<DiscreteScalarFunction>();
      const std::shared_ptr<const DiscreteFunctionVariant>& c_app =
        std::make_shared<const DiscreteFunctionVariant>(aL2_d_app + aT2_d_app);

      dt2 = 0.4 * hyperelastic_dt(c_app);
      if(sum_dt + dt2 > dt1/2){
	      dt2 = dt1/2 - sum_dt;
      }

      auto [sub_ur2, sub_Fjr2] = compute_fluxes(solver_type, rho2_app, aL2_app, aT2_app, u2_app, sigma2_app,
                                                bc_descriptor_list2, CR_ur, CR_Fjr, map2);

      std::tie(m2_app, rho2_app, u2_app, E2_app, CG2_app) = 
        apply_fluxes(dt2, rho2_app, u2_app, E2_app, CG2_app, sub_ur2, sub_Fjr2);

      sum_dt += dt2;
    }

    const auto& [sigma1_app, aL1_app, aT1_app] = _apply_state_law(law,rho1_app,u1_app,E1_app,CG1_app,chi_fluid,lambda_f,mu_f,gamma_f,p_inf_f);
    std::tie(sigma2_app, aL2_app, aT2_app) = _apply_state_law(law,rho2_app,u2_app,E2_app,CG2_app,chi_solid,lambda_s,mu_s,gamma_s,p_inf_s);

    auto [ur1_app, Fjr1_app, ur2_app, Fjr2_app, CR_ur_app, CR_Fjr_app] = compute_fluxes2(solver_type,rho1_app, aL1_app, aT1_app, u1_app, sigma1_app, bc_descriptor_list1, 
                                                                                         rho2_app, aL2_app, aT2_app, u2_app, sigma2_app, bc_descriptor_list2, map1, map2); //Fluxes t^n + dt*0.5

    const auto [new_m1, new_rho1, new_u1, new_E1, new_CG1] = order2_apply_fluxes(dt1, rho1, u1, E1, CG1, rho1_app, CG1_app, ur1_app, Fjr1_app);
    auto [new_m2, new_rho2, new_u2, new_E2, new_CG2] = _compute_classic_midpoint_method(solver_type,law,dt1,rho2,u2,E2,CG2,aL2,aT2,sigma2,bc_descriptor_list2,
                                                                                        CR_ur_app,CR_Fjr_app,map2,chi_solid,lambda_s,mu_s,gamma_s,p_inf_s);

    std::tie(new_m2, new_rho2, new_u2, new_E2, new_CG2) = mesh_correction(new_m1, new_m2, new_rho2, new_u2, new_E2, new_CG2, map1, map2);

    return {new_m1, new_rho1, new_u1, new_E1, new_CG1, new_m2, new_rho2, new_u2, new_E2, new_CG2};

  }

  Order2LocalDtHyperelasticSolver()                            = default;
  Order2LocalDtHyperelasticSolver(Order2LocalDtHyperelasticSolver&&) = default;
  ~Order2LocalDtHyperelasticSolver()                           = default;
};

template <MeshConcept MeshType>
void
Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::_applyPressureBC(
  const BoundaryConditionList& bc_list,
  const MeshType& mesh,
  NodeValue<Rd>& br) const
{
  for (const auto& boundary_condition : bc_list) {
    std::visit(
      [&](auto&& bc) {
        using T = std::decay_t<decltype(bc)>;
        if constexpr (std::is_same_v<PressureBoundaryCondition, T>) {
          MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(mesh);
          if constexpr (Dimension == 1) {
            const NodeValuePerCell<const Rd> Cjr = mesh_data.Cjr();

            const auto& node_to_cell_matrix               = mesh.connectivity().nodeToCellMatrix();
            const auto& node_local_numbers_in_their_cells = mesh.connectivity().nodeLocalNumbersInTheirCells();

            const auto& node_list  = bc.nodeList();
            const auto& value_list = bc.valueList();
            parallel_for(
              node_list.size(), PUGS_LAMBDA(size_t i_node) {
                const NodeId node_id       = node_list[i_node];
                const auto& node_cell_list = node_to_cell_matrix[node_id];
                Assert(node_cell_list.size() == 1);

                CellId node_cell_id              = node_cell_list[0];
                size_t node_local_number_in_cell = node_local_numbers_in_their_cells(node_id, 0);

                br[node_id] -= value_list[i_node] * Cjr(node_cell_id, node_local_number_in_cell);
              });
          } else {
            const NodeValuePerFace<const Rd> Nlr = mesh_data.Nlr();

            const auto& face_to_cell_matrix               = mesh.connectivity().faceToCellMatrix();
            const auto& face_to_node_matrix               = mesh.connectivity().faceToNodeMatrix();
            const auto& face_local_numbers_in_their_cells = mesh.connectivity().faceLocalNumbersInTheirCells();
            const auto& face_cell_is_reversed             = mesh.connectivity().cellFaceIsReversed();

            const auto& face_list  = bc.faceList();
            const auto& value_list = bc.valueList();
            for (size_t i_face = 0; i_face < face_list.size(); ++i_face) {
              const FaceId face_id       = face_list[i_face];
              const auto& face_cell_list = face_to_cell_matrix[face_id];
              Assert(face_cell_list.size() == 1);

              CellId face_cell_id              = face_cell_list[0];
              size_t face_local_number_in_cell = face_local_numbers_in_their_cells(face_id, 0);

              const double sign = face_cell_is_reversed(face_cell_id, face_local_number_in_cell) ? -1 : 1;

              const auto& face_nodes = face_to_node_matrix[face_id];

              for (size_t i_node = 0; i_node < face_nodes.size(); ++i_node) {
                NodeId node_id = face_nodes[i_node];
                br[node_id] -= sign * value_list[i_face] * Nlr(face_id, i_node);
              }
            }
          }
        }
      },
      boundary_condition);
  }
}

template <MeshConcept MeshType>
void
Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::_applyNormalStressBC(
  const BoundaryConditionList& bc_list,
  const MeshType& mesh,
  NodeValue<Rd>& br) const
{
  for (const auto& boundary_condition : bc_list) {
    std::visit(
      [&](auto&& bc) {
        using T = std::decay_t<decltype(bc)>;
        if constexpr (std::is_same_v<NormalStressBoundaryCondition, T>) {
          MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(mesh);
          if constexpr (Dimension == 1) {
            const NodeValuePerCell<const Rd> Cjr = mesh_data.Cjr();

            const auto& node_to_cell_matrix               = mesh.connectivity().nodeToCellMatrix();
            const auto& node_local_numbers_in_their_cells = mesh.connectivity().nodeLocalNumbersInTheirCells();

            const auto& node_list  = bc.nodeList();
            const auto& value_list = bc.valueList();
            parallel_for(
              node_list.size(), PUGS_LAMBDA(size_t i_node) {
                const NodeId node_id       = node_list[i_node];
                const auto& node_cell_list = node_to_cell_matrix[node_id];
                Assert(node_cell_list.size() == 1);

                CellId node_cell_id              = node_cell_list[0];
                size_t node_local_number_in_cell = node_local_numbers_in_their_cells(node_id, 0);

                br[node_id] += value_list[i_node] * Cjr(node_cell_id, node_local_number_in_cell);
              });
          } else {
            const NodeValuePerFace<const Rd> Nlr = mesh_data.Nlr();

            const auto& face_to_cell_matrix               = mesh.connectivity().faceToCellMatrix();
            const auto& face_to_node_matrix               = mesh.connectivity().faceToNodeMatrix();
            const auto& face_local_numbers_in_their_cells = mesh.connectivity().faceLocalNumbersInTheirCells();
            const auto& face_cell_is_reversed             = mesh.connectivity().cellFaceIsReversed();

            const auto& face_list  = bc.faceList();
            const auto& value_list = bc.valueList();
            for (size_t i_face = 0; i_face < face_list.size(); ++i_face) {
              const FaceId face_id       = face_list[i_face];
              const auto& face_cell_list = face_to_cell_matrix[face_id];
              Assert(face_cell_list.size() == 1);

              CellId face_cell_id              = face_cell_list[0];
              size_t face_local_number_in_cell = face_local_numbers_in_their_cells(face_id, 0);

              const double sign = face_cell_is_reversed(face_cell_id, face_local_number_in_cell) ? -1 : 1;

              const auto& face_nodes = face_to_node_matrix[face_id];

              for (size_t i_node = 0; i_node < face_nodes.size(); ++i_node) {
                NodeId node_id = face_nodes[i_node];
                br[node_id] += sign * value_list[i_face] * Nlr(face_id, i_node);
              }
            }
          }
        }
      },
      boundary_condition);
  }
}

template <MeshConcept MeshType>
void
Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::_applySymmetryBC(
  const BoundaryConditionList& bc_list,
  NodeValue<Rdxd>& Ar,
  NodeValue<Rd>& br) const
{
  for (const auto& boundary_condition : bc_list) {
    std::visit(
      [&](auto&& bc) {
        using T = std::decay_t<decltype(bc)>;
        if constexpr (std::is_same_v<SymmetryBoundaryCondition, T>) {
          const Rd& n = bc.outgoingNormal();

          const Rdxd I   = identity;
          const Rdxd nxn = tensorProduct(n, n);
          const Rdxd P   = I - nxn;

          const Array<const NodeId>& node_list = bc.nodeList();
          parallel_for(
            bc.numberOfNodes(), PUGS_LAMBDA(int r_number) {
              const NodeId r = node_list[r_number];

              Ar[r] = P * Ar[r] * P + nxn;
              br[r] = P * br[r];
            });
        }
      },
      boundary_condition);
  }
}

template <MeshConcept MeshType>
void
Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::_applyVelocityBC(
  const BoundaryConditionList& bc_list,
  NodeValue<Rdxd>& Ar,
  NodeValue<Rd>& br) const
{
  for (const auto& boundary_condition : bc_list) {
    std::visit(
      [&](auto&& bc) {
        using T = std::decay_t<decltype(bc)>;
        if constexpr (std::is_same_v<VelocityBoundaryCondition, T>) {
          const auto& node_list  = bc.nodeList();
          const auto& value_list = bc.valueList();

          parallel_for(
            node_list.size(), PUGS_LAMBDA(size_t i_node) {
              NodeId node_id    = node_list[i_node];
              const auto& value = value_list[i_node];

              Ar[node_id] = identity;
              br[node_id] = value;
            });
        } else if constexpr (std::is_same_v<FixedBoundaryCondition, T>) {
          const auto& node_list = bc.nodeList();
          parallel_for(
            node_list.size(), PUGS_LAMBDA(size_t i_node) {
              NodeId node_id = node_list[i_node];

              Ar[node_id] = identity;
              br[node_id] = zero;
            });
        }
      },
      boundary_condition);
  }
}

template <MeshConcept MeshType>
void
Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::_applyCouplingBC(
  NodeValue<Rdxd>& Ar1,
  NodeValue<Rdxd>& Ar2,
  NodeValue<Rd>& br1,
  NodeValue<Rd>& br2,
  const std::vector<NodeId>& map1,
  const std::vector<NodeId>& map2) const
{
  size_t n = map1.size();

  for (size_t i = 0; i < n; i++) {
    NodeId node_id1 = map1[i];
    NodeId node_id2 = map2[i];

    Ar1[node_id1] += Ar2[node_id2];
    Ar2[node_id2] = Ar1[node_id1];

    br1[node_id1] += br2[node_id2];
    br2[node_id2] = br1[node_id1];
  }
}

template <MeshConcept MeshType>
void 
Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::_applyCouplingBC2(NodeValue<Rdxd>& Ar1,
										      NodeValue<Rdxd>& Ar2,
										      NodeValue<Rd>& ur1,
										      NodeValue<Rd>& ur2,
										      const std::vector<NodeId>& map1,
										      const std::vector<NodeId>& map2) const
{
  size_t n = map1.size();
  Rd lambda;

  for(size_t i = 0; i<n; i++){

    NodeId node_id1 = map1[i];
    NodeId node_id2 = map2[i];
    
    lambda = inverse(inverse(Ar2[node_id2]) + inverse(Ar1[node_id1]))*(ur2[node_id2]-ur1[node_id1]); 

    ur1[node_id1] = ur1[node_id1] + inverse(Ar1[node_id1])*lambda;
    ur2[node_id2] = ur2[node_id2] - inverse(Ar2[node_id2])*lambda; 
  }
}

template <MeshConcept MeshType>
void
Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::_applyCouplingBC(
  const MeshType& mesh,
  NodeValue<Rd>& ur,
  NodeValue<Rd>& CR_ur,
  NodeValuePerCell<Rd>& Fjr,
  NodeValuePerCell<Rd>& CR_Fjr,
  const std::vector<NodeId>& map2) const
{
  const size_t& n = map2.size();

  const auto& node_to_cell_matrix               = mesh.connectivity().nodeToCellMatrix();
  const auto& node_local_numbers_in_their_cells = mesh.connectivity().nodeLocalNumbersInTheirCells();

  for (size_t i = 0; i < n; i++) {
    const NodeId node_id                       = map2[i];
    const auto& node_to_cell                   = node_to_cell_matrix[node_id];
    const auto& node_local_number_in_its_cells = node_local_numbers_in_their_cells.itemArray(node_id);

    for (size_t j = 0; j < node_to_cell.size(); j++) {
      const CellId J       = node_to_cell[j];
      const unsigned int R = node_local_number_in_its_cells[j];
      Fjr(J, R)            = CR_Fjr(J, R);
    }
    ur[node_id] = CR_ur[node_id];
  }
}

template <MeshConcept MeshType>
class Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::FixedBoundaryCondition
{
 private:
  const MeshNodeBoundary m_mesh_node_boundary;

 public:
  const Array<const NodeId>&
  nodeList() const
  {
    return m_mesh_node_boundary.nodeList();
  }

  FixedBoundaryCondition(const MeshNodeBoundary& mesh_node_boundary) : m_mesh_node_boundary{mesh_node_boundary} {}

  ~FixedBoundaryCondition() = default;
};

template <MeshConcept MeshType>
class Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::PressureBoundaryCondition
{
 private:
  const MeshFaceBoundary m_mesh_face_boundary;
  const Array<const double> m_value_list;

 public:
  const Array<const FaceId>&
  faceList() const
  {
    return m_mesh_face_boundary.faceList();
  }

  const Array<const double>&
  valueList() const
  {
    return m_value_list;
  }

  PressureBoundaryCondition(const MeshFaceBoundary& mesh_face_boundary, const Array<const double>& value_list)
    : m_mesh_face_boundary{mesh_face_boundary}, m_value_list{value_list}
  {}

  ~PressureBoundaryCondition() = default;
};

template <>
class Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<Mesh<1>>::PressureBoundaryCondition
{
 private:
  const MeshNodeBoundary m_mesh_node_boundary;
  const Array<const double> m_value_list;

 public:
  const Array<const NodeId>&
  nodeList() const
  {
    return m_mesh_node_boundary.nodeList();
  }

  const Array<const double>&
  valueList() const
  {
    return m_value_list;
  }

  PressureBoundaryCondition(const MeshNodeBoundary& mesh_node_boundary, const Array<const double>& value_list)
    : m_mesh_node_boundary{mesh_node_boundary}, m_value_list{value_list}
  {}

  ~PressureBoundaryCondition() = default;
};

template <MeshConcept MeshType>
class Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::NormalStressBoundaryCondition
{
 private:
  const MeshFaceBoundary m_mesh_face_boundary;
  const Array<const Rdxd> m_value_list;

 public:
  const Array<const FaceId>&
  faceList() const
  {
    return m_mesh_face_boundary.faceList();
  }

  const Array<const Rdxd>&
  valueList() const
  {
    return m_value_list;
  }

  NormalStressBoundaryCondition(const MeshFaceBoundary& mesh_face_boundary, const Array<const Rdxd>& value_list)
    : m_mesh_face_boundary{mesh_face_boundary}, m_value_list{value_list}
  {}

  ~NormalStressBoundaryCondition() = default;
};

template <>
class Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<Mesh<1>>::NormalStressBoundaryCondition
{
 private:
  const MeshNodeBoundary m_mesh_node_boundary;
  const Array<const Rdxd> m_value_list;

 public:
  const Array<const NodeId>&
  nodeList() const
  {
    return m_mesh_node_boundary.nodeList();
  }

  const Array<const Rdxd>&
  valueList() const
  {
    return m_value_list;
  }

  NormalStressBoundaryCondition(const MeshNodeBoundary& mesh_node_boundary, const Array<const Rdxd>& value_list)
    : m_mesh_node_boundary{mesh_node_boundary}, m_value_list{value_list}
  {}

  ~NormalStressBoundaryCondition() = default;
};

template <MeshConcept MeshType>
class Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::VelocityBoundaryCondition
{
 private:
  const MeshNodeBoundary m_mesh_node_boundary;

  const Array<const TinyVector<Dimension>> m_value_list;

 public:
  const Array<const NodeId>&
  nodeList() const
  {
    return m_mesh_node_boundary.nodeList();
  }

  const Array<const TinyVector<Dimension>>&
  valueList() const
  {
    return m_value_list;
  }

  VelocityBoundaryCondition(const MeshNodeBoundary& mesh_node_boundary,
                            const Array<const TinyVector<Dimension>>& value_list)
    : m_mesh_node_boundary{mesh_node_boundary}, m_value_list{value_list}
  {}

  ~VelocityBoundaryCondition() = default;
};

template <MeshConcept MeshType>
class Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::SymmetryBoundaryCondition
{
 public:
  using Rd = TinyVector<Dimension, double>;

 private:
  const MeshFlatNodeBoundary<MeshType> m_mesh_flat_node_boundary;

 public:
  const Rd&
  outgoingNormal() const
  {
    return m_mesh_flat_node_boundary.outgoingNormal();
  }

  size_t
  numberOfNodes() const
  {
    return m_mesh_flat_node_boundary.nodeList().size();
  }

  const Array<const NodeId>&
  nodeList() const
  {
    return m_mesh_flat_node_boundary.nodeList();
  }

  SymmetryBoundaryCondition(const MeshFlatNodeBoundary<MeshType>& mesh_flat_node_boundary)
    : m_mesh_flat_node_boundary(mesh_flat_node_boundary)
  {
    ;
  }

  ~SymmetryBoundaryCondition() = default;
};

template <MeshConcept MeshType>
class Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolver<MeshType>::CouplingBoundaryCondition
{
 private:
  const MeshNodeBoundary m_mesh_node_boundary;
  const size_t m_label;

 public:
  const Array<const NodeId>&
  nodeList() const
  {
    return m_mesh_node_boundary.nodeList();
  }

  size_t
  label() const
  {
    return m_label;
  }

  CouplingBoundaryCondition(const MeshNodeBoundary& mesh_node_boundary, const size_t label)
    : m_mesh_node_boundary{mesh_node_boundary}, m_label{label}
  {
    ;
  }

  ~CouplingBoundaryCondition() = default;
};

Order2LocalDtHyperelasticSolverHandler::Order2LocalDtHyperelasticSolverHandler(const std::shared_ptr<const MeshVariant>& i_mesh1,
                                                                   const std::shared_ptr<const MeshVariant>& i_mesh2)
{
  if (not i_mesh1) {
    throw NormalError("mesh1 not defined");
  }

  if (not i_mesh2) {
    throw NormalError("mesh2 not defined");
  }
  std::visit(
    [&](auto&& first_mesh, auto&& second_mesh) {
      using FirstMeshType  = mesh_type_t<decltype(first_mesh)>;
      using SecondMeshType = mesh_type_t<decltype(second_mesh)>;

      if constexpr (!std::is_same_v<FirstMeshType,
                                    SecondMeshType>) {   // <- ICI POUR LE TEST SUR LES TYPES DE MAILLAGES
        throw NormalError("incompatible mesh types");
      }
    },
    i_mesh1->variant(), i_mesh2->variant());

  std::visit(
    [&](auto&& mesh) {
      using MeshType = mesh_type_t<decltype(mesh)>;
      if constexpr (is_polygonal_mesh_v<MeshType>) {
        m_hyperelastic_solver = std::make_unique<Order2LocalDtHyperelasticSolver<MeshType>>();
      } else {
        throw NormalError("unexpected mesh type");
      }
    },
    i_mesh1->variant());
}
