#ifndef DISCRETE_FUNCTION_DPK_HPP
#define DISCRETE_FUNCTION_DPK_HPP

#include <language/utils/ASTNodeDataTypeTraits.hpp>

#include <mesh/ItemArray.hpp>
#include <mesh/ItemArrayUtils.hpp>
#include <mesh/MeshData.hpp>
#include <mesh/MeshDataManager.hpp>
#include <mesh/MeshVariant.hpp>
#include <scheme/DiscreteFunctionDescriptorP0.hpp>
#include <scheme/PolynomialCenteredCanonicalBasisView.hpp>

template <size_t Dimension,
          typename DataType,
          typename BasisView = PolynomialCenteredCanonicalBasisView<Dimension, DataType>>
class DiscreteFunctionDPk
{
 public:
  using BasisViewType = BasisView;
  using data_type     = DataType;

  friend class DiscreteFunctionDPk<Dimension, std::add_const_t<DataType>>;
  friend class DiscreteFunctionDPk<Dimension, std::remove_const_t<DataType>>;

 private:
  std::shared_ptr<const MeshVariant> m_mesh_v;
  size_t m_degree;
  CellArray<DataType> m_by_cell_coefficients;
  CellValue<const TinyVector<Dimension>> m_xj;

 public:
  PUGS_INLINE
  ASTNodeDataType
  dataType() const
  {
    return ast_node_data_type_from<std::remove_const_t<DataType>>;
  }

  PUGS_INLINE size_t
  degree() const
  {
    return m_degree;
  }

  PUGS_INLINE
  const CellArray<DataType>&
  cellArrays() const
  {
    return m_by_cell_coefficients;
  }

  PUGS_INLINE std::shared_ptr<const MeshVariant>
  meshVariant() const
  {
    return m_mesh_v;
  }

  PUGS_FORCEINLINE
  operator DiscreteFunctionDPk<Dimension, const DataType>() const
  {
    return DiscreteFunctionDPk<Dimension, const DataType>(m_mesh_v, m_by_cell_coefficients);
  }

  PUGS_INLINE
  void
  fill(const DataType& data) const noexcept
  {
    static_assert(not std::is_const_v<DataType>, "cannot modify DiscreteFunctionDPk of const");
    m_by_cell_coefficients.fill(data);
  }

  friend PUGS_INLINE DiscreteFunctionDPk<Dimension, std::remove_const_t<DataType>>
  copy(const DiscreteFunctionDPk& source)
  {
    return DiscreteFunctionDPk<Dimension, std::remove_const_t<DataType>>{source.m_mesh_v, copy(source.cellArrays())};
  }

  friend PUGS_INLINE void
  copy_to(const DiscreteFunctionDPk<Dimension, DataType>& source,
          DiscreteFunctionDPk<Dimension, std::remove_const_t<DataType>>& destination)
  {
    Assert(source.m_mesh_v->id() == destination.m_mesh_v->id(), "copy_to target must use the same mesh");
    Assert(source.m_degree == destination.m_degree, "copy_to target must have the same degree");
    copy_to(source.m_by_cell_coefficients, destination.m_by_cell_coefficients);
  }

  PUGS_INLINE
  auto
  coefficients(const CellId cell_id) const
  {
    return m_by_cell_coefficients[cell_id];
  }

  PUGS_FORCEINLINE BasisView
  operator[](const CellId cell_id) const noexcept(NO_ASSERT)
  {
    Assert(m_mesh_v.use_count() > 0, "DiscreteFunctionDPk is not built");
    return BasisView{m_degree, m_by_cell_coefficients[cell_id], m_xj[cell_id]};
  }

  PUGS_INLINE DiscreteFunctionDPk
  operator=(const DiscreteFunctionDPk& f)
  {
    m_mesh_v               = f.m_mesh_v;
    m_degree               = f.m_degree;
    m_by_cell_coefficients = f.m_by_cell_coefficients;
    m_xj                   = f.m_xj;

    return *this;
  }

  DiscreteFunctionDPk(const std::shared_ptr<const MeshVariant>& mesh_v, size_t degree)
    : m_mesh_v{mesh_v},
      m_degree{degree},
      m_by_cell_coefficients{mesh_v->connectivity(), BasisView::dimensionFromDegree(degree)},
      m_xj{MeshDataManager::instance().getMeshData(*m_mesh_v->get<Mesh<Dimension>>()).xj()}
  {}

  DiscreteFunctionDPk(const std::shared_ptr<const MeshVariant>& mesh_v, const CellArray<DataType>& cell_array)
    : m_mesh_v{mesh_v},
      m_degree{BasisView::degreeFromDimension(cell_array.sizeOfArrays())},
      m_by_cell_coefficients{cell_array},
      m_xj{MeshDataManager::instance().getMeshData(*m_mesh_v->get<Mesh<Dimension>>()).xj()}
  {
    Assert(mesh_v->connectivity().id() == cell_array.connectivity_ptr()->id(),
           "cell_array is built on different connectivity");
  }

  template <MeshConcept MeshType>
  DiscreteFunctionDPk(const std::shared_ptr<const MeshType>& p_mesh, size_t degree)
    : DiscreteFunctionDPk{p_mesh->meshVariant(), degree}
  {}

  template <MeshConcept MeshType>
  DiscreteFunctionDPk(const std::shared_ptr<const MeshType>& p_mesh, const CellArray<DataType>& cell_array)
    : DiscreteFunctionDPk{p_mesh->meshVariant(), cell_array}
  {}

  DiscreteFunctionDPk() noexcept = default;

  DiscreteFunctionDPk(const DiscreteFunctionDPk&) noexcept = default;
  DiscreteFunctionDPk(DiscreteFunctionDPk&&) noexcept      = default;

  ~DiscreteFunctionDPk() = default;
};

#endif   // DISCRETE_FUNCTION_DPK_HPP
