#ifndef BUILTIN_FUNCTION_EMBEDDER_HPP
#define BUILTIN_FUNCTION_EMBEDDER_HPP

#include <language/ast/ASTNodeDataType.hpp>
#include <language/utils/ASTNodeDataTypeTraits.hpp>
#include <language/utils/DataHandler.hpp>
#include <language/utils/DataVariant.hpp>
#include <language/utils/FunctionTable.hpp>
#include <utils/Demangle.hpp>
#include <utils/Exceptions.hpp>
#include <utils/PugsTraits.hpp>

#include <functional>
#include <memory>
#include <vector>

class IBuiltinFunctionEmbedder
{
 public:
  virtual size_t numberOfParameters() const = 0;

  virtual ASTNodeDataType getReturnDataType() const = 0;

  virtual std::vector<ASTNodeDataType> getParameterDataTypes() const = 0;

  virtual DataVariant apply(const std::vector<DataVariant>&) const = 0;

  IBuiltinFunctionEmbedder() = default;

  IBuiltinFunctionEmbedder(const IBuiltinFunctionEmbedder&) = delete;
  IBuiltinFunctionEmbedder(IBuiltinFunctionEmbedder&&)      = delete;

  virtual ~IBuiltinFunctionEmbedder() = default;
};

template <typename FX, typename... Args>
class BuiltinFunctionEmbedder : public IBuiltinFunctionEmbedder
{
 private:
  std::function<FX(Args...)> m_f;
  using ArgsTuple = std::tuple<Args...>;

  template <size_t I>
  PUGS_INLINE void
  _copyValue(ArgsTuple& t, const std::vector<DataVariant>& v) const
  {
    std::visit(
      [&](auto&& v_i) {
        using Ti_Type = std::decay_t<decltype(std::get<I>(t))>;
        using Vi_Type = std::decay_t<decltype(v_i)>;

        if constexpr ((std::is_same_v<Vi_Type, Ti_Type>)) {
          std::get<I>(t) = v_i;
        } else if constexpr ((std::is_arithmetic_v<Vi_Type>)and(std::is_arithmetic_v<Ti_Type> or
                                                                std::is_same_v<Ti_Type, std::string>)) {
          std::get<I>(t) = v_i;
        } else if constexpr (is_shared_ptr_v<Ti_Type>) {
          if constexpr (std::is_same_v<Vi_Type, EmbeddedData>) {
            auto& data_handler = static_cast<const DataHandler<typename Ti_Type::element_type>&>(v_i.get());
            std::get<I>(t)     = data_handler.data_ptr();
          } else {
            throw UnexpectedError("unexpected argument types while casting: expecting EmbeddedData");
          }
        } else if constexpr (std::is_same_v<Vi_Type, std::vector<EmbeddedData>>) {
          if constexpr (is_vector_v<Ti_Type>) {
            using Ti_value_type = typename Ti_Type::value_type;
            static_assert(is_shared_ptr_v<Ti_value_type>, "expecting shared_ptr");

            using Ti_handeled_type = typename Ti_value_type::element_type;
            std::get<I>(t).resize(v_i.size());
            for (size_t j = 0; j < v_i.size(); ++j) {
              auto& data_handler = dynamic_cast<const DataHandler<Ti_handeled_type>&>(v_i[j].get());
              std::get<I>(t)[j]  = data_handler.data_ptr();
            }
          } else {
            throw UnexpectedError("Unexpected argument types while casting " + demangle<Vi_Type>() + " -> " +
                                  demangle<Ti_Type>());
          }
        } else {
          throw UnexpectedError("Unexpected argument types while casting " + demangle<Vi_Type>() + " -> " +
                                demangle<Ti_Type>());
        }
      },
      v[I]);
  }

  template <size_t... I>
  PUGS_INLINE void
  _copyFromVector(ArgsTuple& t, const std::vector<DataVariant>& v, std::index_sequence<I...>) const
  {
    Assert(sizeof...(Args) == v.size());
    (_copyValue<I>(t, v), ...);
  }

  template <size_t I>
  PUGS_INLINE ASTNodeDataType
  _getOneParameterDataType(ArgsTuple& t) const
  {
    return ast_node_data_type_from<std::decay_t<decltype(std::get<I>(t))>>;
  }

  template <size_t... I>
  PUGS_INLINE std::vector<ASTNodeDataType>
  _getParameterDataTypes(ArgsTuple t, std::index_sequence<I...>) const
  {
    std::vector<ASTNodeDataType> parameter_type_list;
    (parameter_type_list.push_back(this->_getOneParameterDataType<I>(t)), ...);
    return parameter_type_list;
  }

 public:
  PUGS_INLINE ASTNodeDataType
  getReturnDataType() const final
  {
    return ast_node_data_type_from<FX>;
  }

  PUGS_INLINE std::vector<ASTNodeDataType>
  getParameterDataTypes() const final
  {
    constexpr size_t N = std::tuple_size_v<ArgsTuple>;
    ArgsTuple t;
    using IndexSequence = std::make_index_sequence<N>;

    return this->_getParameterDataTypes(t, IndexSequence{});
  }

  PUGS_INLINE size_t
  numberOfParameters() const final
  {
    return sizeof...(Args);
  }

 private:
  template <typename T>
  PUGS_INLINE std::shared_ptr<IDataHandler>
  _createHandler(std::shared_ptr<T> data) const
  {
    return std::make_shared<DataHandler<T>>(data);
  }

 public:
  PUGS_INLINE
  DataVariant
  apply(const std::vector<DataVariant>& x) const final
  {
    constexpr size_t N = std::tuple_size_v<ArgsTuple>;
    ArgsTuple t;
    using IndexSequence = std::make_index_sequence<N>;

    this->_copyFromVector(t, x, IndexSequence{});
    if constexpr (std::is_arithmetic_v<FX>) {
      return {std::apply(m_f, t)};
    } else if constexpr (std::is_same_v<FX, void>) {
      std::apply(m_f, t);
      return {};
    } else {
      return EmbeddedData(_createHandler(std::apply(m_f, t)));
    }
  }

  template <typename FX2, typename... Args2>
  BuiltinFunctionEmbedder(std::function<FX2(Args2...)> f) : m_f{f}
  {
    static_assert(std::is_same_v<FX, FX2>, "incorrect return type");
    static_assert(sizeof...(Args) == sizeof...(Args2), "invalid number of arguments");
    using Args2Tuple = std::tuple<Args2...>;
    static_assert(std::is_same_v<ArgsTuple, Args2Tuple>, "invalid arguments type");
  }
};

template <typename FX>
class BuiltinFunctionEmbedder<FX, void> : public IBuiltinFunctionEmbedder
{
 private:
  std::function<FX(void)> m_f;

 public:
  PUGS_INLINE ASTNodeDataType
  getReturnDataType() const final
  {
    return ast_node_data_type_from<FX>;
  }

  PUGS_INLINE std::vector<ASTNodeDataType>
  getParameterDataTypes() const final
  {
    return {};
  }

  PUGS_INLINE size_t
  numberOfParameters() const final
  {
    return 0;
  }

 public:
  PUGS_INLINE
  DataVariant
  apply(const std::vector<DataVariant>&) const final
  {
    if constexpr (std::is_arithmetic_v<FX>) {
      return {m_f()};
    } else if constexpr (std::is_same_v<FX, void>) {
      m_f();
      return {};
    } else {
      return EmbeddedData(_createHandler(m_f()));
    }
  }

  BuiltinFunctionEmbedder(const std::function<void(void)>& f) : m_f{f} {}
};

#endif   //  BUILTIN_FUNCTION_EMBEDDER_HPP
