#ifndef ELEMENT_INTEGRAL_RECONSTRUCTION_MATRIX_BUILDER_HPP
#define ELEMENT_INTEGRAL_RECONSTRUCTION_MATRIX_BUILDER_HPP

#include <algebra/ShrinkMatrixView.hpp>
#include <algebra/SmallMatrix.hpp>
#include <analysis/QuadratureFormula.hpp>
#include <analysis/QuadratureManager.hpp>
#include <mesh/CellType.hpp>
#include <mesh/ItemValue.hpp>
#include <mesh/StencilArray.hpp>
#include <scheme/PolynomialReconstruction.hpp>
#include <utils/SmallArray.hpp>

template <MeshConcept MeshTypeT>
class PolynomialReconstruction::ElementIntegralReconstructionMatrixBuilder
{
 public:
  using MeshType = MeshTypeT;

  constexpr static bool handles_high_degrees = true;

 private:
  using Rd = TinyVector<MeshType::Dimension>;

  const size_t m_basis_dimension;
  const size_t m_polynomial_degree;

  const SmallArray<double> m_wq_detJ_ek;
  SmallArray<double> m_mean_j_of_ejk;
  SmallArray<double> m_mean_i_of_ejk;

  const ItemToItemMatrix<ItemType::cell, ItemType::node> m_cell_to_node_matrix;
  const CellToCellStencilArray& m_stencil_array;

  const SmallArray<const Rd> m_symmetry_origin_list;
  const SmallArray<const Rd> m_symmetry_normal_list;

  const CellValue<const CellType> m_cell_type;
  const CellValue<const double> m_Vj;
  const CellValue<const Rd> m_xj;
  const NodeValue<const Rd> m_xr;

  // 2D
  SmallArray<const size_t> m_y_row_index;

  // 3D
  SmallArray<const size_t> m_yz_row_index;
  SmallArray<const size_t> m_z_triangle_index;
  SmallArray<const size_t> m_yz_row_size;

  template <typename ConformTransformationT>
  void _computeEjkMean(const QuadratureFormula<MeshType::Dimension>& quadrature,
                       const ConformTransformationT& T,
                       const Rd& Xj,
                       const double Vi,
                       SmallArray<double>& mean_of_ejk) noexcept(NO_ASSERT);

  void _computeEjkMean(const TinyVector<MeshType::Dimension>& Xj,
                       const CellId& cell_i_id,
                       SmallArray<double>& mean_of_ejk);

  void _computeEjkMeanInSymmetricCell(const Rd& origin,
                                      const Rd& normal,
                                      const Rd& Xj,
                                      const CellId& cell_i_id,
                                      SmallArray<double>& mean_of_ejk);

 public:
  PUGS_INLINE
  SmallArray<const double>
  meanjOfEjk() const
  {
    return m_mean_j_of_ejk;
  }

  void build(const CellId cell_j_id, ShrinkMatrixView<SmallMatrix<double>>& A);

  ElementIntegralReconstructionMatrixBuilder(const MeshType& mesh,
                                             const size_t polynomial_degree,
                                             const SmallArray<const Rd>& symmetry_origin_list,
                                             const SmallArray<const Rd>& symmetry_normal_list,
                                             const CellToCellStencilArray& stencil_array);

  ~ElementIntegralReconstructionMatrixBuilder() = default;
};

#endif   // ELEMENT_INTEGRAL_RECONSTRUCTION_MATRIX_BUILDER_HPP
