#ifndef POLYNOMIAL_RECONSTRUCTION_DESCRIPTOR_HPP
#define POLYNOMIAL_RECONSTRUCTION_DESCRIPTOR_HPP

#include <mesh/IBoundaryDescriptor.hpp>
#include <mesh/StencilDescriptor.hpp>
#include <scheme/IntegrationMethodType.hpp>
#include <utils/PugsMacros.hpp>

#include <cstddef>
#include <memory>
#include <vector>

class PolynomialReconstructionDescriptor
{
 public:
  using BoundaryDescriptorList = std::vector<std::shared_ptr<const IBoundaryDescriptor>>;

 private:
  IntegrationMethodType m_integration_method;
  size_t m_degree;
  StencilDescriptor m_stencil_descriptor;

  BoundaryDescriptorList m_symmetry_boundary_descriptor_list;

  bool m_preconditioning = true;
  bool m_row_weighting   = true;

 public:
  PUGS_INLINE IntegrationMethodType
  integrationMethodType() const
  {
    return m_integration_method;
  }

  PUGS_INLINE
  size_t
  degree() const
  {
    return m_degree;
  }

  PUGS_INLINE
  const StencilDescriptor&
  stencilDescriptor() const
  {
    return m_stencil_descriptor;
  }

  PUGS_INLINE
  const BoundaryDescriptorList&
  symmetryBoundaryDescriptorList() const
  {
    return m_symmetry_boundary_descriptor_list;
  }

  PUGS_INLINE
  bool
  preconditioning() const
  {
    return m_preconditioning;
  }

  PUGS_INLINE
  bool
  rowWeighting() const
  {
    return m_row_weighting;
  }

  PUGS_INLINE
  void
  setPreconditioning(const bool preconditioning)
  {
    m_preconditioning = preconditioning;
  }

  PUGS_INLINE
  void
  setRowWeighting(const bool row_weighting)
  {
    m_row_weighting = row_weighting;
  }

  PolynomialReconstructionDescriptor(const IntegrationMethodType integration_method, const size_t degree)
    : m_integration_method{integration_method},
      m_degree{degree},
      m_stencil_descriptor(degree, StencilDescriptor::ConnectionType::by_nodes)
  {}

  PolynomialReconstructionDescriptor(const IntegrationMethodType integration_method,
                                     const size_t degree,
                                     const BoundaryDescriptorList& symmetry_boundary_descriptor_list)
    : m_integration_method{integration_method},
      m_degree{degree},
      m_stencil_descriptor(degree, StencilDescriptor::ConnectionType::by_nodes),
      m_symmetry_boundary_descriptor_list(symmetry_boundary_descriptor_list)
  {}

  PolynomialReconstructionDescriptor(const IntegrationMethodType integration_method,
                                     const size_t degree,
                                     const StencilDescriptor& stencil_descriptor)
    : m_integration_method{integration_method}, m_degree{degree}, m_stencil_descriptor{stencil_descriptor}
  {}

  PolynomialReconstructionDescriptor(const IntegrationMethodType integration_method,
                                     const size_t degree,
                                     const StencilDescriptor& stencil_descriptor,
                                     const BoundaryDescriptorList& symmetry_boundary_descriptor_list)
    : m_integration_method{integration_method},
      m_degree{degree},
      m_stencil_descriptor{stencil_descriptor},
      m_symmetry_boundary_descriptor_list(symmetry_boundary_descriptor_list)
  {}

  PolynomialReconstructionDescriptor() = delete;

  PolynomialReconstructionDescriptor(const PolynomialReconstructionDescriptor&) = default;
  PolynomialReconstructionDescriptor(PolynomialReconstructionDescriptor&&)      = default;

  ~PolynomialReconstructionDescriptor() = default;
};

#endif   // POLYNOMIAL_RECONSTRUCTION_DESCRIPTOR_HPP
