#ifndef POLYNOMIAL_RECONSTRUCTION_HPP
#define POLYNOMIAL_RECONSTRUCTION_HPP

#include <mesh/MeshTraits.hpp>
#include <scheme/PolynomialReconstructionDescriptor.hpp>

class DiscreteFunctionDPkVariant;
class DiscreteFunctionVariant;

class PolynomialReconstruction
{
 private:
  template <MeshConcept MeshType>
  class Internal;

  class MutableDiscreteFunctionDPkVariant;

  template <MeshConcept MeshType>
  class CellCenterReconstructionMatrixBuilder;

  template <MeshConcept MeshType>
  class ElementIntegralReconstructionMatrixBuilder;

  template <MeshConcept MeshType>
  class BoundaryIntegralReconstructionMatrixBuilder;

  const PolynomialReconstructionDescriptor m_descriptor;

  size_t _getNumberOfColumns(
    const std::vector<std::shared_ptr<const DiscreteFunctionVariant>>& discrete_function_variant_list) const;

  template <MeshConcept MeshType>
  void _checkDataAndSymmetriesCompatibility(
    const std::vector<std::shared_ptr<const DiscreteFunctionVariant>>& discrete_function_variant_list) const;

  template <MeshConcept MeshType>
  std::vector<MutableDiscreteFunctionDPkVariant> _createMutableDiscreteFunctionDPKVariantList(
    const std::shared_ptr<const MeshType>& p_mesh,
    const std::vector<std::shared_ptr<const DiscreteFunctionVariant>>& discrete_function_variant_list) const;

  template <typename ReconstructionMatrixBuilderType, MeshConcept MeshType>
  [[nodiscard]] std::vector<std::shared_ptr<const DiscreteFunctionDPkVariant>> _build(
    const std::shared_ptr<const MeshType>& p_mesh,
    const std::vector<std::shared_ptr<const DiscreteFunctionVariant>>& discrete_function_variant_list) const;

  template <MeshConcept MeshType>
  [[nodiscard]] std::vector<std::shared_ptr<const DiscreteFunctionDPkVariant>> _build(
    const std::shared_ptr<const MeshType>& p_mesh,
    const std::vector<std::shared_ptr<const DiscreteFunctionVariant>>& discrete_function_variant_list) const;

 public:
  [[nodiscard]] std::vector<std::shared_ptr<const DiscreteFunctionDPkVariant>> build(
    const std::vector<std::shared_ptr<const DiscreteFunctionVariant>>& discrete_function_variant_list) const;

  template <typename... DiscreteFunctionT>
  [[nodiscard]] std::vector<std::shared_ptr<const DiscreteFunctionDPkVariant>>
  build(DiscreteFunctionT... input) const
  {
    std::vector<std::shared_ptr<const DiscreteFunctionVariant>> variant_vector;
    auto convert = [&variant_vector](auto&& df) {
      using DF_T = std::decay_t<decltype(df)>;
      if constexpr (is_discrete_function_v<DF_T> or std::is_same_v<DiscreteFunctionVariant, DF_T>) {
        variant_vector.push_back(std::make_shared<DiscreteFunctionVariant>(df));
      } else if constexpr (is_shared_ptr_v<DF_T>) {
        using DF_Value_T = std::decay_t<typename DF_T::element_type>;
        if constexpr (is_discrete_function_v<DF_Value_T> or std::is_same_v<DiscreteFunctionVariant, DF_Value_T>) {
          variant_vector.push_back(std::make_shared<DiscreteFunctionVariant>(*df));
        } else {
          static_assert(is_false_v<DF_T>, "unexpected type");
        }
      } else {
        static_assert(is_false_v<DF_T>, "unexpected type");
      }
    };

    (convert(std::forward<DiscreteFunctionT>(input)), ...);
    return this->build(variant_vector);
  }

  PolynomialReconstruction(const PolynomialReconstructionDescriptor& descriptor) : m_descriptor{descriptor} {}
  ~PolynomialReconstruction() = default;
};

#endif   // POLYNOMIAL_RECONSTRUCTION_HPP
